/*******************************************************************************
 * Copyright (c) 2005, 2010 Intel Corporation and others
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hoang M Nguyen, Intel - Initial API and Implementation
 *
 * $Id: SocketListener.h,v 1.17 2010/12/07 18:36:30 jwest Exp $
 *
 *******************************************************************************/ 



#ifndef _SocketListener_H
#define _SocketListener_H

#ifdef _SOLARIS
#include <sys/types.h>
#endif

#include "tptp/SocketTL.h"
#include "tptp/TPTPMessageHeader.h"
#include "tptp/hashtable.h"

#include "SSLTypes.h"

#define SOCKET_LISTENER_OBJECT_ID  20001

/** instance-specific data block */
typedef struct {
	int                     threadStatus ;
	unsigned int            port;    
	SOCKET                  serverSock ;
	tptp_object*            cmo ;
	transport_layer_data_t	agentControllerDataBlk ;
	processMessage_ptr_t    processMessage;
	int						securityEnabled;
	char*					sslProviderLib;
	tptp_object*            nexto;
	tptp_list_t* 		 	params;

	/* hash table of connections for easy and fast search */
    HashTable               *  connectionTable ;

} server_block_t ;

typedef server_block_t * server_block_ptr_t ;

/**
 *********************************************************
 *
 * @brief
 *    create a socket listener instance
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createSocketListener(tptp_object* cmo, transport_layer_data_t * pTransportData, tptp_object* tlo) ;

/**
 *********************************************************
 *
 * @brief
 *    destroy a socket listener instance
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 destroySocketListener(tptp_object* tlo);

/**
 *********************************************************
 *
 * @brief
 *    set the function for forwarding messages
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 setSocketProcessMessageFunc( server_block_t* pServerData, tptp_object* nexto, processMessage_ptr_t func);


/**
 *********************************************************
 *
 * @brief
 *    create a new running thread to handle socket connections
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 startSocketListener(server_block_t* pServerData);


/**
 *********************************************************
 *
 * @brief
 *    terminate the given connection
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 terminateSocketConnection(server_block_t* pServerData, tptp_uint32 connectionID);


/**
 *********************************************************
 *
 * @brief
 *    send a given message to the given destination (identified by connection id)
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 sendSocketMessage(server_block_t* pServerData, tptp_uint32 connectionID, tptp_uint32 size, tptp_string* pCmdBlock) ;



/**
 *********************************************************
 *
 * @brief
 *    common interface to set up the data path between source and destination
 *    (socket).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 setSocketIncomingDataFunc( server_block_t* pServerData, tptp_uint32 connectionID, tptp_uint32 partnerID, tptp_object* partner, sendData_ptr_t newDataFunc ) ;


/**
 *********************************************************
 *
 * @brief
 *    stop the server socket. Stop accepting connection requests.
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 stopSocketListener(server_block_t* pServerData) ;


/**
 *********************************************************
 *
 * @brief
 *    send a given message to the given destination (identified by connection id)
 *
 * @return
 *    Negative - Error
 *    0 - Socket is closed gracefully
 *    Positive - the number of bytes that have been sent
 *********************************************************/
tptp_int32 sendSocketData(server_block_t* pServerData, tptp_uint32 connectionID, tptp_uint32 dataSize, tptp_string* pDataBlock) ;


/**
 *********************************************************
 *
 * @brief
 *    common interface to retrieve peer monitoring info
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 getSocketPeerConnectionInfo(server_block_t* pServerData, tptp_string* type, tptp_string** ci );


/**
 *********************************************************
 *
 * @brief
 *    common interface to establish a connection with
 *    another Agent Controller
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createSocketPeerConnection( server_block_t* pServerData, tptp_string* sourceConnectionInfo, tptp_uint32* connectionID );



/** Define 64-bit type for use in SocketTL.C and SocketListener.c*/
#ifdef _SOLARIS
	#ifndef _LP64
		typedef unsigned long long uint64_t;
	#endif
#endif

#ifdef _WIN32
	typedef unsigned __int64  Uint64;
#else

	#ifdef MVS
		/* 189110 - uint64_t is now defined on MVS but not as long long so we won't use it here */
		typedef unsigned long long Uint64;
	#else
		typedef uint64_t Uint64;
	#endif

#endif



/** connection request data block */
typedef struct request_block {
	int                 threadStatus ;
	SOCKET              clientSock;
	int                 connectionId ;
	int                 connectionType ;
	int                 flags ;
	server_block_ptr_t  pServerData ;
	sendData_ptr_t      pSendFunc ;
	tptp_object*        partner;
	int                 connectionPartnerID ;
	
	ssl_socket_t		ssl_socket;
	BOOL				secured;
	BOOL				authenticated;

	BOOL                isForConsole ;
	
	Lock_t				locker ;

	/** Data buffer used by sendData to buffer outgoing data traffic */
	tptp_list_t dataList;

	/** Size of the list (in entries) */
	int dataListSize;

	/** Size of the list (In total bytes) */
	int dataListTotalBytes;

	/** Last time data was written from the buffer (in system msecs) */
	Uint64 dataListLastWriteInMsecs;

	/** This thread ensures that data in the data list is written to the workbench at least every X milliseconds */
	BOOL timedDataWriteThreadStarted;

	/** List lock*/
	Lock_t	dataListLock;

} request_block_t ;


typedef request_block_t * request_block_ptr_t ;

Uint64 getCurrentTimeInMsecs();

#endif

