/*******************************************************************************
 * Copyright (c) 2005, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Hoang M Nguyen, Intel - Initial API and Implementation
 *
 * $Id: SocketTL.c,v 1.12 2010/11/20 18:42:36 jwest Exp $
 *
 *******************************************************************************/ 



#include "tptp/SocketTL.h"
#include "tptp/TransportSupport.h"
#include "SocketListener.h"

/**
 *********************************************************
 *
 * @brief
 *    common interface to instantiate a transport layer object
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createTransportListener( tptp_object* cmo, transport_layer_data_t * pTransportData, tptp_object* tlo )
{
	return createSocketListener( cmo, pTransportData, tlo );
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to destroy a transport layer object
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 destroyTransportListener( tptp_object* tlo )
{
	return (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID)? 
		destroySocketListener(tlo) : -1) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set the function and data block 
 * for forwarding messages
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 setProcessMessageFunc(tptp_object* tlo, tptp_object* nexto, processMessage_ptr_t func )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setSocketProcessMessageFunc( pServerData, nexto, func ) ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to start an underlying transport mechanism
 *    (socket).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 startTransportListener(tptp_object* tlo)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = startSocketListener(pServerData);
	}

	return ( rc ) ;
}

/**
 *********************************************************
 *
 * @brief
 *    common interface to stop the transport mechanism
 *    (socket).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 stopTransportListener(tptp_object* tlo)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = stopSocketListener(pServerData)  ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate the given connection
 *    (socket).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 terminateConnection(tptp_object* tlo, tptp_uint32  connectionID)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = terminateSocketConnection(pServerData, connectionID)  ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to send to the given destination the given message
 *    (socket).
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/

tptp_int32 sendMessage( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, tptp_string* pCmdBlock)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = sendSocketMessage(pServerData, connectionID, size, pCmdBlock)  ;
	}

	return ( rc ) ;
}



#ifdef _DEBUG_SOCKET_THROUGHPUT

void afp(char * str) {
	FILE *fp;
	fp = fopen("d:\\log", "a+");

	if(fp == NULL) {
		printf("NULL\n");
	}

	fprintf(fp, "%s\n", str);

	fflush(fp);
	fclose(fp);

}

static int mpsCount = 0;
static int msgCount =0;
static int elapsedBytes = 0;
static DWORD elapsedTime = 0;

static DWORD startTime = 0;
static long totalBytes = 0;

static void updateBPS() {
	char dbgBuf[1024];

	if(elapsedTime == 0) {
		elapsedTime = getCurrentTimeInMsecs();
		startTime = getCurrentTimeInMsecs();
	}
	msgCount++;
	mpsCount++;
		
	if(	(getCurrentTimeInMsecs() - elapsedTime) > 4000) {
		double x = 0;
		double y = 0;
		double z = 0;
		elapsedTime = getCurrentTimeInMsecs() - elapsedTime;

		x = ((double)elapsedBytes/ (double)elapsedTime)*1000;
		y = ((double)mpsCount/ (double)elapsedTime)*1000;

		z = ((double)totalBytes/ (double)(getCurrentTimeInMsecs()-startTime))*1000;

		sprintf(dbgBuf, "(%d) kbps: %f, msg per sec: %f. total avg kbps: %f      (%ld and %ld)", msgCount, (float)(x/1024), (float)y, (float)(z/1024), (long)totalBytes, (long)(getCurrentTimeInMsecs()-startTime));
		afp(dbgBuf);

		/* sprintf(dbgBuf, "(%d) %d and %d", msgCount, totalBytes, getCurrentTimeInMsecs()-startTime);
		afp(dbgBuf); */
		// sprintf(dbgBuf, "(%d) kilobytes per second %f", msgCount, (float)(x/1024));

		elapsedBytes = 0;
		elapsedTime = getCurrentTimeInMsecs();
		mpsCount = 0;
	}
	
}

#endif

/**
 *********************************************************
 *
 * @brief
 *    common interface to send data to the given destination 
 *    (socket).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 sendData( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 size, void * pDataBlock)
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

#ifdef _DEBUG_SOCKET_THROUGHPUT
		elapsedBytes += size;
		totalBytes += size;
		updateBPS();
#endif
		rc = sendSocketData(pServerData, connectionID, size, pDataBlock)  ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to set up the data path between source and destination
 *    (socket).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 setIncomingDataFunc( tptp_object* tlo, tptp_uint32 connectionID, tptp_uint32 partnerID, tptp_object* partner, sendData_ptr_t newDataFunc )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = setSocketIncomingDataFunc(pServerData, connectionID, partnerID, partner, newDataFunc)  ;
	}

	return ( rc ) ;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to retrieve peer monitoring info
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 getPeerConnectionInfo(tptp_object* tlo, tptp_string* type, tptp_string** ci )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = getSocketPeerConnectionInfo(pServerData, type, ci);
	}

	return rc;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to establish a connection with
 *    another Agent Controller
 *
 * @return
 *    0 - Success
 *    nonzero - Error.
 *********************************************************/
tptp_int32 createPeerConnection( tptp_object* tlo, tptp_string* sourceConnectionInfo, tptp_uint32* connectionID )
{
	int rc = -1 ;

	if (isValidTPTPBlock(tlo, SOCKET_LISTENER_OBJECT_ID))
	{
		server_block_t* pServerData = (server_block_t*)tlo->data;

		rc = createSocketPeerConnection(pServerData, sourceConnectionInfo, connectionID);
	}

	return rc;
}


/**
 *********************************************************
 *
 * @brief
 *    common interface to terminate a given data connection
 *    (socket).
 *
 * @return
 *    0 - Success
 *    negative - Not supported
 *********************************************************/

tptp_int32 terminateDataConnection(tptp_object* tlo, tptp_uint32  connectionID)
{
	/* there is no difference between terminating the control or data socket */
	return ( terminateConnection(tlo, connectionID) ) ;
}




