#!/bin/bash

#set some script suffixes
EXECUTE=".execute"
BUILD=".build"
CLEAN=".clean"
NORUN=".norun"


#Agent controller executable
#Note the serviceconfig file will be dependent on this name as well!
AC="../bin/tptpAgentCtlr";

#indentation for final report.
resultIndent="       ";

#This cleans up build, execute, and clean log files.
#As a rule the process of running should not fill the screen with
#warning and errors... The idea we should be able to FOCUS on PASS and FAIL.
let reallyClean=0
let reallyBuild=0

#This next variable controls whether to start the AC... and also whether to
#kill the AC and PC on cleanup.
let startAc=1

#setup environment for running tptpAgentCtlr
PWD=`pwd`

#Start the AC in the background
if [ ${startAc} == 1 ]; then
	${AC} &
fi

let autoTestCount=0; #Count of auto tests run.
let manualCount=0;   #This variable stores the count of Manual tests.
let passCount=0;     #This variable keeps track of auto tests that have passed.

declare -a manualTests #Array of Manual Tests that have not been run.


#Takes 1 parameter which is the test being run.
#Assumes that is in that test directory.
#This routine cleans, builds and executes a auto test.
runAutoTest()
{
	# Setup some script names.
	eName=${1}${EXECUTE}
	bName=${1}${BUILD}
	cName=${1}${CLEAN}

	#Do a clean. This may report errors 
	if [ ${reallyClean} == 1 ]; then
		./${cName} >& clean.log
	fi
	#Then build
	if [ ${reallyBuild} == 1 ]; then
		./${bName} >& build.log
	fi
	#Then execute
	./${eName} >& execute.log
	#Check for the existence of ${1}.PASS
	if [ -a ${1}.PASS ]; then
		#Test has passed
		echo "${resultIndent} ${1} PASS";
		#Clean again before we finish
		#Notice we are only cleaning PASS results.
		#Fails will be available for debugging.
		if [ ${reallyClean} == 1 ]; then
			./${cName} >& clean.log
		fi
		let passCount++;
	elif [ -a ${1}.FAIL ]; then
		#Test has Failed
		echo "${resultIndent} ${1} FAIL";
	else	
		#Test is inconclusive... this is a failure
		echo "${resultIndent} ${1} FAIL";
	fi
}

#Takes 1 Parameter which is the test directory.
#Assumes it is being run in the parent of that test.
#In order for the test to be run we need to move to that test dir.
#This routine is called recursively to examine nested tests.
forEachTest()
{
	if [ -a ${1}/${1}${NORUN} ]; then
		return
	fi
	#We only execute tests that follow our testing protocol.
	#Each test needs a test.execute file and we assume that the test name
	#is the directory name. The other script files are based upon this test name.
	if [ -a ${1}/${1}${EXECUTE} ]; then
		let autoTestCount++;
		echo "Running test # ${autoTestCount} - ${1}";
		cd ${1};	#First move to the directory of our test.
		runAutoTest $1;	#Run the test
		cd ".."; 	#move back up.
	else
		#tests have to be in dirs... assume other files are excluded.
		if [ -d ${1} ]; then
			if [ ${1} != "CVS" ]; then
				# First move to the directory of our test.
				cd ${1};
				#Do an ls to check for nested tests.
				for j in $( ls ); do
					forEachTest $j;
				done
				cd "..";
				manualTests[$manualCount]=${1}
				let manualCount++;
			fi
		fi
	fi
}

echo "";
echo "Automated test that have been run: ";

#Look at everything in this directory... we then exclude all non-subdir's 
for i in $( ls ); do
	forEachTest $i;
done

echo "";
echo "Automated Test Summary ";
let rate=passCount/autoTestCount*100;
echo "${resultIndent} ${autoTestCount} tests run"
echo "${resultIndent} ${passCount} tests passed"
echo "${resultIndent} ${rate} % of tests passed"

echo ""
echo "${manualCount} Manual tests that have not been run: ";
for ((a=0; a < ${manualCount}; a++))
do
	echo "${resultIndent} ${manualTests[$a]} "
done


#kill the AC and PC
if [ ${startAc} == 1 ]; then
	#OSTYPE is defined on cygwin... use this to tell if we are windows.
	if [ ${OSTYPE} ]; then
		kill -9 $! 
	else
		#On linux the following does not bring down the PC... so we need
		#to do extra to kill the PC.
		kill -9 $! >& /dev/null
		PCID=`ps -g | grep tptpProcessController | grep -v "grep" | awk ' {printf("%s", $1);}' |  cut -f1`
		kill -9 $PCID >& /dev/null
	fi
fi
