/**********************************************************************
 * Copyright (c) 2007, 2009 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: SecurityPreferencePage.java,v 1.2 2009/08/12 19:02:23 ewchan Exp $
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.security.internal.preference;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.UnrecoverableKeyException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;

import org.eclipse.core.resources.IResourceStatus;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.execution.security.KeystoreHelper;
import org.eclipse.hyades.security.internal.util.CertificateProperties;
import org.eclipse.hyades.security.internal.util.SecurityMessages;
import org.eclipse.hyades.security.internal.util.SecurityUI;
import org.eclipse.hyades.ui.HyadesUIImages;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.dialogs.ErrorDialog;
import org.eclipse.jface.dialogs.IDialogConstants;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnPixelData;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.window.Window;
import org.eclipse.osgi.util.NLS;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;
import org.eclipse.tptp.platform.common.ui.internal.CommonUIPlugin;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;

public class SecurityPreferencePage
	extends PreferencePage
	implements IWorkbenchPreferencePage, SelectionListener {

	private final static String[] COLUMNS = { "STRING", "STRING" };

	private TableViewer tableViewer;
	private ArrayList tableElements = new ArrayList();
	private Button btnAdd;
	private Button btnRemove;
	private Button btnRename;
	private Button btnProperties;
	private KeyStore ks;
	
	private class MyTableElement {
		private String alias;
		private Object value;

		public MyTableElement(String alias,String value)
		{
			this.alias = alias;
			this.value = value;
		}
		public String getAlias() {
			return alias;
		}
		public void setAlias(String text) {
			this.alias = text;
		}
		public Object getValue() {
			return value;
		}
		public String getType(){
			if(value instanceof Certificate){
				return SecurityMessages._62;
			}
			else if(value instanceof Key){
				return SecurityMessages._63;
			}
			return SecurityMessages._64;
		}
		
		public void setValue(Object value) {
			this.value = value;
		}
	}
	private class PreferenceContentProvider implements IStructuredContentProvider {
		/**
		 * @see IStructuredContentProvider#getElements(Object)
		 */
		public Object[] getElements(Object element) {
			if (element instanceof ArrayList)
				return ((ArrayList)element).toArray();

			return new Object[0];
		}
		/**
		 * @see IContentProvider#dispose()
		 */
		public void dispose() {
		}
		/**
		 * @see IContentProvider#inputChanged(Viewer, Object, Object)
		 */
		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}
	}
	private class PreferenceLabelProvider
		extends LabelProvider
		implements ITableLabelProvider {
		/**
		 * @see ITableLabelProvider#getColumnImage(Object, int)
		 */
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		/**
		 * @see ITableLabelProvider#getColumnText(Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			if (element instanceof MyTableElement) {
				MyTableElement myTableElement = (MyTableElement) element;

				if (columnIndex == 0)
					return myTableElement.getAlias();

				if (columnIndex == 1)
					return myTableElement.getType();
			}
			return "";
		}
	}
	public class AddDialog extends Dialog implements Listener{

        private SecurityUI _securityUI;
        private Certificate fCertificate;
        private Shell shell;
		
		public AddDialog(Shell shell){
			super(shell);
			this.shell = shell;
		}
		
		protected Control createDialogArea(Composite parent){ 

			getShell().setText(SecurityMessages._12);
			Composite content = new Composite(parent, SWT.NULL);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			layout.numColumns = 1;
			content.setLayout(layout);
			content.setLayoutData(data);

			_securityUI = new SecurityUI(shell);
			_securityUI.createContents(content);
			_securityUI.registerListener(this);
			return content;
		}
		
		protected Control createButtonBar(Composite parent) {
			Control control = super.createButtonBar(parent);
			//shell.setDefaultButton(getButton(IDialogConstants.OK_ID));
			getButton(IDialogConstants.OK_ID).setEnabled(false);			
			return control;
			
		}
		
		protected void okPressed(){

			try{	
				fCertificate = _securityUI.loadCertificate(ks);
			}
			catch(FileNotFoundException e){

				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES
						,IResourceStatus.ERROR,NLS.bind(SecurityMessages._45, _securityUI.getPath()),e);
				ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
						, SecurityMessages._31,err);
				return;
				
			}
			catch(IOException e){

				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
						,NLS.bind(SecurityMessages._61, CommonUIPlugin.getKeyStoreLocation()),e);
				ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
						, SecurityMessages._31,err);
				return;
											
			}
			catch(CertificateException exc){

				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
						,NLS.bind(SecurityMessages._60, _securityUI.getPath()),exc);
				ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
						, SecurityMessages._31,err);
				return;

			}
			catch(KeyStoreException exc){
				Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
						,NLS.bind(SecurityMessages._49, CommonUIPlugin.getKeyStoreLocation()),exc);
				ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
						, SecurityMessages._31,err);
				return;

			}

			MyTableElement elem = new MyTableElement(_securityUI.getAliasName(), SecurityMessages._62);
			elem.setValue(fCertificate);
			tableElements.add(elem);
						
			super.okPressed();
		}
		
		public void handleEvent(Event e){
			getButton(IDialogConstants.OK_ID).setEnabled(_securityUI.validateDialog());
		}
		
		
		public Certificate getCertificate(){
			return fCertificate;
		}
		
	}

	public class RenameDialog extends Dialog implements Listener{
		
		private Text txtName;
		private String newAlias;
		private String oldAlias;
		public RenameDialog(Shell shell, String oldValue){
			super(shell);
			oldAlias = oldValue;
		}
		
		protected Control createDialogArea(Composite parent){

			getShell().setText(SecurityMessages._13);
			Composite content = new Composite(parent, SWT.NONE);
			GridLayout layout = new GridLayout();
			GridData data = GridUtil.createFill();
			data.widthHint = 350;
			layout.numColumns = 2;
			content.setLayout(layout);
			content.setLayoutData(data);

			Label lblName = new Label(content, SWT.NONE);
			lblName.setText(SecurityMessages._7);
			
			txtName = new Text(content, SWT.BORDER);
			txtName.setText(oldAlias);
			txtName.selectAll();
			txtName.addListener(SWT.Modify, this);
			
			data = GridUtil.createHorizontalFill();
			data.widthHint = 200;
			txtName.setLayoutData(data);

			return content;
		}
		
		private void validateDialog(){
			if(txtName.getText().trim().length()==0)
				getButton(IDialogConstants.OK_ID).setEnabled(false);
			else
				getButton(IDialogConstants.OK_ID).setEnabled(true);
		}
		
		public void handleEvent(Event e){
			if(e.widget.equals(txtName))
				validateDialog();
		}
		
		protected Control createButtonBar(Composite parent) {
			Control control = super.createButtonBar(parent);			
			validateDialog();
			return control;
			
		}

		public String getNewAlias(){
			return newAlias;
		}
		
		protected void okPressed(){
			newAlias = txtName.getText();
			super.okPressed();
		}

	}
		
	public SecurityPreferencePage() {
		super();
		setPreferenceStore(HyadesUIPlugin.getInstance().getPreferenceStore());
	}
	
	protected Control createContents(Composite parent) {
		
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setLayoutData(GridUtil.createFill());
		composite.setLayout(new GridLayout());

		TabFolder folder = new TabFolder(composite, SWT.NONE);
		folder.setLayout(new GridLayout());
		folder.setLayoutData(GridUtil.createFill());
		
		TabItem item;

		item = new TabItem(folder, SWT.NONE);
		item.setText(SecurityMessages._4);
		item.setImage(HyadesUIImages.INSTANCE.getImage(HyadesUIImages.IMG_CERTIF_FILE));
		item.setControl(createSecurityTab(folder));
		
		Dialog.applyDialogFont( composite );
		
		return composite;
	}
	
	private Control createSecurityTab(Composite parent)
	{		
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.marginWidth = 5;
		layout.verticalSpacing = 10;
		layout.numColumns = 2;
		composite.setLayout(layout);
		composite.setLayoutData(GridUtil.createFill());
		
		Label label = new Label(composite, SWT.NULL);
		label.setText(SecurityMessages._15);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
				
		createTableViewer(composite);

		Composite buttons = new Composite(composite, SWT.NONE);
		data = new GridData(GridData.FILL_VERTICAL);
		buttons.setLayoutData(data);
		layout = new GridLayout();
		layout.numColumns = 1;
		buttons.setLayout(layout);
		
		createButtons(buttons);
		initializeValues();
		
		btnAdd.addSelectionListener(this);
		btnRename.addSelectionListener(this);
		btnRemove.addSelectionListener(this);
		btnProperties.addSelectionListener(this);
		
		return composite;
	}
	
	private void createTableViewer(Composite parent) {
		// Create the table viewer.
		tableViewer =
			new TableViewer(
				parent,
				SWT.BORDER | SWT.SINGLE | SWT.FULL_SELECTION );

		// Create the table control.
		Table table = tableViewer.getTable();
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		GridData data = GridUtil.createFill();
		data.heightHint = 50;
		data.widthHint = 100;
		table.setLayoutData(data);

		TableLayout tableLayout = new TableLayout();
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];

		TableColumn aliasColumn = new TableColumn(table, SWT.LEFT);
		aliasColumn.setText(SecurityMessages._1);
		tableLayout.addColumnData(new ColumnPixelData(100));
		cellEditors[0] = new TextCellEditor(table);

		TableColumn certificateColumn = new TableColumn(table, SWT.LEFT);
		certificateColumn.setText(SecurityMessages._2);
		tableLayout.addColumnData(new ColumnPixelData(120));

		table.setLayout(tableLayout);
		

		// Adjust the table viewer.
		tableViewer.setColumnProperties(COLUMNS);
		tableViewer.setContentProvider(new PreferenceContentProvider());
		tableViewer.setLabelProvider(new PreferenceLabelProvider());
		tableViewer.getTable().addSelectionListener(this);

		TableViewerSorter.setTableSorter(tableViewer, 0, true);
	}
	
	private void createButtons(Composite parent){
		
		btnAdd = new Button(parent, SWT.PUSH);
		btnAdd.setText(SecurityMessages._9);
		GridData data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnAdd.setLayoutData(data);
		
		btnRemove = new Button(parent, SWT.PUSH);
		btnRemove.setText(SecurityMessages._10);
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;		
		btnRemove.setLayoutData(data);
		btnRemove.setEnabled(false);		

		Composite space = new Composite(parent, SWT.NONE);
		data = GridUtil.createHorizontalFill();
		data.verticalSpan = 2;
		//data.horizontalIndent = 6;
		space.setLayoutData(data);

		btnRename = new Button(parent, SWT.PUSH);
		btnRename.setText(SecurityMessages._11);
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnRename.setLayoutData(data);
		btnRename.setEnabled(false);
		
		btnProperties = new Button(parent, SWT.PUSH);
		btnProperties.setText(SecurityMessages._16);
		data = GridUtil.createHorizontalFill();
		//data.horizontalIndent = 6;
		btnProperties.setLayoutData(data);
		btnProperties.setEnabled(false);		
	}


	public void widgetDefaultSelected(SelectionEvent e) {
		widgetSelected(e);
	}
	public void widgetSelected(SelectionEvent event) {
		
		if(event.widget == btnAdd){
			AddDialog dlg = new AddDialog(getShell());
			dlg.open();
			
			if(dlg.getReturnCode() == Window.OK)
			{				
				tableViewer.refresh();			
			}
			
		}else if(event.widget == btnRemove){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() > 0){
				Iterator i = elem.iterator();
				while(i.hasNext()){
					try{	
						MyTableElement current = (MyTableElement)i.next();
						ks.deleteEntry(current.getAlias());
						tableElements.remove(current);
					}
					catch(KeyStoreException e){
					}
				}
				
				tableViewer.refresh();				
			}
		
		}
		else if(event.widget == btnRename){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() == 1){
				MyTableElement sel = (MyTableElement)elem.getFirstElement();
				RenameDialog dlg = new RenameDialog(getShell(), sel.getAlias());
				dlg.open();
														
				if(dlg.getReturnCode() == Window.OK)
				{
					try{	
						KeystoreHelper.addCertificateToKeyStore(SecurityPreferencePage.this.ks,(Certificate)sel.getValue(),dlg.getNewAlias());
						ks.deleteEntry(sel.getAlias());
						sel.setAlias(dlg.getNewAlias());
						tableViewer.refresh();							
					}
					catch(KeyStoreException e){
					}						
				}
			}
			
		}else if(event.widget == btnProperties){
			IStructuredSelection elem = (IStructuredSelection)tableViewer.getSelection();
			if(elem.size() == 1){
				MyTableElement sel = (MyTableElement)elem.getFirstElement();
				CertificateProperties dlg = new CertificateProperties(getShell());
				dlg.init(sel.getAlias(), sel.getValue());
				
				dlg.open();
														
			}
			
		} 

		boolean sel = tableViewer.getSelection().isEmpty();
		btnRename.setEnabled(!sel);
		btnRemove.setEnabled(!sel);
		btnProperties.setEnabled(!sel);

	}

	/**
	 * @see PreferencePage#computeSize()
	 */
	public Point computeSize() {
		Point p = super.computeSize();

		return p;
	}
	/**
	 * @see IWorkbenchPreferencePage#init(IWorkbench)
	 */
	public void init(IWorkbench workbench) {
	}
	/**
	 * @see PreferencePage#performDefaults()
	 */
	protected void performDefaults() {
		super.performDefaults();

	}
	/**
	 * @see PreferencePage#performOk()
	 */
	public boolean performOk() {

		String storePath = CommonUIPlugin.getKeyStoreLocation();
		String passw = CommonUIPlugin.getDefault().getWorkspaceName();
		try {
							        
			KeystoreHelper.persistKeyStore(ks, storePath, passw);
			
		}
		catch(IOException e){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._54, storePath),e);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._53,err);
			return false;
										
		}
		catch(CertificateException exc){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._55, storePath),exc);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._53,err);
			return false;

		}
		catch(KeyStoreException exc){
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._56, CommonUIPlugin.getKeyStoreLocation()),exc);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._53,err);
			return false;

		}
		catch(NoSuchAlgorithmException exc2)
		{
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._47, CommonUIPlugin.getKeyStoreLocation()),exc2);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._53,err);
			return false;

		}
		return true;

	}

	/**
	 * Loads certificates from the key store.
	 */
	private void initializeValues() {
		
		
		String storePath = CommonUIPlugin.getKeyStoreLocation();
		String passw = CommonUIPlugin.getDefault().getWorkspaceName();

		try {
			ks = KeystoreHelper.createKeyStore(storePath, passw);
			
			Enumeration aliases = ks.aliases();
			
			
			while(aliases.hasMoreElements())
			{
				String alias = (String)(aliases.nextElement());
				/* The alias may be either a key or a certificate */
				java.security.cert.Certificate cert = ks.getCertificate(alias);
				if(cert != null) {
					MyTableElement elem = new MyTableElement(alias, SecurityMessages._62);
					elem.setValue(cert);
					tableElements.add(elem);
				}
				else {
					try {

						Key key=ks.getKey(alias, passw.toCharArray());
						MyTableElement elem = new MyTableElement(alias, SecurityMessages._63);
						elem.setValue(key);
						tableElements.add(elem);						
					}
					catch(UnrecoverableKeyException e) {
						/* Probably ignore the key in this case */	
					}
				}
					
					
			}			
		}
		catch(IOException e){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._61, storePath),e);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._59,err);
										
		}
		catch(CertificateException exc){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
							,NLS.bind(SecurityMessages._46, storePath),exc);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._59,err);

		}
		catch(KeyStoreException exc){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
						,NLS.bind(SecurityMessages._57, CommonUIPlugin.getKeyStoreLocation()),exc);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._59,err);

		}
		catch(NoSuchProviderException exc2){

			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,SecurityMessages._58,exc2);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, NLS.bind(SecurityMessages._57, CommonUIPlugin.getKeyStoreLocation()),err);

		
		}
		catch(NoSuchAlgorithmException exc2)
		{
			Status err = new Status(Status.ERROR,ResourcesPlugin.PI_RESOURCES,IResourceStatus.ERROR
					,NLS.bind(SecurityMessages._47, CommonUIPlugin.getKeyStoreLocation()),exc2);
			ErrorDialog.openError(Display.getDefault().getActiveShell(),SecurityMessages._41
					, SecurityMessages._59,err);

		}
		
		tableViewer.setInput(tableElements);
	}

	//https://bugs.eclipse.org/bugs/show_bug.cgi?id=286427
	public void createControl(Composite parent) {
		super.createControl(parent);
		getDefaultsButton().setText(SecurityMessages._80);
		getApplyButton().setText(SecurityMessages._81);
	}
		
}
