/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesUIPlugin.java,v 1.12 2009/03/24 16:12:35 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import org.eclipse.core.runtime.Platform;
import org.eclipse.hyades.ui.adapter.HyadesAdapterFactory;
import org.eclipse.hyades.ui.extension.IAssociationConstants;
import org.eclipse.hyades.ui.filters.internal.util.ScopeContainer;
import org.eclipse.hyades.ui.internal.extension.NavigatorExtensionUtil;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.ResourceBundleManager;
import org.eclipse.hyades.ui.util.ILogger;
import org.eclipse.tptp.platform.common.internal.CommonPlugin;
import org.eclipse.ui.plugin.AbstractUIPlugin;
import org.osgi.framework.BundleContext;

/**
 * <p>Hyades UI plug-in class.</p>
 * 
 * <p><b>Note:</b> This class should not be used outside 
 * the context of Eclipse.</p>
 * 
 * 
 * @author  Marcelo Patmarcelop
 * @author  Valentia Popescu
 * @author  Paul Slauenwhite
 * @version March 24, 2009
 * @since   January 26, 2005
 */
public class HyadesUIPlugin 
extends AbstractUIPlugin
{
	
	/**
	 * Extension point id used to register the editor extensions.
	 */
	public static final String EP_EDITOR_EXTENSIONS = "editorExtensions";

	/**
	 * Extension point id used to register the analyzer extensions.
	 * @deprecated use {@link IAssociationConstants#EP_ANALYZER_EXTENSIONS}
	 */
	public static final String EP_ANALYZER_EXTENSIONS = "analyzerExtensions";
	
	/**
	 * Extension point id used to register the type descriptions.
	 * @deprecated use {@link IAssociationConstants#EP_TYPE_DESCRIPTIONS}
	 */
	public static final String EP_TYPE_DESCRIPTIONS = "typeDescriptions";
	
	/**
	 * Extension point id used to register the type validators.
	 */
	public static final String EP_TYPE_VALIDATORS = "typeValidators";
	
	/**
	 * Extension point id used to register the sample projects.
	 */
	public static final String EP_SAMPLE_WIZARD = "sampleWizards";
	
	public static final String PLUGIN_ID = "org.eclipse.hyades.ui"; //$NON-NLS-1$	
	
	private static HyadesUIPlugin instance;
	private ILogger logger;
	private ResourceBundleManager resourceBundleManager;
	private ResourceBundle resourceBundle;	
	
	private static String PREFERENCE_FILTER_SELECTION_KEY = "filters.filterSelection";
	private static String PREFERENCE_STRING_SEPARATOR = "\n";
	private static String PREFERENCE_STRING_VALUE_SEPARATOR = ";";		
	
	/**
	 * Constructor for HyadesUIPlugin
	 * @param descriptor
	 */
	public HyadesUIPlugin()
	{
		instance = this;
	}
	
	/**
	 * Returns the instance of this class created by the eclipse framework.
	 * @return HyadesUIPlugin
	 */
	public static HyadesUIPlugin getInstance()
	{
		return instance;
	}
	
	
	/**
	 * @see org.osgi.framework.BundleActivator#start(org.osgi.framework.BundleContext)
	 */
	public void start(BundleContext context) throws Exception {
		resourceBundleManager = new ResourceBundleManager(); 
		resourceBundleManager.add(getResourceBundle());

		Platform.getAdapterManager().registerAdapters(HyadesAdapterFactory.INSTANCE, LogicalFolder.class);
		
		super.start(context);		
		
		restoreFilterSelections();		
	}		
	
	/**
	 * @see org.osgi.framework.BundleActivator#stop(org.osgi.framework.BundleContext)
	 */
	public void stop(BundleContext context) throws Exception {
		storeFilterSelections();		
		NavigatorExtensionUtil.disposeAll();
		resourceBundleManager.dispose();
		instance = null;
		savePluginPreferences();
		logger = null;
		Platform.getAdapterManager().unregisterAdapters(HyadesAdapterFactory.INSTANCE);
		HyadesAdapterFactory.INSTANCE.dispose();
		
		super.stop(context);		
	}		
	
	/**
	 * Returns the an instance of {@link ILogger} that used the
	 * <code>log</code> methods of this class.
	 * @return ILogger
	 * 
	 * @deprecated use {@link CommonPlugin#getLogger()}
	 */
	public static ILogger getLogger()
	{
		if(instance.logger == null)
		{
			instance.logger = new ILogger()
			{
				public void logError(Throwable throwable)
				{
					logError(throwable);
				}

				public void logError(String text)
				{
					logError(text);
				}

				public void logInfo(String text)
				{
					logInfo(text);
				}
			};
		}
		return instance.logger;
	}

	/**
	 * Returns this plugin's id.
	 * @return String
	 */
	public static String getID()
	{
		return PLUGIN_ID;
	}
	
	/**
	 * Logs an error described by a throwable.
	 * 
	 * <p>This method should be used whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param throwable
	 * @deprecated use {@link CommonPlugin#logError(Throwable)}
	 */
	public static void logError(Throwable throwable)
	{
		CommonPlugin.logError(throwable);
	}
	
	/**
	 * Logs an error described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an error since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 * @deprecated use {@link CommonPlugin#logError(String)}
	 */
	public static void logError(String text)
	{
		CommonPlugin.logError(text);
	}
	
	/**
	 * Logs an information described by a text.
	 * 
	 * <p>This method should be whenever a class in this plugin
	 * has to log an information since it adheres to the global logging
	 * strategy.
	 * 
	 * @param text
	 * @deprecated use {@link CommonPlugin#logInfo(String)}
	 */
	public static void logInfo(String text)
	{
		CommonPlugin.logInfo(text);
	}
	
	/**
	 * Returns the resource bundle used by this plugin.
	 * 
	 * <p>IMPORTANT: Don't use this method to retrieve values from the
	 * resource bundle.  For this purpose use the static <code>getString()</code> 
	 * defined in this class.
	 * 
	 * <p>This method is provided so this resource bundle can
	 * be used as the parent of another resource bundle.
	 * 
	 * @return ResourceBundle
	 */
	public ResourceBundle getResourceBundle()
	{
		if(resourceBundle == null)
		{
		
			try
			{
				resourceBundle = Platform.getResourceBundle(Platform.getBundle(PLUGIN_ID));
			}
			catch (Exception e) {
				resourceBundle = null;
			}
		}
		
		return resourceBundle;		
	}

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * @param key
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key)
	{
		try {
			return instance.resourceBundleManager.getString(key);
		} catch (MissingResourceException e) {
			return key;
		}

	}
	
	/**
	 * Returns the string value associate to a given key.  The key is passed to
	 * each resource bundle in the order they are retrieve by the 
	 * {@link #iterator()} method.
	 * 
	 * <p>The <code>arg</code> string defined replaces the %1 
	 * variable defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and arg is <code>"v1"</code>,
	 * the return of this method is <code>"%0 v1  %2 %3 %4"</code>.
	 * 
	 * @param key
	 * @param arg
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String arg)
	{
		try {
			return instance.resourceBundleManager.getString(key, arg);
		} catch (MissingResourceException e) {
			return key;
		}
	}	

	/**
	 * Returns the "plugin.properties" file's value associate to a given key.
	 * 
	 * <p>The strings defined in <code>args</code> replaces the %n (where n>=1) 
	 * variables defined in the file's values.
	 * 
	 * <p>Example: If the value associated to the key <code>"a"</code> is 
	 * <code>"%0 %1 %2 %3 %4"</code> and args is <code>{"v1", null, "v3"}</code>,
	 * the return of this method is <code>"%0 v1  v3 %4"</code>.
	 * 
	 * @param key
	 * @param args
	 * @return String
	 * @throws java.util.MissingResourceException if the key is not in the file
	 * @throws NullPointerException if key is null
	 */
	public static String getString(String key, String[]args)
	{
		try {
			return instance.resourceBundleManager.getString(key, args);
		}
		catch(MissingResourceException exc)
		{
			return key;
		}
	}	
	
	private void storeFilterSelections()
	{
		ScopeContainer scopeContainer = ScopeContainer.instance();
		StringBuffer filterSelectionStr = new StringBuffer();		
		
		String[] scopes = scopeContainer.scopes();
		String selectedFilterName = null;
		for (int i = 0; i < scopes.length; i++)
		{
			selectedFilterName = scopeContainer.getSelectedFilterName(scopes[i]);
		
			if (selectedFilterName != null)
			{
				filterSelectionStr.append(scopes[i]);
				filterSelectionStr.append(PREFERENCE_STRING_VALUE_SEPARATOR);
				filterSelectionStr.append(selectedFilterName);
				filterSelectionStr.append(PREFERENCE_STRING_SEPARATOR);				
			}
		}
		
		filterSelectionStr.append(PREFERENCE_STRING_SEPARATOR);
		
		String[] viewers = scopeContainer.viewers();
		String selectedScope = null;		
		for (int i = 0; i < viewers.length; i++)
		{
			selectedScope = scopeContainer.getSelectedScope(viewers[i]);
			
			if (selectedScope != null)
			{
				filterSelectionStr.append(viewers[i]);
				filterSelectionStr.append(PREFERENCE_STRING_VALUE_SEPARATOR);
				filterSelectionStr.append(selectedScope);
				filterSelectionStr.append(PREFERENCE_STRING_SEPARATOR);				
			}
		}		
		
		getPluginPreferences().setValue(PREFERENCE_FILTER_SELECTION_KEY, filterSelectionStr.toString());
	}
	
	private void restoreFilterSelections()
	{
		String filterSelectionStr = getPluginPreferences().getString(PREFERENCE_FILTER_SELECTION_KEY);
		
		if (filterSelectionStr == null || "".equals(filterSelectionStr))
			return;
		
		ScopeContainer scopeContainer = ScopeContainer.instance();		
		int idx, sIdx;
		
		String scopeSelectedFilterStr;
		String scope;
		String selectedFilter;
		
		idx = filterSelectionStr.indexOf(PREFERENCE_STRING_SEPARATOR);		
		while (idx > 0)
		{
			scopeSelectedFilterStr = filterSelectionStr.substring(0, idx);
			
			sIdx = scopeSelectedFilterStr.indexOf(PREFERENCE_STRING_VALUE_SEPARATOR);
			
			scope = scopeSelectedFilterStr.substring(0, sIdx);
			selectedFilter = scopeSelectedFilterStr.substring(sIdx+1);
			scopeContainer.setSelectedFilterName(scope, selectedFilter);
		
			filterSelectionStr = filterSelectionStr.substring(idx+1);
			idx = filterSelectionStr.indexOf(PREFERENCE_STRING_SEPARATOR);			
		}

		filterSelectionStr = filterSelectionStr.substring(1);		
		
		
		String viewerSelectedScope;
		String viewer;
		String selectedScope;
		
		idx = filterSelectionStr.indexOf(PREFERENCE_STRING_SEPARATOR);		
		while (idx >= 0)
		{
			viewerSelectedScope = filterSelectionStr.substring(0, idx);
			
			sIdx = viewerSelectedScope.indexOf(PREFERENCE_STRING_VALUE_SEPARATOR);
			
			viewer = viewerSelectedScope.substring(0, sIdx);
			selectedScope = viewerSelectedScope.substring(sIdx+1);
			scopeContainer.setSelectedScope(viewer, selectedScope);
		
			filterSelectionStr = filterSelectionStr.substring(idx+1);
			idx = filterSelectionStr.indexOf(PREFERENCE_STRING_SEPARATOR);			
		}		
	}	
}
