/*******************************************************************************
 * Copyright (c) 2005, 2009 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: EditorSynchronizer.java,v 1.11 2009/12/09 20:35:39 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.editor;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IResourceDeltaVisitor;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.adapter.ISynchronizedEditorAdapter;
import org.eclipse.hyades.ui.editor.HyadesEditorActionContributor;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.ui.IEditorActionBarContributor;
import org.eclipse.ui.IEditorInput;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IFileEditorInput;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IPartListener2;
import org.eclipse.ui.IWindowListener;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartReference;
import org.eclipse.ui.IWorkbenchSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.part.FileEditorInput;

/**
 * <p>Synchronizer to maintain synchronization between an editor, file system, and 
 * Eclipse workspace.</p>
 * 
 * <p>The synchronizer automatically monitors the file associated with an editor
 * ({@link IFileEditorInput} only).  Optionally, files can be added and monitored.</p>
 * 
 * <p>Editors can support synchronization by adapting to {@link ISynchronizedEditorAdapter}.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Jerome Bozier
 * @version December 9, 2009
 * @since   January 26, 2005
 */
public class EditorSynchronizer implements IDisposable{	
	
	private IEditorPart editorPart = null;
	private ISynchronizedEditorAdapter synchronizedEditorAdapter = null;
	private FileEditorInput editorFile = null;
	private OSFile editorOSFile = null;
	
	private static class SynchronizedRunnable
	implements Runnable
	{
		private AllInOneListener listener; 
		private Runnable runnable;
		
		public SynchronizedRunnable(AllInOneListener listener, Runnable runnable)
		{
			this.listener = listener;
			this.runnable = runnable;
		}
		
		/**
		 * @see java.lang.Runnable#run()
		 */
		public void run()
		{
			synchronized(listener)
			{
				boolean initialState = listener.isResourceChangeListenerEnabled();
				try
				{
					listener.setResourceChangeListenerEnabled(false);
					runnable.run();
				}
				finally
				{
					if(listener != null)
						listener.setResourceChangeListenerEnabled(initialState);
				}
				listener = null;
				runnable = null;
			}
		}
	}
	
	private AllInOneListener listener;
	
	/**
	 * Returns whether the specified file is ready only.
	 * @param file
	 * @return boolean
	 */

	public static boolean isReadOnly(FileEditorInput workspaceFile)
	{
		if(workspaceFile == null)
            //- jerome.gout@fr.ibm.com: #81225, if file is open from CVS the workspacefile is null then it is read-only
			return true;
			
		IPath path = workspaceFile.getPath();
		if(path == null)
			path = workspaceFile.getFile().getFullPath();
			if(path == null)
				return workspaceFile.exists();

		File file = new File(path.toString());
		if(file.exists())
			return !file.canWrite();
			
		return false;
	}

	
	/**
	 * Returns whether the specified file is ready only.
	 * @param file
	 * @return boolean
	 */

	public static boolean exists(FileEditorInput workspaceFile)
	{
		if(workspaceFile == null)
			return false;
		IPath path = workspaceFile.getPath();
		if(path == null)
			path = workspaceFile.getFile().getFullPath();
			if(path == null)
				return workspaceFile.exists();

		return new File(path.toString()).exists();
	}
	
	
	/**
	 * Returns the workspace file associated with the given editor input or <code>null</code>
	 * if the specified editor input does not represent a file in the workspace..
	 * @param editorInput
	 * @return IFile
	 */
	
	public static FileEditorInput getEditorFile(IEditorInput editorInput)
	{
		if(editorInput != null)
		{
			if(editorInput instanceof IFileEditorInput)
			{
				IFile iFile = ((IFileEditorInput)editorInput).getFile();
				if(iFile != null)
					return new FileEditorInput(iFile);
			}
		}
		return null; 		
	}

	/**
	 * Constructor for ResourceEditorSynchronizer
	 * @param editorPart 
	 * @throws IllegalArgumentException if the editor part is <code>null</code>
	 * or if the editor part's site is <code>null</code>.
	 */
	public EditorSynchronizer(IEditorPart editorPart)
	throws IllegalArgumentException
	{
		this.editorPart = editorPart;
		if((editorPart == null) || (editorPart.getSite() == null))
			throw new IllegalArgumentException(UIMessages._10);
		
		synchronizedEditorAdapter = (ISynchronizedEditorAdapter)editorPart.getAdapter(ISynchronizedEditorAdapter.class);
		listener = new AllInOneListener();
		
		editorFile = getEditorFile(editorPart.getEditorInput());
		if(editorFile != null)
		{
			editorFile.getFile().getWorkspace().addResourceChangeListener(listener, IResourceChangeEvent.POST_CHANGE);
			editorOSFile = new OSFile(editorFile);
		}
		
		PlatformUI.getWorkbench().addWindowListener(listener);
		editorPart.getSite().getPage().addPartListener((IPartListener2)listener);
	}

	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		//Check if it was already disposed.
		if(listener == null)
			return;
			
		if(editorFile != null)
		{
			editorFile.getFile().getWorkspace().removeResourceChangeListener(listener);
			editorFile = null;
		}
		PlatformUI.getWorkbench().removeWindowListener(listener);
		editorPart.getSite().getPage().removePartListener((IPartListener2)listener);
		
		listener = null;
		editorPart = null;
		synchronizedEditorAdapter = null;
		editorOSFile = null;
	}
	
	/**
	 * Returns the editor's file.
	 * @return IFile
	 */

	public FileEditorInput getEditorFile()
	{
		return editorFile;
	}
	
	protected OSFile getEditorOSFile()
	{
		return editorOSFile;
	}
	
	/**
	 * <p>Updates the editor input of the synchronizer.</p>
	 * 
	 * <p>If the new editor input is <code>null</code>, not an 
	 * {@link FileEditorInput}, or contains a <code>null</code> {@link IFile}
	 * (see {@link #getEditorFile(IEditorInput)}, the synchronizer is 
	 * 'disconnected' from a file.</p>
	 * 
	 * @param newEditorInput The new editor input.
	 * @see #getEditorFile(IEditorInput)
	 */
    protected void updateEditorSynchronizer(IEditorInput newEditorInput) {
    	
    	//Remove the listener for the previous editor input:
    	editorFile.getFile().getWorkspace().removeResourceChangeListener(listener);
    	
    	//Update the editor file:
    	editorFile = getEditorFile(newEditorInput);
    	
    	if (editorFile != null) {
    		
        	//Update the OS file:
    		editorOSFile = new OSFile(editorFile);
    		
    		listener = new AllInOneListener();
    		
        	//Add the listener for the new editor input:
    		editorFile.getFile().getWorkspace().addResourceChangeListener(listener, IResourceChangeEvent.POST_CHANGE);
    	}
    	else{
    		editorOSFile = null;
    	}
    }
    
	protected IEditorPart getEditorPart()
	{
		return editorPart;
	}
	
	protected ISynchronizedEditorAdapter getSynchronizedEditorAdapter()
	{
		return synchronizedEditorAdapter;
	} 

	/**
	 * Sets whether the resource changes are listened.
	 * @param enabled
	 */
	public void setResourceChangeListenerEnabled(boolean enabled)
	{
		listener.setResourceChangeListenerEnabled(enabled);
	}
		
	/**
	 * Returns whether the resource changes are listened.
	 * @return boolean
	 */
	public boolean isResourceChangeListenerEnabled()
	{
		return listener.isResourceChangeListenerEnabled();
	}
	
	/**
	 * Indicates to this synchronizer that the editor has been saved
	 */
	public synchronized void editorSaved()
	{
		if(getEditorOSFile() != null)
			getEditorOSFile().wasChanged();
	}

	/**
	 * Calls the Eclipse validate edit API for the editor file.
	 * @return <code>true</code> if the editor's file can be saved.
	 * @see org.eclipse.core.resources.IWorkspace#validateEdit(org.eclipse.core.resources.IFile[], java.lang.Object)
	 */
	public boolean validateEdit()
	{
		if(editorFile != null)
		{
			if(isReadOnly(editorFile))
			{
				IStatus status = ResourcesPlugin.getWorkspace().validateEdit(new IFile[]{editorFile.getFile()}, editorPart.getSite().getShell());
				if(status.getCode() != IStatus.OK)
					return false;
			}
		}
			
		return true;
	}
	
	/**
	 * Calls the Eclipse validate edit API for an array of files.
	 * @return <code>true</code> if the files can be saved.
	 * @see org.eclipse.core.resources.IWorkspace#validateEdit(org.eclipse.core.resources.IFile[], java.lang.Object)
	 */
	public boolean validateEdit(IFile[] files)
	{
		if(editorFile != null)
		{
			if(isReadOnly(editorFile))
			{
				IStatus status = ResourcesPlugin.getWorkspace().validateEdit(files, editorPart.getSite().getShell());
				if(status.getCode() != IStatus.OK)
					return false;
			}
		}
			
		return true;
	}

	/**
	 * Check the files looking for changes in the OS file system.
	 * 
	 * <p>This method is not invoked when workspace changes are notified.
	 */
	protected void checkOSFiles()
	{
		if(getEditorOSFile() != null)
		{
			if(getEditorOSFile().wasDeleted())
			{
				if(editorFileDeleted())
					return;
			}
			
			else if(getEditorOSFile().wasChanged())
			{
				if(editorFileChanged())
					return;
			}
			
			else if(getEditorOSFile().writeAccessChanged())
			{
				if(editorFileWriteAccessChanged(getEditorOSFile().isReadOnly()))
					return;				
			}
		}
	}
	
	/**
	 * This method is invoked when the editor file has been deleted.
	 * @return <code>true</code> if this synchronizer should stop
	 * its work.
	 */
	protected boolean editorFileDeleted()
	{
		Runnable runnable = new Runnable()
		{
			public void run()
			{
				if((getSynchronizedEditorAdapter() != null) && getSynchronizedEditorAdapter().editorFileDeleted())
				{
					return;
				}

				boolean close = !getEditorPart().isDirty();
				if(getEditorPart().isDirty())
				{
					if(isActiveEditor(getEditorPart()))
					{
						MessageDialog messageDialog = new MessageDialog(getEditorPart().getSite().getShell(), 
						UIMessages._41, null, UIMessages._42,
						MessageDialog.QUESTION,	new String[]{UIMessages._45, UIMessages._46}, 0);
						
						close = (messageDialog.open() == 1);
					}
					else if(getEditorOSFile() != null)
					{
						getEditorOSFile().setIsDeleted();
					}
				}
				
				if(close)
				{
					getEditorPart().getSite().getPage().closeEditor(getEditorPart(), false);			
				}
				else if(getSynchronizedEditorAdapter() != null)
				{
					getSynchronizedEditorAdapter().doSaveEditorFile(true);
				}
			}
		};
		return execute(runnable);		
	}
	
	/**
	 * This method is invoked when the editor file has changed.
	 * @return <code>true</code> if this synchronizer should stop
	 * its work.
	 */
	protected boolean editorFileChanged()
	{
		Runnable runnable = new Runnable()
		{
			public void run()
			{
				if((getSynchronizedEditorAdapter() != null) && getSynchronizedEditorAdapter().editorFileChanged())
				{
					return;
				}

				if(getEditorFile() == null)
					return;

				try
				{
//					getEditorFile().refreshLocal(IResource.DEPTH_ONE, null);
					getEditorFile().getFile().refreshLocal(IResource.DEPTH_ONE, null);
				}
				catch (CoreException e)
				{
					HyadesUIPlugin.logError(e);
				}

				if(isActiveEditor(getEditorPart()))
				{
					if(MessageDialog.openQuestion(getEditorPart().getSite().getShell(), UIMessages._43, UIMessages._44))
					{
						if(getSynchronizedEditorAdapter() != null)
							getSynchronizedEditorAdapter().reload();				
					}
				}
				else if(getEditorOSFile() != null)
				{
					getEditorOSFile().setIsChanged();
				}	
			}
		};	
		return execute(runnable);	
	}
	
	/**
	 * This method is invoked when the editor file became read only.
	 * @return <code>true</code> if this synchronizer should stop
	 * its work.
	 */
	protected boolean editorFileWriteAccessChanged(final boolean isReadOnly)
	{		
		final AllInOneListener finalListener = listener; 
		Runnable runnable = new Runnable()
		{
			public void run()
			{
				if(finalListener.isResourceChangeListenerEnabled())
					return;

				if(getSynchronizedEditorAdapter() != null)
					getSynchronizedEditorAdapter().editorFileWriteAccessChanged(isReadOnly);
			}
		};
		return execute(runnable);
	}
		
	/**
	 * Handles modifications in one or more of the monitored files.  The
	 * lists contains the modified IFiles.
	 * @param changedFiles
	 * @param removedFiles
	 */
	protected void handle(final List changedFiles, final List removedFiles)
	{
		Runnable runnable = new Runnable()
		{
			public void run()
			{
				if(getSynchronizedEditorAdapter() != null)
					getSynchronizedEditorAdapter().handle(changedFiles, removedFiles);
			}
		};
		execute(runnable);
	}

	/**
	 * Utility method to invoke each operation method.
	 * @param runnable
	 * @return <code>true</code> if this synchronizer should stop
	 * its work.
	 */
	protected boolean execute(Runnable runnable)
	{
		getEditorPart().getSite().getShell().getDisplay().asyncExec(new SynchronizedRunnable(listener, runnable));
		return false;		
	}





	private static class OSFile
	{
		private File file;
		private long lastModified;
		private boolean writable = false;
		private boolean exists = false;
		
		public OSFile(FileEditorInput workspaceFile)
		{
			IPath path = workspaceFile.getPath();
			if(path == null)
				return;
				
			file = new File(path.toOSString());
			if(file.exists())
			{
				lastModified = file.lastModified();
				writable = file.canWrite();
				exists = file.exists();
			}
		}

		public boolean writeAccessChanged()
		{
			if(file != null)
			{
				if(!file.exists())
					return false;
					
				boolean canWrite = file.canWrite();
				if(canWrite != writable)
				{
					writable = canWrite;
					return true;
				}
			}
			
			return false;
		}
		
		public boolean wasDeleted()
		{
			if(file != null)
			{
				if(exists && (!file.exists()))
				{
					exists = false;
					return true;
				}
			}
				
			return false;
		}
		
		public boolean wasChanged()
		{
			if(file != null)
			{
				if(!file.exists())
					return false;

				long newLastModified = file.lastModified();
				if(newLastModified != lastModified)
				{
					lastModified = newLastModified;
					return true;
				}
			}
			return false;
		}
		
		public void setIsChanged()
		{
			lastModified = lastModified-1;
		}

		public void setIsDeleted()
		{
			exists = true;
		}
		
		public boolean isReadOnly()
		{
			if(file != null)
			{
				if(file.exists())
					return !file.canWrite();
			}
			return false;
		}
		
		public boolean exists()
		{
			if(file != null)
				return file.exists();
			return true;
		}
	}
	
	private class ResourceDeltaVisitor
	implements IDisposable, IResourceDeltaVisitor
	{
		private List changedFiles = new ArrayList();
		private List removedFiles = new ArrayList();

		/**
		 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
		 */
		public void dispose()
		{
			changedFiles.clear();
			removedFiles.clear();
		}

		/**
		 * @see org.eclipse.core.resources.IResourceDeltaVisitor#visit(org.eclipse.core.resources.IResourceDelta)
		 */
		public boolean visit(IResourceDelta delta)
		{
			if(delta.getResource().getType() == IResource.FILE)
			{
				if((delta.getKind() & (IResourceDelta.CHANGED | IResourceDelta.REMOVED)) != 0)
				{
					IFile file = (IFile)delta.getResource();
					
					if((delta.getKind() & IResourceDelta.REMOVED) != 0)
					{
						if(getEditorFile() != null && file.equals(getEditorFile().getFile()))
						{
							if(getEditorOSFile() != null)
								getEditorOSFile().wasDeleted();
								
							if(editorFileDeleted())
							{
								dispose();
								return false;
							}
						}
					}						
					else
					{
						if(getEditorFile() != null && file.equals(getEditorFile().getFile()) && (delta.getFlags() != IResourceDelta.MARKERS))
						{
							if(getEditorOSFile() != null)
								getEditorOSFile().wasChanged();

							if(editorFileChanged())
							{
								dispose();
								return false;
							}
						}
					}
				}
			}
			return true;
		}
		
		/**
		 * Returns a list with the changed IFile that are associated with this
		 * editor as indicated by {@link FileEditorExtension#isAssociatedFile(IFile)}
		 * @return List
		 */
		public List getChangedFiles()
		{
			return changedFiles;
		}

		/**
		 * Returns a list with the removed IFile that are associated with this
		 * editor as indicated by {@link FileEditorExtension#isAssociatedFile(IFile)}
		 * @return List
		 */
		public List getRemovedFiles()
		{
			return removedFiles;
		}
	}
	
	private class AllInOneListener
	implements IPartListener, IPartListener2, IWindowListener, IResourceChangeListener
	{
		private boolean resourceChangeListenerEnabled = true;
		
		/**
		 * Sets whether the resource changes are listened.
		 * @param enabled
		 */
		public void setResourceChangeListenerEnabled(boolean enabled)
		{
			resourceChangeListenerEnabled = enabled;
		}
		
		/**
		 * Returns whether the resource changes are listened.
		 * @return boolean
		 */
		public boolean isResourceChangeListenerEnabled()
		{
			return resourceChangeListenerEnabled;
		}
		
		/**
		 * @see org.eclipse.ui.IPartListener#partActivated(org.eclipse.ui.IWorkbenchPart)
		 */
		public void partActivated(IWorkbenchPart part)
		{
			if(!resourceChangeListenerEnabled)
				return;

			if(part == getEditorPart())
				checkOSFiles();
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener#partBroughtToTop(org.eclipse.ui.IWorkbenchPart)
		 */
		public void partBroughtToTop(IWorkbenchPart part)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener#partClosed(org.eclipse.ui.IWorkbenchPart)
		 */
		public void partClosed(IWorkbenchPart part)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener#partDeactivated(org.eclipse.ui.IWorkbenchPart)
		 */
		public void partDeactivated(IWorkbenchPart part)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener#partOpened(org.eclipse.ui.IWorkbenchPart)
		 */
		public void partOpened(IWorkbenchPart part)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partActivated(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partActivated(IWorkbenchPartReference ref)
		{
			if(!resourceChangeListenerEnabled)
				return;

			if(ref.getPart(false) == getEditorPart())
				checkOSFiles();
			updateStatusLineWritableField(ref);
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partBroughtToTop(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partBroughtToTop(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partClosed(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partClosed(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partDeactivated(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partDeactivated(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partOpened(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partOpened(IWorkbenchPartReference ref)
		{
			updateStatusLineWritableField(ref);
		}
	
		private void updateStatusLineWritableField(IWorkbenchPartReference ref) {
			if(ref.getPart(false) == getEditorPart()) {
				IWorkbenchSite site = ref.getPart(false).getSite();
				if (site instanceof IEditorSite) {
					IEditorSite eSite = (IEditorSite) site;
					IEditorActionBarContributor contributor = eSite.getActionBarContributor();
					if (contributor instanceof HyadesEditorActionContributor) {
						HyadesEditorActionContributor hyadesContrib = (HyadesEditorActionContributor) contributor;
                        hyadesContrib.setWritable(editorOSFile != null && !editorOSFile.isReadOnly());
					}
				}
			}
		}

		/**
		 * @see org.eclipse.ui.IPartListener2#partHidden(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partHidden(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partVisible(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partVisible(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IPartListener2#partInputChanged(org.eclipse.ui.IWorkbenchPartReference)
		 */
		public void partInputChanged(IWorkbenchPartReference ref)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IWindowListener#windowActivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowActivated(IWorkbenchWindow window)
		{
			if(!resourceChangeListenerEnabled)
				return;

			if(window == PlatformUI.getWorkbench().getActiveWorkbenchWindow())
			{
				if((window.getActivePage() != null) && (window.getActivePage().getActivePart() == getEditorPart()))
					checkOSFiles();
			}
		}
	
		/**
		 * @see org.eclipse.ui.IWindowListener#windowDeactivated(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowDeactivated(IWorkbenchWindow window)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IWindowListener#windowClosed(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowClosed(IWorkbenchWindow window)
		{
		}
	
		/**
		 * @see org.eclipse.ui.IWindowListener#windowOpened(org.eclipse.ui.IWorkbenchWindow)
		 */
		public void windowOpened(IWorkbenchWindow window)
		{
		}
	
		/**
		 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
		 */
		public void resourceChanged(IResourceChangeEvent event)
		{
			if(!resourceChangeListenerEnabled)
				return;
				
			IResourceDelta delta = event.getDelta();
			ResourceDeltaVisitor visitor = new ResourceDeltaVisitor();
			try
			{
				delta.accept(visitor);
	
				if((!visitor.getChangedFiles().isEmpty()) && (!visitor.getRemovedFiles().isEmpty()))
				{
					handle(visitor.getChangedFiles(), visitor.getRemovedFiles());
					visitor.dispose();
				}			
			}
			catch(CoreException e)
			{
				HyadesUIPlugin.logError(e);
			}
		}
	}
	
	protected boolean isActiveEditor(IEditorPart editorPart)
	{
		try
		{
			return (editorPart.getSite().getPage().getActiveEditor() == editorPart);
		}
		catch(RuntimeException re)
		{
			return false;
		}
	}
}