/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ActionHandlerListener.java,v 1.7 2010/03/30 15:15:24 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.editor.action;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.jface.action.MenuManager;
import org.eclipse.jface.window.ApplicationWindow;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.MenuEvent;
import org.eclipse.swt.events.MenuListener;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.IActionBars;
import org.eclipse.ui.IEditorSite;
import org.eclipse.ui.IPartListener;
import org.eclipse.ui.IViewSite;
import org.eclipse.ui.IWorkbenchActionConstants;
import org.eclipse.ui.IWorkbenchPage;
import org.eclipse.ui.IWorkbenchPart;
import org.eclipse.ui.IWorkbenchPartSite;
import org.eclipse.ui.IWorkbenchWindow;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.actions.ActionFactory;

/** 
 * <p>Action handler for the following actions:</p>
 * 
 * <ul>
 * <li>Cut</li>
 * <li>Copy</li>
 * <li>Paste</li>
 * <li>Delete</li>
 * <li>Select All</li>
 * <li>Revert</li>
 * </ul>
 * 
 * 
 * @author  Patrick Nedelec
 * @author  Paul Slauenwhite
 * @version March 30, 2010
 * @since   February 27, 2007
 */
public class ActionHandlerListener implements IPartListener, MenuListener {
	
	public static final ActionHandlerListener DEFAULT = new ActionHandlerListener();
	
	private List workbenchWindows;
	private List registeredParts;
	
    protected EditAction cut;
    protected EditAction copy;
    protected EditAction paste;
    protected EditAction selectAll;
    protected EditAction delete;
    protected RevertAction revert;

	/**
	 * Constructor for ActionHandlerListener.
	 */
	public ActionHandlerListener()
	{
		super();
		
		workbenchWindows = new ArrayList();
		registeredParts = new ArrayList();
		
		createEditActions();
	}

	protected void createEditActions()
	{
		cut = new CutAction();
		copy = new CopyAction();
		paste = new PasteAction();
		selectAll = new SelectAllAction();
		delete = new DeleteAction();
		revert = new RevertAction();
	}

	protected void listenToWorkbenchWindow(IWorkbenchWindow workbenchWindow)
	{
		if (!workbenchWindows.contains(workbenchWindow))
		{
			workbenchWindows.add(workbenchWindow);
			workbenchWindow.getPartService().addPartListener(this);
			MenuManager menuManager = (MenuManager)((ApplicationWindow)workbenchWindow).getMenuBarManager().findMenuUsingPath(IWorkbenchActionConstants.M_EDIT);
			if(menuManager != null)
				menuManager.getMenu().addMenuListener(this);
			menuManager = (MenuManager)((ApplicationWindow)workbenchWindow).getMenuBarManager().findMenuUsingPath(IWorkbenchActionConstants.M_FILE);
			if(menuManager != null)
				menuManager.getMenu().addMenuListener(this);			
		}
	}

	protected IActionBars getActionBars(IWorkbenchPart part)
	{
		IActionBars actionBars = null;
		if (part != null)
		{
			IWorkbenchPartSite partSite = part.getSite();

			if (partSite instanceof IViewSite )
				actionBars = ((IViewSite)partSite).getActionBars();
			else if (partSite instanceof IEditorSite)
				actionBars = ((IEditorSite)partSite).getActionBars();			

		}
		
		return actionBars;
	}

	/**
	 * Method connectPart.  Ensure we are already listening to the workbenchwindow,
	 * register the part's id and then connect the cut/copy/paste actions
	 * @param part
	 */
    public void connectPart(IWorkbenchPart part) {
        if (filter(part)) {
            return;
        }
        IWorkbenchWindow wbw = part.getSite().getWorkbenchWindow();
        listenToWorkbenchWindow(wbw);
        registerPartId(part);
        connectCutCopyPasteActions(part);
    }

    protected boolean filter(IWorkbenchPart part) {
        return false;
    }

	/**
	 * Method isRegisteredPart.  Returns whether the part has already been
	 * registered as being a candidate for the generic cut/copy/paste actions.
	 * 
	 * @param part
	 * @return boolean
	 */
	protected boolean isRegisteredPart(IWorkbenchPart part)
	{
		String partId = part.getSite().getId();
		return registeredParts.contains(partId);
	}

	protected void registerPartId(IWorkbenchPart part)
	{
		if (!isRegisteredPart(part))
		{
			String partId = part.getSite().getId();
			registeredParts.add(partId);
		}
	}

	/**
	 * Method connectCutCopyPasteActions.  Connect the cut/copy/paste actions
	 * to the workbench part.
	 * 
	 * @param part
	 */
	protected void connectCutCopyPasteActions(IWorkbenchPart part)
	{
		revert.setWorkbenchPart(part);
		IActionBars actionBars = getActionBars(part);
		if (actionBars != null)
		{
			part.getSite().getKeyBindingService().registerAction(cut);
			part.getSite().getKeyBindingService().registerAction(copy);
			part.getSite().getKeyBindingService().registerAction(paste);
			part.getSite().getKeyBindingService().registerAction(selectAll);
			part.getSite().getKeyBindingService().registerAction(delete);
			part.getSite().getKeyBindingService().registerAction(revert);
			
			connectCutCopyPasteActions(actionBars);
		}
	}

	/**
	 * Method connectCutCopyPasteActions.  Only set the actionhandlers if
	 * there isn't one active for the appropriate action.
	 * 
	 * @param actionBars
	 */
	protected void connectCutCopyPasteActions(IActionBars actionBars)
	{
		if (actionBars.getGlobalActionHandler(ActionFactory.CUT.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.CUT.getId(), cut);
		}
		if (actionBars.getGlobalActionHandler(ActionFactory.COPY.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.COPY.getId(), copy);
		}
		if (actionBars.getGlobalActionHandler(ActionFactory.PASTE.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.PASTE.getId(), paste);
		}
		if (actionBars.getGlobalActionHandler(ActionFactory.SELECT_ALL.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.SELECT_ALL.getId(), selectAll);
		}
		if (actionBars.getGlobalActionHandler(ActionFactory.DELETE.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.DELETE.getId(), delete);
		}
		if (actionBars.getGlobalActionHandler(ActionFactory.REVERT.getId()) == null)
		{
			actionBars.setGlobalActionHandler(ActionFactory.REVERT.getId(), revert);
		}
		enableActions();
		actionBars.updateActionBars();
	}

	protected IWorkbenchPart getWorkbenchPart()
	{
		IWorkbenchWindow wbw = PlatformUI.getWorkbench().getActiveWorkbenchWindow();
		if (wbw == null)
			return null;

		IWorkbenchPage wbp = wbw.getActivePage();
		if (wbp == null)
			return null;

		return wbp.getActivePart();
	}

	protected void updateActions()
	{
		IWorkbenchPart part = getWorkbenchPart();
		if (part != null && isRegisteredPart(part))
		{
			revert.setWorkbenchPart(part);
			IActionBars actionBars = getActionBars(part);

			Control focusControl = cut.getFocusControl();
			if (isValidFocusControl() && actionBars != null)
			{
				String clipboardText = paste.getClipboardText();
				String selectionText = copy.getSelectionText();
				boolean isReadOnly = paste.isReadOnlyFocusControl();
				boolean clipboardNotEmpty = clipboardText != null && !clipboardText.equals("");
				boolean textSelected = selectionText != null && !selectionText.equals("");

				cut.setEnabled(!isReadOnly && textSelected);
				copy.setEnabled(textSelected && !(focusControl instanceof Combo && isReadOnly));
				paste.setEnabled(!isReadOnly && clipboardNotEmpty);
				delete.setEnabled(!isReadOnly && (focusControl instanceof StyledText));
				selectAll.setEnabled(!(focusControl instanceof Combo));
			}
			else
			{
				cut.setEnabled(false);
				copy.setEnabled(false);
				paste.setEnabled(false);
				delete.setEnabled(false);
				selectAll.setEnabled(false);
			}
			actionBars.updateActionBars();
		}
	}

	protected boolean isValidFocusControl()
	{
		// delgate to one of the edit actions
		return cut.isValidFocusControl();
	}

	public void enableActions()
	{
		cut.setEnabled(true);
		copy.setEnabled(true);
		paste.setEnabled(true);
		delete.setEnabled(true);
		selectAll.setEnabled(true);
	}

	/**
	 * Connect the cut/copy/paste actions if a registered part is activated.
	 * 
	 * @see IPartListener#partActivated(IWorkbenchPart)
	 */
	public void partActivated(IWorkbenchPart part)
	{
		if (isRegisteredPart(part))
		{
			connectCutCopyPasteActions(part);
		}
	}

	/**
	 * @see IPartListener#partBroughtToTop(IWorkbenchPart)
	 */
	public void partBroughtToTop(IWorkbenchPart part)
	{
	}
	
	/**
	 * @see IPartListener#partClosed(IWorkbenchPart)
	 */
	public void partClosed(IWorkbenchPart part)
	{
	}
	
	/**
	 * @see IPartListener#partDeactivated(IWorkbenchPart)
	 */
	public void partDeactivated(IWorkbenchPart part)
	{
	}
	
	/**
	 * @see IPartListener#partOpened(IWorkbenchPart)
	 */
	public void partOpened(IWorkbenchPart part)
	{
	}

	/**
	 * If the menu is hidden on a registered part, then renable all actions.  This
	 * takes away the need for us to monitor the control traversal.  If the individual
	 * action isn't applicable when it is invoked, the action becomes a no op.
	 * @see MenuListener#menuHidden(MenuEvent)
	 */
	public void menuHidden(MenuEvent e)
	{
		IWorkbenchPart part = getWorkbenchPart();
		if (part != null && isRegisteredPart(part))
		{
			enableActions();

			IActionBars actionbars = getActionBars(part);
			actionbars.updateActionBars();
		}
	}

	/**
	 * Update the cut/copy/paste enablement (if a registered part is active) just
	 * before showing the edit menu.
	 * 
	 * @see MenuListener#menuShown(MenuEvent)
	 */
	public void menuShown(MenuEvent e)
	{
		updateActions();
	}

	/**
	 * 
	 */
	public void disableActions() {
		cut.setEnabled(false);
		copy.setEnabled(false);
		paste.setEnabled(false);
		selectAll.setEnabled(false);
		delete.setEnabled(false);
		revert.setEnabled(false);
	}
    /**
     * 
     * @return the cut action
     * @author pnedelec
     * @since 4.4
     */
    public EditAction getCutAction() {
        return cut;
    }
    
    /**
     * 
     * @return the copy action
     * @author pnedelec
     * @since 4.4
     */
    public EditAction getCopyAction() {
        return copy;
    }
    
    /**
     * 
     * @return the paste action
     * @author pnedelec
     * @since 4.4
     */
    public EditAction getPasteAction() {
        return paste;
    }
  
    /**
     * 
     * @return the delete action
     * @author pnedelec
     * @since 4.4
     */
    public EditAction getDeleteAction() {
        return delete;
    }
    
    /**
     * 
     * @return the select all action
     * @author pnedelec
     * @since 4.4
     */
    public EditAction getSelectAllAction() {
        return selectAll;
    }
}
