/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NavigatorViewerSorter.java,v 1.3 2010/04/14 19:29:36 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.navigator;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.hyades.ui.internal.logicalfolder.LogicalFolder;
import org.eclipse.hyades.ui.internal.util.HyadesViewerSorter;
import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.INamedElement;
import org.eclipse.jface.viewers.Viewer;

/**
 * <p>Sorter for viewers that display items of type:</p>
 * 
 * <ul>
 * <li>{@link IResource}</li>
 * <li>{@link INamedElement}</li>
 * <li>{@link LogicalFolder}</li>
 * <li>{@link IAdaptable} with an adapter for {@link IResource}</li>
 * </ul>
 * 
 * <p>The sorter supports two sort criteria:</p>
 * 
 * <ul>
 * <li>{@link #NAME}: Folders are given order precedence, followed by files.
 * Within these two groups resources are ordered by name.  All name comparisons
 * are case-insensitive.</li>
 * <li>{@link #TYPE}: Folders are given order precedence, followed by files.
 * Within these two groups resources are ordered by extension.</li>
 * </ul>
 * 
 * <p>All extension comparisons are case-insensitive.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version April 14, 2010
 * @since   January 26, 2005
 * @see     HyadesViewerSorter
 * @see     org.eclipse.ui.views.navigator.ResourceSorter
 */
public class NavigatorViewerSorter extends HyadesViewerSorter implements IDisposable {
	
	private INavigator navigator = null;
	private int criteria = -1;
	
	/**
	 * Change type to be passed to the navigator's <code>handleChange(int)</code>
	 * method.
	 */
	public static final int SORTER_TYPE = NavigatorViewerSorter.class.hashCode();
	
	/**
	 * Constructor argument value that indicates to sort items by name.
	 */
	public final static int NAME = 1;

	/**
	 * Constructor argument value that indicates to sort items by extension.
	 */
	public final static int TYPE = 2;
	
	/**
	 * <p>The resource value delimiter.</p>
	 * 
	 * @see #getResourceValue(Viewer, IResource)
	 */
	public final static String RESOURCE_VALUE_DELIMITER = "_+_"; //$NON-NLS-1$
	
	/**
	 * Constructor for NavigatorViewerSorter
	 */
	public NavigatorViewerSorter(INavigator navigator, int criteria)
	{
		super();
		
		this.navigator = navigator;
		
		if((criteria != NAME) && (criteria != TYPE))
			this.criteria = NAME;
		else
			this.criteria = criteria;
	}
	
	/**
	 * @see org.eclipse.hyades.ui.util.IDisposable#dispose()
	 */
	public void dispose()
	{
		navigator = null;
	}
	
	/**
	 * Returns this sorter's navigator
	 * @return INavigator
	 */
	public INavigator getNavigator()
	{
		return navigator;
	}
	
	/**
	 * Sets the sort criteria of this this sorter.
	 * @param the sort criterion: one of <code>NAME</code> or <code>TYPE</code>.  
	 * Any other value is not considered.
	 */
	public void setCriteria(int newCriteria)
	{
		if(criteria == newCriteria)
			return;
			
		if((newCriteria != NAME) && (newCriteria != TYPE))
			return;
			
		criteria = newCriteria;
		if(navigator.getViewer() != null)
			navigator.getViewer().refresh();
		
		navigator.handleChange(SORTER_TYPE);
	}	
	
	/**
	 * Returns the sort criteria of this this sorter.
	 * @return the sort criterion: one of <code>NAME</code> or <code>TYPE</code>
	 */
	public int getCriteria()
	{
		return criteria;
	}

	/**
	 * @see org.eclipse.jface.viewers.ViewerSorter#category(java.lang.Object)
	 */
	public int category(Object element)
	{
		if(element instanceof IProject)
			return 1000;
			
		if(element instanceof IFolder)
			return 400;

		if(element instanceof LogicalFolder)
			return 200;			
		
		return 600;
	}

	/**
	 * @see org.eclipse.hyades.ui.internal.util.HyadesViewerSorter#getValue(org.eclipse.jface.viewers.Viewer, java.lang.Object)
	 */
	protected String getValue(Viewer viewer, Object e)
	{
		String value = null;

		if(e instanceof LogicalFolder)
		{
			value = getLogicalFolderValue(viewer, (LogicalFolder)e);
			if(value != null)
				return value;
		}

		if(e instanceof IResource)
		{
			value = getResourceValue(viewer, (IResource)e);
			if(value != null)
				return value;
		}
		
		if(e instanceof IAdaptable)
		{
			value = getAdaptableValue(viewer, (IAdaptable)e);
			if(value != null)
				return value;
		}
		
		if(e instanceof INamedElement)
		{
			value = getNamedElementValue(viewer, (INamedElement)e);
			if(value != null)
				return value;
		}
		
		return super.getValue(viewer, e);
	}
	
	/**
	 * Returns the value for a LogicFolder or <code>null</code> if no value
	 * can be assigned.  This implementation returns always the same value
	 * so the initial order should be preserved.
	 * @param viewer
	 * @param logicalFolder
	 * @return String
	 */
	protected String getLogicalFolderValue(Viewer viewer, LogicalFolder logicalFolder)
	{
		return "0";
	}

	/**
	 * <p>Resolves the resource value with one of the following values:</p>
	 * 
	 * <ul>
	 * <li>&lt;resource name&gt;_+_[&lt;file extension ({@link IFile} only)&gt;] ({@link #NAME} category).</li>
	 * <li>[&lt;file extension ({@link IFile} only)&gt;]_+_&lt;resource name&gt; ({@link #TYPE} category).</li>
	 * </ul>
	 *  
	 * @param viewer Currently not used.
	 * @param resource The resource ({@link IContainer} or {@link IFile}).
	 * @return String The resource value (see above).
	 * @see #RESOURCE_VALUE_DELIMITER
	 */
	protected String getResourceValue(Viewer viewer, IResource resource){
		
		String resourceName = resource.getName().trim();
		String fileExtension = resource.getFileExtension();
		
		if((resource instanceof IContainer) || (fileExtension == null)){
			fileExtension = ""; //$NON-NLS-1$
		}
		
		//Remove the file extension from the file name:
		else if(resourceName.endsWith(".".concat(fileExtension.trim()))){ //$NON-NLS-1$
			resourceName = resourceName.substring(0, (resourceName.length() - fileExtension.length() - 1));
		}
			
		if(getCriteria() == NAME){
			return (resourceName + RESOURCE_VALUE_DELIMITER + fileExtension);
		}
		
		return (fileExtension + RESOURCE_VALUE_DELIMITER + resourceName); 
	}

	/**
	 * Returns the value for an adaptable or <code>null</code> if no value
	 * can be assigned. 
	 * @param viewer
	 * @param adaptable
	 * @return String
	 */
	protected String getAdaptableValue(Viewer viewer, IAdaptable adaptable)
	{
		IResource resource = (IResource)adaptable.getAdapter(IResource.class);
		if(resource == null)
			return null;
		return getResourceValue(viewer, resource);
	}
	
	/**
	 * Returns the value for an namedElement or <code>null</code> if no value
	 * can be assigned. 
	 * @param viewer
	 * @param namedElement
	 * @return String
	 */
	protected String getNamedElementValue(Viewer viewer, INamedElement namedElement)
	{
		return namedElement.getName();
	}	
}
