/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AssociationMappingPreferencePage.java,v 1.8 2010/05/18 19:55:32 jwest Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.preference;

import java.util.Arrays;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;

import org.eclipse.hyades.ui.extension.AssociationDescriptorLabelProvider;
import org.eclipse.hyades.ui.extension.IAssociationDescriptor;
import org.eclipse.hyades.ui.extension.IAssociationMapping;
import org.eclipse.hyades.ui.internal.util.ChooseElementDialog;
import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.preference.PreferencePage;
import org.eclipse.osgi.util.NLS;
import org.eclipse.osgi.util.TextProcessor;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.TabFolder;
import org.eclipse.swt.widgets.TabItem;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.IWorkbenchPreferencePage;
import org.eclipse.ui.PlatformUI;

/**
 * <p>Abstract preference page to define the default descriptor for a group of 
 * mappings.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul Slauenwhite
 * @version May 5, 2010
 * @since   January 26, 2005
 */
public abstract class AssociationMappingPreferencePage extends PreferencePage implements IWorkbenchPreferencePage, Listener {

	private IWorkbench workbench;

	private IAssociationMapping selectionMapping;

	private Table selectionTable;
	private TabFolder detailTabFolder;
	private Map mappingByTable;
	
	private Button addAssociationButton;
	private Button removeAssociationButton;
	private Button defaultAssociationButton;
	private String contextHelpID;
	
	/**
	 * Constructor for AssociationMappingPreferencePage
	 */
	public AssociationMappingPreferencePage()
	{
		super();
		
		mappingByTable = new HashMap();
		noDefaultAndApplyButton();
	}
	
	public AssociationMappingPreferencePage(String contextHelpID)
	{
		super();
		
		mappingByTable = new HashMap();
		noDefaultAndApplyButton();
		this.contextHelpID = contextHelpID;
		
	}
	
	/**
	 * @see org.eclipse.ui.IWorkbenchPreferencePage#init(org.eclipse.ui.IWorkbench)
	 */
	public void init(IWorkbench workbench)
	{
		this.workbench = workbench;
	}
	
	/**
	 * Returns the workbench passed in the {@link #init(IWorkbench)} method.
	 * @return IWorkbench
	 */
	protected IWorkbench getWorkbench()
	{
		return workbench;
	}

	/**
	 * Returns the association mappings that are presented as the details of this
	 * preference page.
	 * @return IAssociationMapping[]
	 */
	abstract public IAssociationMapping[] getDetailMappings();
	
	/**
	 * Returns the selection mapping extension point.  The selection mapping is the
	 * one that is present on the table in the top of the preference page.
	 * 
	 * <p>If this method returns <code>null</code> then no selection table is 
	 * presented.
	 * 
	 * @return IAssociationMapping
	 */
	abstract protected IAssociationMapping getSelectionAssociationMapping();
	
	/**
	 * Returns the label for each mapping.
	 * @param associationMapping
	 * @return String
	 */
	abstract protected String getLabel(IAssociationMapping associationMapping);

	/**
	 * @see org.eclipse.jface.preference.IPreferencePage#performCancel()
	 */
	abstract public boolean performCancel();

	/**
	 * @see org.eclipse.jface.preference.IPreferencePage#performOk()
	 */
	abstract public boolean performOk();	

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
	 */
	public void dispose()
	{
		if(mappingByTable != null)
			mappingByTable.clear();
			
		selectionMapping = null;
		super.dispose();
	}

	/**
	 * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
	 */
	public void createControl(Composite parent)
	{
		super.createControl(parent);
		beforeIsPresented();
	}

	/**
	 * Subclasses can overwrite this method to perform some actions immediately
	 * before the page is presented to the user.  This method is invoke only once.
	 */
	protected void beforeIsPresented()
	{
	}

	/**
	 * @see org.eclipse.jface.preference.PreferencePage#createContents(org.eclipse.swt.widgets.Composite)
	 */
	protected Control createContents(Composite parent){
		
		initializeDialogUnits(parent);
		
		Composite pageComponent = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		pageComponent.setLayout(gridLayout);
		pageComponent.setLayoutData(GridDataUtil.createFill());

		IAssociationMapping mapping = getSelectionAssociationMapping();
		if((mapping != null) && (mapping.associationDescriptors().length > 0))
		{
			selectionMapping = mapping;
			Label label = new Label(pageComponent, SWT.LEFT);
			label.setText(getLabel(selectionMapping));
		
			selectionTable = new Table(pageComponent, SWT.SINGLE | SWT.BORDER | SWT.FULL_SELECTION);
			selectionTable.addListener(SWT.Selection, this);
			selectionTable.addListener(SWT.DefaultSelection, this);
		}
			
		IAssociationMapping[] mappings = getDetailMappings();
		if(mappings.length > 0)
		{
			if(selectionTable != null)
			{
				GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
				gridData.heightHint = selectionTable.getItemHeight()* (UIUtil.availableRows(parent)/16);
				selectionTable.setLayoutData(gridData);		

				Label spacer = new Label(pageComponent, SWT.LEFT);
				GridData data = new GridData();
				data.heightHint = 1;
				spacer.setLayoutData(data);
			}
			
			createDetailControl(pageComponent, mappings);
		}
		else
		{
			if(selectionTable != null)
				selectionTable.setLayoutData(GridDataUtil.createFill());
		}
		
		if(populateSelection())
		{
			selectionTable.select(0);
			
			Event event = new Event();
			event.widget = selectionTable;
			event.type = SWT.Selection;
			selectionTable.notifyListeners(SWT.Selection, event);
		}
		else
		{
			populateDetails();
		} 
		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, HyadesUIPlugin.getID() + ContextIds.ASSOC_TABLE);				
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(selectionTable, HyadesUIPlugin.getID() + ContextIds.ASSOC_TABLE);		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(defaultAssociationButton, HyadesUIPlugin.getID() + ContextIds.ASSOC_DEF_BTN);		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(addAssociationButton, HyadesUIPlugin.getID() + ContextIds.ASSOC_ADD_BTN);		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(removeAssociationButton, HyadesUIPlugin.getID() + ContextIds.ASSOC_DEL_BTN);		
//		PlatformUI.getWorkbench().getHelpSystem().setHelp(detailTabFolder, HyadesUIPlugin.getID() + ContextIds.ASSOC_DETAIL_TAB);	
		if (contextHelpID!=null && !contextHelpID.equals("")) {
			PlatformUI.getWorkbench().getHelpSystem().setHelp(parent, contextHelpID);				
			PlatformUI.getWorkbench().getHelpSystem().setHelp(selectionTable, contextHelpID);		
			PlatformUI.getWorkbench().getHelpSystem().setHelp(defaultAssociationButton, contextHelpID);		
			PlatformUI.getWorkbench().getHelpSystem().setHelp(addAssociationButton, contextHelpID);		
			PlatformUI.getWorkbench().getHelpSystem().setHelp(removeAssociationButton, contextHelpID);		
			PlatformUI.getWorkbench().getHelpSystem().setHelp(detailTabFolder, contextHelpID);		
		}
		return pageComponent;
	}
	
	/**
	 * Creates the selection control.
	 * @param pageComponent
	 */
	protected void createSelectionControl(Composite parent)
	{
		Label label = new Label(parent, SWT.LEFT);
		label.setText(getLabel(selectionMapping));
		
		selectionTable = new Table(parent, SWT.SINGLE | SWT.BORDER | SWT.FULL_SELECTION);
		selectionTable.addListener(SWT.Selection, this);
		selectionTable.addListener(SWT.DefaultSelection, this);

		GridData gridData = new GridData(GridData.FILL_HORIZONTAL);
		gridData.heightHint = selectionTable.getItemHeight()* (UIUtil.availableRows(parent)/16);
		selectionTable.setLayoutData(gridData);		
	}
	
	/**
	 * Creates the detail control.
	 * @param pageComponent
	 * @param mappings
	 */
	protected void createDetailControl(Composite parent, IAssociationMapping[] mappings)
	{
		Composite composite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.numColumns = 2;
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		composite.setLayout(gridLayout);
		composite.setLayoutData(GridDataUtil.createFill());
		
		detailTabFolder = new TabFolder(composite, SWT.TOP);
		detailTabFolder.setLayoutData(GridDataUtil.createFill());
		for(int i = 0, maxi = mappings.length; i < maxi; i++)
		{
			String label = getLabel(mappings[i]);
			if(label == null)
				label = "";
			
			Table table = createDetailTable(detailTabFolder, mappings[i]);			
			// workaround SWT table always set item width as the width of the first item.
			new TableColumn(table, SWT.LEFT, 0);
			detailTabFolder.addControlListener(new ControlListener() {
				public void controlMoved(ControlEvent e) {
				}
				public void controlResized(ControlEvent e) {
					Control[] children = detailTabFolder.getChildren();
					for(int i = 0; i < children.length; i++)
					{
						if(children[i] instanceof Table)
						{
							Table table = (Table)children[i];
							table.setRedraw(false);
							TableColumn column = table.getColumn(0);
							if(column != null)
								column.setWidth(detailTabFolder.getClientArea().width - 5);
							table.setRedraw(true);
							table.update();
						}
					}
				}
			});
			mappingByTable.put(table, mappings[i]);
			
			TabItem tabItem = new TabItem(detailTabFolder, SWT.NONE);
			tabItem.setText(label);
			tabItem.setControl(table);
		}
		detailTabFolder.setSelection(0);		
		
		createDetailButtonsControl(composite);			
		
	}

	/**
	 * Creates one detail table that presents the descriptors provided by one mapping.
	 * @param parent
	 * @param associationMapping
	 * @return Table
	 */
	protected Table createDetailTable(Composite parent, IAssociationMapping associationMapping)
	{
		Table table = new Table(parent, SWT.SINGLE | SWT.BORDER | SWT.FULL_SELECTION);
		table.setLayoutData(new GridData(GridData.FILL_BOTH));
		table.addListener(SWT.Selection, this);
		table.addListener(SWT.DefaultSelection, this);
		
		return table;
	}
	
	/**
	 * Creates the buttons of the detail control.
	 * @param parent
	 */
	protected void createDetailButtonsControl(Composite parent)
	{
		Composite buttonComposite = new Composite(parent, SWT.NONE);
		GridLayout gridLayout = new GridLayout();
		gridLayout.marginWidth = 0;
		gridLayout.marginHeight = 0;
		buttonComposite.setLayout(gridLayout);
		GridData gridData = new GridData();
		gridData.verticalAlignment = GridData.FILL;
		gridData.horizontalAlignment = GridData.FILL;
		buttonComposite.setLayoutData(gridData);

		new Label(buttonComposite, SWT.LEFT);

		if(selectionTable != null)
		{
			addAssociationButton= new Button(buttonComposite, SWT.PUSH);
			addAssociationButton.setText(UIMessages._11);
			addAssociationButton.addListener(SWT.Selection, this);
			setButtonLayoutData(addAssociationButton);		
	
			removeAssociationButton= new Button(buttonComposite, SWT.PUSH);
			removeAssociationButton.setText(UIMessages._12);
			removeAssociationButton.addListener(SWT.Selection, this);
			setButtonLayoutData(removeAssociationButton);
		}		

		defaultAssociationButton= new Button(buttonComposite, SWT.PUSH);
		defaultAssociationButton.setText(UIMessages._13);
		defaultAssociationButton.addListener(SWT.Selection, this);
		setButtonLayoutData(defaultAssociationButton);		
	}
	
	/**
	 * Populates the preference page selection table. 
	 * @return <code>true</code> if the table was populated or 
	 * <code>false</code> otherwise. 
	 */
	protected boolean populateSelection()
	{
		if(selectionMapping == null)
			return false;

		populate(selectionTable, selectionMapping.associationDescriptors(), null);
		return true;
	}

	/**
	 * Populates the detail tables.  If the selection table is available this method uses its
	 * selection to populate the detail tables.  If it is not available then the details are
	 * populated with the descriptors that are not associated to any particular type.
	 * @return <code>true</code> if the tables were populated or 
	 * <code>false</code> otherwise.
	 */
	protected boolean populateDetails()
	{
		if(mappingByTable == null)
			return false;
		
		IAssociationDescriptor descriptor = getAssociationDescriptor(selectionTable);	
		for (Iterator iterator = mappingByTable.keySet().iterator(); iterator.hasNext();)
			populateDetail(descriptor, (Table)iterator.next());
		
		return true;
	}
	
	/**
	 * Populates a detail table based on the association descriptor selected in the
	 * selection table. 
	 * @param selectionDescriptor
	 * @param table
	 */
	protected void populateDetail(IAssociationDescriptor selectionDescriptor, Table table)
	{
		String[] types = null;
		if(selectionDescriptor != null)
			types = selectionDescriptor.types();
			
		IAssociationMapping mapping = (IAssociationMapping)mappingByTable.get(table);
		IAssociationDescriptor[] descriptors = null;
		IAssociationDescriptor defaultDescriptor = null;
		
		if((types == null) || (types.length == 0))
		{
			descriptors = mapping.getAssociationDescriptors();
			defaultDescriptor = mapping.getDefaultAssociationDescriptor();
		}
		else
		{
			Set descriptorSet = new HashSet();
			for(int i = 0, maxi = types.length; i < maxi; i++)
				descriptorSet.addAll(Arrays.asList(mapping.getAssociationDescriptors(types[i])));
			descriptors = (IAssociationDescriptor[])descriptorSet.toArray(new IAssociationDescriptor[descriptorSet.size()]);
			defaultDescriptor = mapping.getDefaultAssociationDescriptor(types[0]);
		}

		populate(table, descriptors, defaultDescriptor);
	}
	
	/**
	 * Populate one table using the specified association descriptors.
	 * @param table
	 * @param associationDescriptors
	 * @param defaultAssociationDescriptor The default association descriptor.  This 
	 * argument should be <code>null</code> if there is no default.
	 */
	protected void populate(Table table, IAssociationDescriptor[] associationDescriptors, IAssociationDescriptor defaultAssociationDescriptor)
	{
		// sort alphabetically
		Arrays.sort(associationDescriptors, new Comparator() {
			public int compare(Object o1, Object o2) {
				if (o1 != null && o2 != null) {
					String name1 = ((IAssociationDescriptor)o1).getName();
					String name2 = ((IAssociationDescriptor)o2).getName();
					if (name1 != null && name2 != null) {
						return name1.compareTo(name2);
					}
					return o1.hashCode() - o2.hashCode();
				}
				return 0;
			}
		});
		
		table.removeAll();
		
		TableItem defaultItem = null;
		if(defaultAssociationDescriptor != null)
			defaultItem = new TableItem(table, SWT.NONE);
		
		for(int i = 0, maxi = associationDescriptors.length; i < maxi; i++)
		{
			TableItem item = null;
			if(associationDescriptors[i] == defaultAssociationDescriptor)
				item = defaultItem;
			else
				item = new TableItem(table, SWT.NONE);
				
			item.setData(associationDescriptors[i]);
			
			Image image = associationDescriptors[i].getImage();
			if(image != null)
				item.setImage(image);
	
			String label = getLabel(associationDescriptors[i]);
			if(item == defaultItem)
				label = NLS.bind(UIMessages._14, label);
			
			if(label != null) {
				// Process the string for proper BiDi conversion - Bug 273740
				label = TextProcessor.process(label);
			}
			
			item.setText(label);	
		}		
	}
	
	/**
	 * Returns the association descriptor selected in table or <code>null</code>
	 * if the number of selected items is different than 1.
	 * @param table
	 * @return IAssociationDescriptor
	 */
	protected IAssociationDescriptor getAssociationDescriptor(Table table)
	{
		if((table == null) || (table.getSelectionCount() != 1))
			return null;
		
		return (IAssociationDescriptor)table.getSelection()[0].getData();
	}
	
	/**
	 * Returns the label to be used when displaying an association descriptor
	 * in a table. 
	 * @param associationDescriptor
	 * @return String
	 */
	protected String getLabel(IAssociationDescriptor associationDescriptor)
	{
		if(associationDescriptor != null)
		{
			String label = associationDescriptor.getName();
			if(label != null)
				return label;
				
			label = associationDescriptor.getId();
			if(label != null)
				return label;				

			String[] types = associationDescriptor.types();
			if(types.length == 1)
				return types[0];
		}

		return "";
	}

	/**
	 * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
	 */
	public void handleEvent(Event event)
	{
		if(event.widget == selectionTable)
		{
			populateDetails();
		}
		else if(event.widget == addAssociationButton)
		{
			addAssociation((Table)detailTabFolder.getSelection()[0].getControl());
		}
		else if(event.widget == removeAssociationButton)
		{
			removeAssociation((Table)detailTabFolder.getSelection()[0].getControl());
		}
		else if(event.widget == defaultAssociationButton)
		{
			setDefaultAssociation((Table)detailTabFolder.getSelection()[0].getControl());
		}
		
		updateEnabledState();
	}
	
	/**
	 * Adds a descriptor to the mapping associated with the detail table. 
	 * @param detailTable
	 */
	protected void addAssociation(Table detailTable)
	{
		IAssociationDescriptor selectionDescriptor = getAssociationDescriptor(selectionTable);
		if(selectionDescriptor == null)
			return;		
		
		Set descriptors = new HashSet();
		IAssociationMapping mapping = (IAssociationMapping)mappingByTable.get(detailTable);
		String[] types = selectionDescriptor.types();
		for(int i = 0, maxi = types.length; i < maxi; i++)
			descriptors.addAll(Arrays.asList(mapping.getAvoidedAssociationDescriptors(types[i])));
			
		IAssociationDescriptor descriptor = (IAssociationDescriptor)ChooseElementDialog.open(getShell(), UIMessages._17, descriptors.toArray(), new AssociationDescriptorLabelProvider());
		if(descriptor != null)
		{
			for(int i = 0, maxi = types.length; i < maxi; i++)
				mapping.removeFromAvoidedSet(types[i], descriptor);			
		}
		
		populateDetail(selectionDescriptor, detailTable);
	}

	/**
	 * Removes a descriptor to the mapping associated with the detail table. 
	 * @param detailTable
	 */
	protected void removeAssociation(Table detailTable)
	{
		IAssociationDescriptor descriptor = getAssociationDescriptor(detailTable);
		if(descriptor == null)
			return;
		IAssociationDescriptor selectionDescriptor = getAssociationDescriptor(selectionTable);
		if(selectionDescriptor == null)
			return;		
		
		int selectionIndex = detailTable.getSelectionIndex();
		IAssociationMapping mapping = (IAssociationMapping)mappingByTable.get(detailTable);
		String[] types = selectionDescriptor.types();
		for(int i = 0, maxi = types.length; i < maxi; i++)
			mapping.addToAvoidedSet(types[i], descriptor);
			
		populateDetail(selectionDescriptor, detailTable);
		if(selectionIndex == detailTable.getItemCount())
			selectionIndex--;
		detailTable.select(selectionIndex);
	}

	/**
	 * Sets the selected item in the detail table as the default's association 
	 * descriptor in the mapping. 
	 * @param detailTable
	 */
	protected void setDefaultAssociation(Table detailTable)
	{
		IAssociationDescriptor descriptor = getAssociationDescriptor(detailTable);
		if(descriptor == null)
			return;
		
		IAssociationMapping mapping = (IAssociationMapping)mappingByTable.get(detailTable);
		IAssociationDescriptor selectionDescriptor = getAssociationDescriptor(selectionTable);
		IAssociationDescriptor[] descriptors = null;

		if((selectionDescriptor == null) || (selectionDescriptor.types().length == 0))
		{
			mapping.setDefaultAssociationDescriptor(descriptor);
			descriptors = mapping.getAssociationDescriptors();
		}
		else
		{
			String[] types = selectionDescriptor.types();
			Set descriptorSet = new HashSet();
			for(int i = 0, maxi = types.length; i < maxi; i++)
			{
				mapping.setDefaultAssociationDescriptor(types[i], descriptor);
				descriptorSet.addAll(Arrays.asList(mapping.getAssociationDescriptors(types[i])));
			}	
			descriptors = (IAssociationDescriptor[])descriptorSet.toArray(new IAssociationDescriptor[descriptorSet.size()]);
		}

		populate(detailTable, descriptors, descriptor);
		detailTable.select(0);
	}
		
	/**
	 * Updates this preference's page buttons state.
	 */
	protected void updateEnabledState()
	{
		IAssociationDescriptor selectionDescriptor = getAssociationDescriptor(selectionTable);
		IAssociationDescriptor detailDescriptor = null;
		Table detailTable = null;
		if((detailTabFolder != null) && (detailTabFolder.getSelection() != null))
		{
			detailTable = (Table)detailTabFolder.getSelection()[0].getControl();
			detailDescriptor = getAssociationDescriptor(detailTable);
		}
			
			
		if(addAssociationButton != null)
			addAssociationButton.setEnabled((selectionMapping == null) || (selectionDescriptor != null));
			
		if(removeAssociationButton != null)
			removeAssociationButton.setEnabled((detailDescriptor != null) && (detailTable.getItemCount() > 1));
			
		if(defaultAssociationButton != null)
			defaultAssociationButton.setEnabled((detailDescriptor != null) && (detailTable.getSelectionIndex() != 0));
	}
}
