/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: WorkbenchAdapterContentProvider.java,v 1.4 2005/08/16 15:30:39 popescu Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.provider;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IResourceChangeEvent;
import org.eclipse.core.resources.IResourceChangeListener;
import org.eclipse.core.resources.IResourceDelta;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IAdaptable;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.model.IWorkbenchAdapter;

import org.eclipse.hyades.ui.adapter.IHyadesWorkbenchAdapter;
import org.eclipse.hyades.ui.internal.util.UIMessages;

/**
 * Tree content provider that used the Eclipse's adapter framework.  All the methods 
 * are delegated to an adapter that is an instance of 
 * {@link org.eclipse.ui.model.IWorkbenchAdapter}.
 * 
 * <p>One of the functions available in this content provider is to monitor changes in 
 * workspace triggering refreshes and updates in a 
 * {@link org.eclipse.hyades.ui.internal.provider.IResourceChangeUpdater}.  If no updater
 * is available then the changes are not processed.
 * 
 * <p>This implementation is based on the 
 * {@link org.eclipse.ui.model.WorkbenchContentProvider} class.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class WorkbenchAdapterContentProvider 
implements ITreeContentProvider, IResourceChangeListener
{
	private Class adapterClass;
	private Viewer viewer;
	private IResourceChangeUpdater resourceChangeUpdater;
	private IWorkspace currentWorkspace;
	
	/**
	 * Constructor for WorkbenchAdapterContentProvider
	 * @param adapter. A subclass of <code>IWorkbenchAdapter</code>.
	 * @throws IllegalArgumentException if the <code>adapterClass</code> is
	 * <code>null</code> or if it is not an instance of IWorkbenchAdapter. 
	 */
	public WorkbenchAdapterContentProvider(Class adapterClass)
	throws IllegalArgumentException
	{
		if((adapterClass == null) || !IWorkbenchAdapter.class.isAssignableFrom(adapterClass))
			throw new IllegalArgumentException(UIMessages._10);
			
		this.adapterClass = adapterClass;
		inputChanged(null, null, ResourcesPlugin.getWorkspace());
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#dispose()
	 */
	public void dispose()
	{
		if(currentWorkspace != null)
		{
			currentWorkspace.removeResourceChangeListener(this);
			currentWorkspace = null;
		}
		adapterClass = null;
		resourceChangeUpdater = null;
		viewer = null;
	}
	
	/**
	 * Sets the resource changer updater that implements the workspace changes listened by 
	 * this provider.  If the updater is <code>null</code> then the changes are not 
	 * processed.
	 * @param resourceChangeUpdater
	 */
	public void setResourceChangeUpdater(IResourceChangeUpdater resourceChangeUpdater)
	{
		this.resourceChangeUpdater = resourceChangeUpdater;
	}
	
	/**
	 * Returns the resource changer updater that implements the workspace changes listened by 
	 * this provider.
	 * @return IResourceConverter
	 */
	public IResourceChangeUpdater getResourceChangeUpdater()
	{
		return resourceChangeUpdater;
	}
	
	/**
	 * Returns the IWorkbenchAdapter for the given object.  Returns <code>null</code> if 
	 * the adapter is not defined or the object is not adaptable.
	 */
	protected IWorkbenchAdapter getAdapter(Object o)
	{
		if (!(o instanceof IAdaptable))
			return null;

		return (IWorkbenchAdapter)((IAdaptable)o).getAdapter(adapterClass);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getParent(java.lang.Object)
	 */
	public Object getParent(Object element)
	{
		IWorkbenchAdapter adapter = getAdapter(element);
		if(adapter == null)
			return null;
			
		return adapter.getParent(element);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#hasChildren(java.lang.Object)
	 */
	public boolean hasChildren(Object element)
	{
		IWorkbenchAdapter adapter = getAdapter(element);
		if(adapter == null)
			return false;
		
		if(adapter instanceof IHyadesWorkbenchAdapter)
			return ((IHyadesWorkbenchAdapter)adapter).hasChildren(element);
				
		return (adapter.getChildren(element).length > 0);
	}

	/**
	 * @see org.eclipse.jface.viewers.ITreeContentProvider#getChildren(java.lang.Object)
	 */
	public Object[] getChildren(Object parentElement)
	{
		IWorkbenchAdapter adapter = getAdapter(parentElement);
		if(adapter == null)
			return new Object[0];

		return adapter.getChildren(parentElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
	 */
	public Object[] getElements(Object inputElement)
	{
		return getChildren(inputElement);
	}

	/**
	 * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
	 */
	public void inputChanged(Viewer viewer, Object oldInput, Object newInput)
	{
		this.viewer = viewer;
		
		IWorkspace newWorkspace = null;
		if(newInput instanceof IWorkspace)
			newWorkspace = (IWorkspace)newInput;
		else if(newInput instanceof IContainer)
			newWorkspace = ((IContainer)newInput).getWorkspace();

		if((newWorkspace != null) && (currentWorkspace != newWorkspace))
		{
			if(currentWorkspace != null)
				currentWorkspace.removeResourceChangeListener(this);
			
			currentWorkspace = newWorkspace;
			currentWorkspace.addResourceChangeListener(this, IResourceChangeEvent.POST_CHANGE);
		}
	}

	/**
	 * @see org.eclipse.core.resources.IResourceChangeListener#resourceChanged(org.eclipse.core.resources.IResourceChangeEvent)
	 */
	public void resourceChanged(IResourceChangeEvent event)
	{
		if((getResourceChangeUpdater() == null) || (!getResourceChangeUpdater().isActive()))
			return;
			
		final IResourceDelta delta = event.getDelta();
		Control ctrl = viewer.getControl();
		if (ctrl != null && !ctrl.isDisposed())
		{
			getResourceChangeUpdater().started();
			
			try
			{
				// Do a sync exec, not an async exec, since the resource delta
				// must be traversed in this method.  It is destroyed
				// when this method returns.
				ctrl.getDisplay().syncExec(new Runnable()
				{
					public void run()
					{
						processDelta(delta);
					}
				});
			}
			finally
			{
				getResourceChangeUpdater().ended();			
			}
		}
	}

	/**
	 * Process the resource delta for a resource change.
	 * @param delta
	 */
	protected void processDelta(IResourceDelta delta)
	{
		IResourceChangeUpdater resourceChangeUpdater = getResourceChangeUpdater();
		if((resourceChangeUpdater == null) || (!getResourceChangeUpdater().isActive()))
			return;
			
		// This method runs inside a syncExec.  The widget may have been destroyed
		// by the time this is run.  Check for this and do nothing if so.
		Control ctrl = viewer.getControl();
		if (ctrl == null || ctrl.isDisposed())
			return;

		// Get the affected resource
		IResource resource = delta.getResource();

		// If any children have changed type, just do a full refresh of this parent,
		// since a simple update on such children won't work, 
		// and trying to map the change to a remove and add is too dicey.
		// The case is: folder A renamed to existing file B, answering yes to overwrite B.
		IResourceDelta[] affectedChildren =	delta.getAffectedChildren(IResourceDelta.CHANGED);
		for (int i = 0; i < affectedChildren.length; i++)
		{
			int flags = affectedChildren[i].getFlags();
			if((flags & IResourceDelta.TYPE) != 0)
			{
				if(resourceChangeUpdater.updateChildrenType(resource))
					return;
			}
			else if((flags & IResourceDelta.CONTENT) != 0)
			{
				if(resourceChangeUpdater.updateContent(resource, affectedChildren[i].getResource()))
					return;
			}
		}

		// Check the flags for changes the Navigator cares about.
		// See ResourceLabelProvider for the aspects it cares about.
		// Notice we don't care about F_CONTENT or F_MARKERS currently.
		int changeFlags = delta.getFlags();
		if ((changeFlags & (IResourceDelta.OPEN | IResourceDelta.SYNC))	!= 0)
		{
			if(resourceChangeUpdater.updateProperties(resource))
				return;
		}

		// Replacing a resource may affect its label and its children
		if ((changeFlags & IResourceDelta.REPLACED) != 0)
		{
			if(resourceChangeUpdater.replaced(resource))
				return;
		}

		// Handle changed children .
		for (int i = 0; i < affectedChildren.length; i++)
		{
			processDelta(affectedChildren[i]);
		}

		// Process removals before additions, to avoid multiple equal elements in the viewer.

		// Handle removed children. Issue one update for all removals.
		affectedChildren = delta.getAffectedChildren(IResourceDelta.REMOVED);
		if (affectedChildren.length > 0)
		{
			IResource[] affected = new IResource[affectedChildren.length];
			for (int i = 0; i < affectedChildren.length; i++)
				affected[i] = affectedChildren[i].getResource();
			
			if(resourceChangeUpdater.remove(resource, affected))
				return;				
		}

		// Handle added children. Issue one update for all insertions.
		affectedChildren = delta.getAffectedChildren(IResourceDelta.ADDED);
		if (affectedChildren.length > 0)
		{
			IResource[] affected = new IResource[affectedChildren.length];
			for (int i = 0; i < affectedChildren.length; i++)
				affected[i] = affectedChildren[i].getResource();
			if(resourceChangeUpdater.add(resource, affected))
				return;
		}
	}
}
