/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ChooseElementComposite.java,v 1.5 2006/10/30 18:08:50 amehregani Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.util.EventObject;

import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.hyades.ui.util.ILabelAndDescriptionProvider;
import org.eclipse.jface.viewers.ArrayContentProvider;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.IContentProvider;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredViewer;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.Text;

/**
 * A composite that allows a user to choose from one of the elements of a given array.
 * 
 * <p>If no specific content provider is defined then the 
 * {@link org.eclipse.jface.viewers.ArrayContentProvider} is used.
 * 
 * <p>If the specified label provider is an instance of 
 * {@link ILabelAndDescriptionProvider} then this dialog displays a description text 
 * area that can be used to present some information that may help the user to choose 
 * the appropriate element.
 * 
 * <p>Clients are expected to invoke the {@link #createContents(ILabelProvider)} 
 * method before displaying instances of this class.
 * 
 * <p>Important:  There is no guarantees that instances of this class return the 
 * appropriate values for the get methods after being disposed.
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class ChooseElementComposite 
extends Composite implements IDisposable, ISelectionProvider, ISelectionChangedListener
{
	private StructuredViewer structuredViewer;
	private Label structuredViewerLabel;
	private Text descriptionText;
	private Label descriptionLabel;

	/**
	 * Constructor for ChooseElementComposite
	 * @param parent
	 * @param style
	 */
	public ChooseElementComposite(Composite parent, int style)
	{
		super(parent, style);
		
		structuredViewer = createStructuredViewer(this);
		setContentProvider(new ArrayContentProvider());
	}
	
	/**
	 * @see org.eclipse.swt.widgets.Widget#dispose()
	 */
	public void dispose()
	{
		structuredViewer.removeSelectionChangedListener(this);
		
		super.dispose();
		
		if(!structuredViewer.getControl().isDisposed())
			structuredViewer.getControl().dispose();
		
		if(!structuredViewerLabel.isDisposed())
			structuredViewerLabel.dispose();

		if(!descriptionLabel.isDisposed())
			descriptionLabel.dispose();
		
		if(!descriptionText.isDisposed())
			descriptionText.dispose();
		descriptionText = null;		
	}

	/**
	 * Creates the structured viewer section.
	 * @param parent
	 * @return StructuredViewer
	 */
	protected StructuredViewer createStructuredViewer(Composite parent)
	{
		return new TableViewer(parent, SWT.V_SCROLL |SWT.H_SCROLL | SWT.SINGLE | SWT.BORDER);
				
	}
	
	/**
	 * Selects one specific element in the structured content provider
	 * @param index
	 * @param fireSelection. If <code>true</code> then the selection listeners are notified.
	 * @return <code>true</code> if the index was selected.
	 */
	public boolean setSelection(int index, boolean fireSelection)
	{
		if(structuredViewer.getControl() instanceof Table)
		{
			Table table = (Table)structuredViewer.getControl();
			table.select(index);
			if(fireSelection)
				UIUtil.fireCurrentSelection(structuredViewer);

			return true;
		}
		
		return false;
	}

	/**
	 * Creates the internal objects of this composite. 
	 * @param labelProvider
	 */
	public void createContents(ILabelProvider labelProvider)
	{
		setLayout(new GridLayout());

		structuredViewer.setLabelProvider(labelProvider);
		adjustStructuredViewer(structuredViewer);

		if(structuredViewer.getLabelProvider() instanceof ILabelAndDescriptionProvider)
			descriptionText = createDescriptionText(this);	
	}
	
	/**
	 * Adjusts the structured view by setting its control properties.
	 * @param structuredViewer
	 */
	protected void adjustStructuredViewer(StructuredViewer structuredViewer)
	{
		TableViewer tableViewer = (TableViewer)structuredViewer;
		tableViewer.addSelectionChangedListener(this);

		TableLayout tableLayout = new TableLayout();
		
		Table table = tableViewer.getTable();
		table.setLayout(tableLayout);
		GridData gridData = GridDataUtil.createFill();
		gridData.heightHint= 150;
		table.setLayoutData(gridData);
		
		new TableColumn(table, SWT.NONE);
		tableLayout.addColumnData(new ColumnWeightData(10));
		
		ViewerSorter sorter = new ViewerSorter()
		{
			public boolean isSorterProperty(Object element, String property)
			{
				return true;
			}
		};
		tableViewer.setSorter(sorter);
	}

	/**
	 * Creates the description text.
	 * @param parent
	 * @return Text
	 */
	protected Text createDescriptionText(Composite parent)
	{
		Text text = new Text(parent, SWT.BORDER | SWT.MULTI | SWT.V_SCROLL | SWT.WRAP);
		text.setLayoutData(GridDataUtil.createFill());
		text.setEditable(false);
				
		return text;
	}
	
	/**
	 * Adds a listener for double-clicks in the structured viewer.
	 * Has no effect if an identical listener is already registered.
	 * @param listener a double-click listener
	 */
	public void addDoubleClickListener(IDoubleClickListener listener)
	{
		structuredViewer.addDoubleClickListener(listener);
	}
	
	/**
	 * Removes the given double-click listener from the the structured viewer.
	 * Has no affect if an identical listener is not registered.
	 * @param listener a double-click listener
	 */
	public void removeDoubleClickListener(IDoubleClickListener listener)
	{
		structuredViewer.removeDoubleClickListener(listener);
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#addSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void addSelectionChangedListener(ISelectionChangedListener listener)
	{
		structuredViewer.addSelectionChangedListener(listener);
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionProvider#removeSelectionChangedListener(org.eclipse.jface.viewers.ISelectionChangedListener)
	 */
	public void removeSelectionChangedListener(ISelectionChangedListener listener)
	{
		structuredViewer.removeSelectionChangedListener(listener);
	}

	/**
	 * Sets the label to be used for the description text area.
	 * @param string
	 */
	public void setDescriptionLabel(String string)
	{
		if(string == null)
		{
			if(descriptionLabel == null)
				return;
			string = "";
		}
			
		if(descriptionLabel == null)
		{
			descriptionLabel = new Label(this, SWT.NONE);
			if(descriptionText != null)
			{
				descriptionLabel.moveAbove(descriptionText);
				layout();
			}
		}
		descriptionLabel.setText(string);
		if(descriptionText != null) descriptionLabel.pack();
	}

	/**
	 * Returns the label used for the description text area.
	 * @return String
	 */
	public String getDescriptionLabel()
	{
		if(descriptionLabel == null)
			return null;
		return descriptionLabel.getText();
	}	

	/**
	 * Sets the label to be used for the selection area.
	 * @param string
	 */
	public void setStructuredViewerLabel(String string)
	{
		if(string == null)
		{
			if(structuredViewerLabel == null)
				return;
			string = "";
		}
			
		if(structuredViewerLabel == null)
		{
			structuredViewerLabel = new Label(this, SWT.NONE);
			structuredViewerLabel.moveAbove(structuredViewer.getControl());
			layout();
		}
		structuredViewerLabel.setText(string);
		structuredViewerLabel.pack();
	}

	/**
	 * Returns the label used for the selection area.
	 * @return String
	 */
	public String getStructuredViewerLabel()
	{
		if(structuredViewerLabel == null)
			return null;
			
		return structuredViewerLabel.getText();
	}

	/**
	 * Sets the content provider of the structured view.
	 * @param contentProvider
	 */
	public void setContentProvider(IContentProvider contentProvider)
	{
		structuredViewer.setContentProvider(contentProvider);
	}
	
	/**
	 * Returns the content provider of the structured view.
	 * @return IContentProvider
	 */
	public IContentProvider getContentProvider()
	{
		return structuredViewer.getContentProvider();
	}

	/**
	 * Returns the label provider of the structured view.
	 * @return ILabelProvider
	 */
	public ILabelProvider getLabelProvider()
	{
		return (ILabelProvider)structuredViewer.getLabelProvider();
	}

	/**
	 * Sets the input of the structured viewer.
	 * @param input
	 */
	public void setInput(Object[] input)
	{
		structuredViewer.setInput(input);
	}
	
	/**
	 * Returns the structured viewer's input.
	 * @return Object[]
	 */
	public Object[] getInput()
	{
		return (Object[])structuredViewer.getInput();
	}

	/**
	 * Sets the selection of the structured viewer.  This method 
	 * is equivalent to <code>setSelection(selection,false,false)</code>.
	 * @param selection the structured viewer's new selection
	 * @see org.eclipse.jface.viewers.ISelectionProvider#setSelection(org.eclipse.jface.viewers.ISelection)
	 */
	public void setSelection(ISelection selection)
	{
		setSelection(selection, false, false);
	}
	
	/**
	 * Sets the selection of the structured viewer. 
	 * @param selection the structured viewer's new selection
	 * @param reveal if <code>true</code> then the selection is displayed.
	 * @param fireSelection.  If <code>true</code> then the selection listeners are
	 * notified. 
	 */
	public void setSelection(ISelection selection, boolean reveal, boolean fireSelection)
	{
		structuredViewer.setSelection(selection, reveal);
		if(fireSelection)
			UIUtil.fireCurrentSelection(structuredViewer);
	}
		
	/**
	 * Returns the selection of the structured viewer.
	 * @return ISelection
	 * @see org.eclipse.jface.viewers.ISelectionProvider#getSelection()
	 */
	public ISelection getSelection()
	{
		return structuredViewer.getSelection();
	}
	
	/**
	 * Returns the control that is presented by the structured view used to 
	 * present the elements.
	 * @return Control
	 */
	public Control getControl()
	{
		return structuredViewer.getControl();
	}

	/**
	 * Returns true if the event object was created by this component or by one
	 * of its items.
	 * @param eventObject
	 * @return boolean
	 */
	public boolean fired(EventObject eventObject)
	{
		if(eventObject == null)
			return false;
		return
			(eventObject.getSource() == structuredViewer)
			||(eventObject.getSource() == getControl())
			||(eventObject.getSource() == this);
	}

	/**
	 * Returns true if the event was created by this component or by one
	 * of its items.
	 * @param eventObject
	 * @return boolean
	 */
	public boolean fired(Event event)
	{
		if(event == null)
			return false;
			
		return
			(event.widget == getControl())
			||(event.widget == this);
	}

	/**
	 * @see org.eclipse.jface.viewers.ISelectionChangedListener#selectionChanged(org.eclipse.jface.viewers.SelectionChangedEvent)
	 */
	public void selectionChanged(SelectionChangedEvent event)
	{
		if(event.getSelectionProvider() == structuredViewer)
		{
			if(descriptionText == null)
				return;
				
			IStructuredSelection structuredSelection = (IStructuredSelection)event.getSelection();
			if(structuredSelection.size() == 0)
			{
				descriptionText.setText("");
			}
			else
			{
				String description = ((ILabelAndDescriptionProvider)getLabelProvider()).getDescription(structuredSelection.getFirstElement());
				if(description == null)
					description = "";
				descriptionText.setText(description);
			}
		}
	}
}
