/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CoreUtil.java,v 1.8 2006/09/05 22:17:59 ewchan Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;

import org.eclipse.hyades.ui.util.IDisposable;
import org.eclipse.ui.IElementFactory;
import org.eclipse.ui.PlatformUI;

/**
 * Contains UI utility methods. 
 * For non-ui utility methods, see {@link org.eclipse.hyades.test.core.util.CoreUtil CoreUtil}
 * 
 * @author marcelop
 * @since 0.0.1
 */
public class CoreUtil extends org.eclipse.tptp.platform.common.internal.util.CoreUtil
{	
	/**
	 * Returns the {@link IElementFactory} registered with the specified id.  This
	 * methods looks for the factory in the plugin registry returning <code>null</code> 
	 * if no factory was found.
	 * @param factoryId
	 * @return IElementFactory
	 */
	public static IElementFactory getElementFactory(String factoryId)
	{
		return PlatformUI.getWorkbench().getElementFactory(factoryId);
	}
	
	
	/**
	 * Disposes all the instances of {@link IDisposable} in the <code>value</code> 
	 * Collection of the given map and clears the map at the end catching any
	 * <code>UnsupportedOperationException</code> that may be throw. 
	 * @param map
	 * @return boolean <code>true</code> if the collection is cleared
	 * or false otherwise.
	 */
	public static boolean dispose(Map map)
	{
		if(map == null)
			return true;
			
		boolean ret = dispose(map.values());
		map.clear();
		return ret;
	}
	/**
	 * Disposes all the instances of {@link IDisposable} in the given 
	 * Collection and clears the collection at the end catching any
	 * <code>UnsupportedOperationException</code> that may be throw.
	 * 
	 * <p>If the collection item is a collection or map then this method
	 * evaluates its contents in order to dispose any {@link IDisposable}
	 * instance.
	 * 
	 * @param disposableCandidates
	 * @return boolean <code>true</code> if the collection is cleared
	 * or false otherwise.
	 */
	public static boolean dispose(Collection disposableCandidates)
	{
		if(disposableCandidates == null)
			return true;
			
		for (Iterator i = disposableCandidates.iterator(); i.hasNext();)
		{
			Object object = i.next();
			if(object instanceof IDisposable)
				((IDisposable)object).dispose();
			else if(object instanceof Collection)
				dispose((Collection)object);
			else if(object instanceof Map)
				dispose((Map)object);
		}
		
		try
		{
			disposableCandidates.clear();
		}
		catch(UnsupportedOperationException e)
		{
			return false;
		}
		
		return true;
	}
	
	/**
	 * Returns the Object value of a given object's field - or, using OO words, 
	 * attribute.  If the field is not available in the object's class, this
	 * method will look for it in the superclass hierarchy.
	 * 
	 * <p>If the value of <code>throwException</code> is true then any exception 
	 * during the introspection process is thrown as a RuntimeExceptionDecorator. 
	 * If it is false and there is an exception the method returns null.
	 * 
	 * @param object
	 * @param fileName
	 * @param throwException
	 * @return Object
	 * @throws RuntimeExceptionDecorator if there is an error and <code>throwException</code>
	 * is true.
	 * 
	 * @see Class#getDeclaredField(java.lang.String)
	 * @see Field#get(java.lang.Object)
	 */
	public static Object getObjectFieldValue(Object object, String fieldName, boolean throwException)
	throws RuntimeException
	{
		if((object == null) || (fieldName == null))
			return null;
		
		RuntimeExceptionDecorator red = null;	
		Field field = null;
		Class currentClass = object.getClass();
		while((field == null) && (currentClass != null))
		{
			try
			{
				field = currentClass.getDeclaredField(fieldName);
			}
			catch(Exception e)
			{
				if(throwException && (red == null))
					red = new RuntimeExceptionDecorator(e);

				currentClass = currentClass.getSuperclass();
			}			
		}
		if(field == null)
		{
			if(red != null)
				throw red;
				
			return null;
		}

		try
		{
			field.setAccessible(true);
			return field.get(object);
		}
		catch(Exception e)
		{
			if(throwException)
				throw new RuntimeExceptionDecorator(e);
		}
		
		return null;
	}
	
	/**
	 * See {@link #invokeObjectMethod(Object, String, Class[], Object[], boolean)}.
	 * This method is a simple utility that creates the parameter type array based
	 * on the classes of the arguments. 
	 * @param object
	 * @param methodName
	 * @param arguments
	 * @param throwException
	 * @return Object
	 * @throws RuntimeException
	 */
	public static Object invokeObjectMethod(Object object, String methodName, Object[] arguments, boolean throwException)
	throws RuntimeException
	{
		Class[] parameterTypes = null;
		if(arguments != null)
		{
			parameterTypes = new Class[arguments.length];
			for(int i = 0, maxi = arguments.length; i < maxi; i++)
				parameterTypes[i] = arguments[i].getClass();
		}
		
		return invokeObjectMethod(object, methodName, parameterTypes, arguments, throwException);
	}

	/**
	 * Invokes an specific method of a given object.  If the method is not available 
	 * in the object's class, this
	 * method will look for it in the superclass hierarchy.
	 * 
	 * <p>If the Object's method has no arguments, callers can set 
	 * <code>parameterType</code> and <code>argument</code> as null. 
	 * 
	 * <p>If the value of <code>throwException</code> is true then any exception 
	 * during the introspection process is thrown as a RuntimeExceptionDecorator. 
	 * If it is false and there is an exception the method returns null.
	 * 
	 * @param object
	 * @param methodName
	 * @param parameterTypes
	 * @param arguments
	 * @param throwException
	 * @return An Object returned by the invoked method.
	 * @throws RuntimeException if there is an error and <code>throwException</code>
	 * is true.
	 * 
	 * @see Class#getDeclaredMethod(java.lang.String, java.lang.Class[]);
	 * @see Method#invoke(java.lang.Object, java.lang.Object[])
	 */
	public static Object invokeObjectMethod(Object object, String methodName, Class[] parameterTypes, Object[] arguments, boolean throwException)
	throws RuntimeException
	{
		if((object == null) || (methodName == null))
			return null;
			
		if(parameterTypes == null)
			parameterTypes = new Class[0];
			
		if(arguments == null)
			arguments = new Object[0];
		
		RuntimeExceptionDecorator red = null;	
		Method method = null;
		Class currentClass = object.getClass();
		while((method == null) && (currentClass != null))
		{
			try
			{
				method = currentClass.getDeclaredMethod(methodName, parameterTypes);
			}
			catch(Exception e)
			{
				if(throwException && (red == null))
					red = new RuntimeExceptionDecorator(e);

				currentClass = currentClass.getSuperclass();
			}			
		}
		if(method == null)
		{
			if(red != null)
				throw red;
				
			return null;
		}

		try
		{
			method.setAccessible(true);
			return method.invoke(object, arguments);
		}
		catch(Exception e)
		{
			if(throwException)
				throw new RuntimeExceptionDecorator(e);
		}
		
		return null;
	}
}