/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaUtil.java,v 1.7 2010/05/11 17:09:56 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.util;

import java.io.File;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.jdt.core.IClasspathEntry;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.ui.JavaUI;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.text.BadLocationException;
import org.eclipse.jface.text.IDocument;
import org.eclipse.ui.IEditorPart;
import org.eclipse.ui.actions.WorkspaceModifyOperation;
import org.eclipse.ui.texteditor.ITextEditor;

/**
 * <p>Internal UI utility APIs wrapping Eclipse JDT APIs.</p>
 * 
 * 
 * @author  Paul Slauenwhite
 * @author  Marcelo Paternostro
 * @version May 11, 2010
 * @since   May 4, 2009
 */
public class JavaUtil {
	

	/**
	 * Sets the value of a classpath variable.
	 * @param variable
	 * @param value
	 * @param progressMonitor
	 * @throws JavaModelException
	 * @throws InterruptedException
	 * @throws InvocationTargetException
	 */
	public static void setVariableValue(final String variable, final String value, IProgressMonitor progressMonitor)
	throws JavaModelException, InterruptedException, InvocationTargetException
	{
		if((variable == null) || (value == null))
			return;
			
		String currentValue = getVariableValue(variable);
		if(currentValue != null)
		{
			if(value.equals(currentValue))
				return;
				
			if(new File(value).equals(new File(currentValue)))
				return;
		}
			
		IRunnableWithProgress operation = new WorkspaceModifyOperation()
		{
			public void execute(IProgressMonitor monitor)
			throws CoreException, InvocationTargetException, InterruptedException
			{
				JavaCore.setClasspathVariable(variable, new Path(value), monitor);
			}
		};
		
		operation.run(progressMonitor);
	}
	
	/**
	 * Returns the value of the classpath variable specified in the
	 * argument.
	 * @param variable
	 * @return String
	 */
	public static String getVariableValue(String variable)
	{
		IPath path = JavaCore.getClasspathVariable(variable);		
		if(path != null)
		{
			String value = path.makeAbsolute().toString();
			if(!value.equals(""))
				return value;
		}
			
		return null;
	}
	

	
	/**
	 * Returns a Project's classpath.  If the project has a dependency on another 
	 * project the exported entries of the last one will be added to the classpath
	 * of the former. 
	 * 
	 * @param project
	 * @param exportedOnly
	 * @return String[]
	 */
	public static String[] getProjectClasspath(IProject project, boolean exportedOnly)
	{
		if(project == null)
			return new String[0];
			
		IJavaProject javaProject = JavaCore.create(project);
		if(javaProject == null)
			return new String[0];

		List cpEntries = null;
		try
		{
			cpEntries = new ArrayList(Arrays.asList(javaProject.getRawClasspath()));
		}
		catch(JavaModelException e)
		{
		}
		if((cpEntries == null) || cpEntries.isEmpty())
			return new String[0];
		
		Set classpath = new HashSet(cpEntries.size());
		try
		{	
			IPath path = javaProject.getOutputLocation();
			if(path != null)
				classpath.add(ResourceUtil.getFullPath(path));
		}
		catch(JavaModelException e)
		{
		}
				
		for (Iterator i = cpEntries.iterator(); i.hasNext();)
		{
			IClasspathEntry entry = JavaCore.getResolvedClasspathEntry((IClasspathEntry)i.next());
			if(exportedOnly && (!entry.isExported()))
				continue;
			
			String strcp = null;
			switch(entry.getEntryKind())
			{
				case IClasspathEntry.CPE_LIBRARY:
				case IClasspathEntry.CPE_VARIABLE:
					strcp = ResourceUtil.getFullPath(entry.getPath());
					if(strcp != null)
						classpath.add(strcp);
					break;

				case IClasspathEntry.CPE_PROJECT:
					IPath projectPath = entry.getPath().makeAbsolute();
					IProject parentProject = ResourcesPlugin.getWorkspace().getRoot().getProject(projectPath.toString());				
					classpath.addAll(Arrays.asList(getProjectClasspath(parentProject, true)));
					break;
				
				case IClasspathEntry.CPE_CONTAINER:
					break;
					
				case IClasspathEntry.CPE_SOURCE:
					strcp = ResourceUtil.getFullPath(entry.getOutputLocation());
					if(strcp != null)
						classpath.add(strcp);
			}
		}
				
		return (String[])classpath.toArray(new String[classpath.size()]);
	}
	
	/**
	 * Opens an editor containing the specified java element, selects and reveals the specified
	 * line in the editor's content.
	 * @param javaElement A Java Element contained in a file (e.g. IType, IMethod, IClass)
	 * @param line A line number
	 * @throws CoreException
	 */
	public static void revealJavaElementAtLine(IJavaElement javaElement, int line) throws CoreException {
		if (javaElement != null) {
			IEditorPart editorPart = JavaUI.openInEditor(javaElement);
			if (editorPart != null) {
				if (editorPart instanceof ITextEditor) {
					ITextEditor textEditor = (ITextEditor) editorPart;
					try {
						IDocument document= textEditor.getDocumentProvider().getDocument(textEditor.getEditorInput());
						textEditor.selectAndReveal(document.getLineOffset(line-1), document.getLineLength(line-1));
					} catch (BadLocationException x) {
						// Line not found -> do nothing
					}
				}
			}
		}
	}

}
