/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: 
 * 
 * Contributors: 
 * IBM - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import java.io.File;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.hyades.ui.filters.internal.util.FilterQueries;
import org.eclipse.hyades.ui.internal.filters.FilterTypesManager;
import org.eclipse.hyades.ui.internal.filters.IFilterType;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.hyades.ui.util.GridUtil;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnLayoutData;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.DoubleClickEvent;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IDoubleClickListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.IWorkbench;
import org.eclipse.ui.dialogs.WizardDataTransferPage;

public abstract class ExportImportFilterWizardPage extends
		WizardDataTransferPage implements Listener, ModifyListener, SelectionListener, IRunnableWithProgress {

    // widgets
    protected Button sourceBrowseButton;
	protected Text sourceNameField;
	private DoubleList doubleList;

    //~ Constructors -------------------------------------------------------------------------------

    /**
     *  Creates an instance of this class
     * @param aWorkbench IWorkbench
     * @param selection IStructuredSelection
     */
    public ExportImportFilterWizardPage(String name, IWorkbench aWorkbench, IStructuredSelection selection) {
        super(name); //$NON-NLS-1$
    }
    
    protected DoubleList getDoubleList()
    {
    	return doubleList;
    }

    //~ Methods ------------------------------------------------------------------------------------

	protected void initializeValues() {
		List source = new ArrayList();
		List target = new ArrayList();
		source = populateFilterList();
		doubleList.removeAll();
		
		doubleList.initializeSourceList((ArrayList)source);
		doubleList.initializeTargetList((ArrayList)target);
		
		doubleList.refreshViews();		
	}
	
    /**
     *  Handle all events and enablements for widgets in this dialog
     *
     * @param event Event
     */
    public void handleEvent(Event event) {
        if (event.widget == sourceBrowseButton) {
            handleSourceBrowseButtonPressed();
        }
    }

    /**
     * Sent when the text is modified.
     *
     * @param e an event containing information about the modify
     */
    public void modifyText(ModifyEvent e) {
        setPageComplete(determinePageCompletion());
    }

    public void run(IProgressMonitor mon) {
    }

    public void widgetDefaultSelected(SelectionEvent e) {
    }

    public void widgetSelected(SelectionEvent e) {
    	setPageComplete(determinePageCompletion());
    }

    protected boolean allowNewContainerName() {
        return true;
    }

    protected void createSourceGroup(Composite parent) {
        Composite sourceContainerGroup = new Composite(parent, SWT.NONE);
        GridLayout layout = new GridLayout();

        layout.numColumns = 3;
        sourceContainerGroup.setLayout(layout);
        sourceContainerGroup.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL));

        new Label(sourceContainerGroup, SWT.NONE).setText(UIMessages._125); 

        // source name entry field
        sourceNameField = new Text(sourceContainerGroup, SWT.BORDER);

        GridData data = new GridData(GridData.HORIZONTAL_ALIGN_FILL | GridData.GRAB_HORIZONTAL);

        data.widthHint = SIZING_TEXT_FIELD_WIDTH;
        sourceNameField.setLayoutData(data);

        // source browse button
        sourceBrowseButton = new Button(sourceContainerGroup, SWT.PUSH);
        sourceBrowseButton.setText(UIMessages._133); 
        sourceBrowseButton.addListener(SWT.Selection, this);
        sourceBrowseButton.setLayoutData(new GridData(GridData.HORIZONTAL_ALIGN_FILL));

        sourceNameField.setFocus();
        
        sourceNameField.addModifyListener(this);        
    }
    
    protected void createDoubleList(Composite composite, String leftListHeading, String rightListHeading)
    {
        doubleList = new DoubleList(composite, SWT.NULL, leftListHeading, rightListHeading);
		doubleList.createControl();

		initializeValues();
		doubleList.addSelectionListener(this);
    }

    /**
     * Returns whether this page is complete. This determination is made based upon
     * the current contents of this page's controls.  Subclasses wishing to include
     * their controls in this determination should override the hook methods
     * <code>validateSourceGroup</code> and/or <code>validateOptionsGroup</code>.
     *
     * @return <code>true</code> if this page is complete, and <code>false</code> if
     *   incomplete
     * @see #validateSourceGroup
     * @see #validateOptionsGroup
     */
    protected boolean determinePageCompletion() {
        boolean complete = validateSourceGroup() && validateOptionsGroup();
        
        boolean validLists = validateLists();

        // Avoid draw flicker by not clearing the error
        // message unless all is valid.
        if (validLists) {
            setErrorMessage(null);
        }

        return complete && validLists;
    }

    /**
     *  Open an appropriate source browser so that the user can specify a source
     *  to import from
     */
    protected void handleSourceBrowseButtonPressed() {
        String currentSource = this.sourceNameField.getText();

        FileDialog dlg = new FileDialog(sourceNameField.getShell());

        dlg.setFilterPath(currentSource);
        dlg.setFilterExtensions(new String[] { "*.filtxml", "*.*" }); //$NON-NLS-1$ //$NON-NLS-2$
        dlg.open();

        String fileName = dlg.getFileName();

        if ((fileName != null) && !fileName.equals("")) { //$NON-NLS-1$
            sourceNameField.setText(dlg.getFilterPath() + File.separator + fileName);
        }
    }

    /**
     * Check if widgets are enabled or disabled by a change in the dialog.
     * @param event Event
     */
    protected void updateWidgetEnablements() {
        boolean pageComplete = determinePageCompletion();

        setPageComplete(pageComplete);

        if (pageComplete) {
            setMessage(null);
        }

        super.updateWidgetEnablements();
    }

     /**
     * Returns whether this page's source specification controls currently all
     * contain valid values.
     * <p>
     * The <code>WizardDataTransferPage</code> implementation of this method returns
     * <code>true</code>. Subclasses may reimplement this hook method.
     * </p>
     *
     * @return <code>true</code> indicating validity of all controls in the
     *   source specification group
     */
    protected boolean validateSourceGroup() {
    	if (sourceNameField == null)
    		return false;
    	
        return (!sourceNameField.getText().trim().equals("") && doubleList.getTargetItems().size()>0); //$NON-NLS-1$
    }

	class DoubleList
	{
	  Group group;
	  Composite groupWidget;  
	
	  private String NAME_TEXT = "name";
	  private String[] tableColumnProperties =
	  {
	  	NAME_TEXT
	  };
		
	  private TableViewer sourceList;
	  private TableViewer targetList;
	  private List sourceContent = new ArrayList();
	  private List targetContent = new ArrayList();  
	
	  private  Button    add;
	  private  Button    remove;
	  private  Button    addAll;
	  private  Button    removeAll;
	  private  List listeners = new ArrayList(2);
	
	  private TableLabelProvider labelProvider;
	  private Composite parent;
	  private int style;
	  private String sourceText;
	  private String targetText;
	
	  
	  protected void sourceListDoubleClick()
	  {
		  if (sourceList.getSelection().isEmpty())
			  return;
		  
		  IStructuredSelection sel = (IStructuredSelection)sourceList.getSelection();
		  for (Iterator i = sel.iterator(); i.hasNext();)
		  {
			  Object item = i.next();
			  targetContent.add(item);
			  sourceContent.remove(item);
		  }
		  sourceList.setSelection(StructuredSelection.EMPTY);		 
		  refreshViews();
	  }
	  
	  protected void targetListDoubleClick()
	  {
		  if (targetList.getSelection().isEmpty())
			  return;
		  
		  IStructuredSelection sel = (IStructuredSelection)targetList.getSelection();
		  for (Iterator i = sel.iterator(); i.hasNext();)
		  {
			  Object item = i.next();
			  sourceContent.add(item);
			  targetContent.remove(item);
		  }
		  targetList.setSelection(StructuredSelection.EMPTY);		 
		  refreshViews();
	  }	  
	  
	  class WidgetListener implements SelectionListener
	  {
		public void widgetDefaultSelected(SelectionEvent e){	
		}
	
		public void widgetSelected(SelectionEvent e) 
		{
			boolean refreshViews = false;
	
			if(e.widget == add)
			{
			 IStructuredSelection sel = (IStructuredSelection)sourceList.getSelection();
			  for (Iterator i = sel.iterator(); i.hasNext();)
			  {
				Object item = i.next();
				targetContent.add(item);
				sourceContent.remove(item);
			  }
			  sourceList.setSelection(StructuredSelection.EMPTY);		 
			  refreshViews = true;
			}
			else if(e.widget == addAll)
			{
			  for (int i=0; i < sourceContent.size(); i++)
			  {
				 Object item = sourceContent.get(i);
				 targetContent.add(item);
			  }
			  sourceContent.clear();         
			  refreshViews = true;
			}
			else if(e.widget == remove)
			{
			  IStructuredSelection sel = (IStructuredSelection)targetList.getSelection();
			  
			  for (Iterator i = sel.iterator(); i.hasNext();)
			  {
				Object item = i.next();
				sourceContent.add(item);
				targetContent.remove(item);
			  }
			  targetList.setSelection(StructuredSelection.EMPTY);
			  refreshViews = true;
			}
			else if(e.widget == removeAll)
			{
			  for (int i=0; i < targetContent.size(); i++)
			  {
				 Object item = targetContent.get(i);
				 sourceContent.add(item);
			  }
			  targetContent.clear();
			  refreshViews = true;
			}else if(e.widget instanceof Table){
				TableItem[] sel = ((Table)e.widget).getSelection();
				if(sel!=null && sel.length>0){
					((Table)e.widget).setToolTipText(sel[0].getText());	
				}
			}
				
			if (refreshViews)
				refreshViews();

			fireSelectionNotification(e);
		}	
	  }
	
	  class TableContentProvider implements IStructuredContentProvider{
	
		private List tableContents = new ArrayList(0);	
		
		/* (non-Javadoc)
		   * @see org.eclipse.jface.viewers.IStructuredContentProvider#getElements(java.lang.Object)
		   */
		  public Object[] getElements(Object inputElement) {			
			  return tableContents.toArray();
		  }
	
		  /* (non-Javadoc)
		   * @see org.eclipse.jface.viewers.IContentProvider#dispose()
		   */
		  public void dispose() {
			//tableContents.clear();
		  }
	
		  /* (non-Javadoc)
		   * @see org.eclipse.jface.viewers.IContentProvider#inputChanged(org.eclipse.jface.viewers.Viewer, java.lang.Object, java.lang.Object)
		   */
		  public void inputChanged(
			  Viewer viewer,
			  Object oldInput,
			  Object newInput) {
			  	tableContents = (List)newInput;				
		  }
	
	  }
	
	  class TableLabelProvider extends LabelProvider implements ITableLabelProvider{	
		public TableLabelProvider(){
		}
		
		public Image getColumnImage(Object element, int columnIndex) {
			if(columnIndex==0){
				if (element instanceof FilterQueries)
				{
					FilterQueries fq = (FilterQueries)element;
					IFilterType filterType = FilterTypesManager.instance().getFilterType(fq.type());
				
					return filterType.icon().createImage();
				}
				else
					return null;
				
			}
			return null; 
		}
	
		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ITableLabelProvider#getColumnText(java.lang.Object, int)
		 */
		public String getColumnText(Object element, int columnIndex) {
			
			if(columnIndex==0){
				if (element instanceof FilterQueries)
					return ((FilterQueries)element).name();
				else
					return element.toString();
			}
			return ""; //$NON-NLS-1$
		}
				
		/*
		 * @see IBaseLabelProvider#dispose()
		 */
		public void dispose() {			
			super.dispose();
		}
	
	  }
	
	  class TableCellModifier implements ICellModifier
	  {

	  	public boolean canModify(Object element, String property) {
			return true;
		}

		public Object getValue(Object element, String property) {
			if (NAME_TEXT.equals(property))
				return ((FilterQueries)element).name();
			else
				return "";
		}

		/* (non-Javadoc)
		 * @see org.eclipse.jface.viewers.ICellModifier#modify(java.lang.Object, java.lang.String, java.lang.Object)
		 */
		public void modify(Object element, String property, Object value) {
			if (NAME_TEXT.equals(property))
			{
				((FilterQueries)((TableItem)element).getData()).name(value.toString());
				refreshViews();
			}
		}
	  }
	  
	  public DoubleList(Composite parent, int style,
						 String sourceText, String targetText)
	  {
	   		this.parent = parent;
	   		this.style = style;
	   		this.sourceText = sourceText;
	   		this.targetText = targetText;
	  }  
	
	  public Control createControl(){
		group = new Group(parent, style);
		group.setText(""); //$NON-NLS-1$
	
		GridLayout layout= new GridLayout();
		layout.horizontalSpacing= 0;    
		group.setLayout(layout);
		group.setLayoutData(GridUtil.createFill());
	
		createDoubleList(group, sourceText, targetText);
	
		//add listeners
		WidgetListener listener = new WidgetListener(); 
		add.addSelectionListener(listener);
		remove.addSelectionListener(listener);
		addAll.addSelectionListener(listener);
		removeAll.addSelectionListener(listener);
		sourceList.getTable().addSelectionListener(listener);
		targetList.getTable().addSelectionListener(listener);
		
		return group; 	  
	  } 
	
	  public void refreshViews(){
	  	sourceList.refresh();
	  	targetList.refresh();
	  	
	  	updatePageCompletion();
	  }
	
	  /**
	  * Create double list components
	  */
	  public void createDoubleList(Composite parent, 
						 String sourceText, String targetText)    
	  {
		  	  
		groupWidget = parent;
	
		GridLayout layout = new GridLayout();
		layout.numColumns = 3;
		groupWidget.setLayout(layout);
	
		createTableGroup(groupWidget, sourceText, true);		
	
		createButtonGroup(groupWidget);		
			
		createTableGroup(groupWidget, targetText, false);		
		
		labelProvider = new TableLabelProvider();
	
	  }  
	
		// Initialize the list on the left.
		public void initializeSourceList(List source) {			
			sourceContent.addAll(source);
			sourceList.setContentProvider(new TableContentProvider());
			sourceList.setLabelProvider(labelProvider);
			sourceList.setInput(sourceContent);
			if(source.size()>0){
				sourceList.setSelection(new StructuredSelection(source.get(0)));
			}			
		}
	
		// Initialize the list on the right.
		public void initializeTargetList(List target) {
			targetContent.addAll(target);
			targetList.setContentProvider(new TableContentProvider());
			targetList.setLabelProvider(labelProvider);
			targetList.setInput(targetContent);		
		}
	
	  public void removeAll()
	  {
		  targetContent.clear();
		  sourceContent.clear();   
	  } 
	
	  public List getSourceItems(){
	  	return sourceContent;
	  }
	
	  public List getTargetItems(){
	  	return targetContent;
	  }
	
	  public void addSelectionListener(SelectionListener listener){
	
		if(!listeners.contains(listener)){
			listeners.add(listener);
		}  
	  }
	
	  public void removeSelectionListener(SelectionListener listener){
		listeners.remove(listener);
	  }
	
	  public void fireSelectionNotification(SelectionEvent e){
	
		for (int i = 0; i < listeners.size(); i++) {
			((SelectionListener)listeners.get(i)).widgetSelected(e);
		}	   
	  }
	
	 private void createTableGroup(Composite parent, String text, boolean source){
		Composite tableGroup =  new Composite (parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 1;
		tableGroup.setLayout(layout);
		tableGroup.setLayoutData(GridUtil.createFill());
		
		Label label = new Label (tableGroup, SWT.NONE);
		label.setText(text);
		GridData spec = new GridData();		
		spec.horizontalAlignment = GridData.FILL;
		label.setLayoutData(spec);

		CellEditor[] cellEditors = new CellEditor[1];
		if(source){
			sourceList = new TableViewer(createTable(tableGroup)); 
			sourceList.getTable().setLayoutData(GridUtil.createFill());
			sourceList.addDoubleClickListener(new IDoubleClickListener(){
				public void doubleClick(DoubleClickEvent event) {
					sourceListDoubleClick();
				}
			});			
			sourceList.setColumnProperties(tableColumnProperties);
		}else{
			targetList = new TableViewer(createTable(tableGroup)); 
			targetList.getTable().setLayoutData(GridUtil.createFill());
			cellEditors[0] = new TextCellEditor(targetList.getTable());	
			targetList.setCellEditors(cellEditors);
			targetList.setCellModifier(new TableCellModifier());
			targetList.addDoubleClickListener(new IDoubleClickListener(){
				public void doubleClick(DoubleClickEvent event) {
					targetListDoubleClick();
				}
			});
			targetList.setColumnProperties(tableColumnProperties);			
		}
	
	 }
	
	 private Table createTable(Composite parent){
		Table table = new Table(parent, SWT.BORDER|SWT.H_SCROLL|SWT.V_SCROLL|SWT.FULL_SELECTION|SWT.MULTI);
		GridData tblData = GridUtil.createFill();
		table.setLayoutData(tblData);
	 
		new TableColumn(table, SWT.NONE);					
		ColumnLayoutData c0Layout = new ColumnWeightData(1, true);
	
		// set columns in Table layout
		TableLayout tblLayout = new TableLayout();
		tblLayout.addColumnData(c0Layout);
	
		table.setLayout(tblLayout);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		return table;
	
	 }
	
	 private void createButtonGroup(Composite groupWidget){
		Composite buttonGroup = new Composite (groupWidget, SWT.NULL);   
		GridLayout blayout = new GridLayout();
		blayout.numColumns = 1;
		buttonGroup.setLayout(blayout);     
	
		Label label = new Label (buttonGroup, SWT.PUSH);
		label.setText(""); //$NON-NLS-1$
		GridData spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);
	
		add = new Button(buttonGroup, SWT.PUSH);
		add.setText(UIMessages._134); 
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		add.setLayoutData(spec);
	
		remove = new Button(buttonGroup, SWT.PUSH);
		remove.setText(UIMessages._135); 
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		remove.setLayoutData(spec);
		
		new Label(buttonGroup, SWT.NONE);
		
		addAll = new Button(buttonGroup, SWT.PUSH);
		addAll.setText(UIMessages._136); 
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		addAll.setLayoutData(spec);
	
		removeAll = new Button(buttonGroup, SWT.PUSH);
		removeAll.setText(UIMessages._137); 
		spec = new GridData();
		spec.horizontalAlignment = GridData.FILL;
		removeAll.setLayoutData(spec);
	
		Label label1 = new Label (buttonGroup, SWT.PUSH);
		label1.setText(""); //$NON-NLS-1$
		spec = new GridData();
		spec.grabExcessVerticalSpace = true;
		spec.verticalAlignment = GridData.FILL;
		label.setLayoutData(spec);	
	 }
	}

    public abstract boolean finish();	
	protected abstract List populateFilterList();   
    public abstract void createControl(Composite parent);
    protected abstract boolean validateLists();
}
