/*******************************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: HyadesNewWizard.java,v 1.9 2010/05/05 12:32:08 paules Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspaceRoot;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.ResourceUtil;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.tptp.platform.common.ui.wizard.LocationPage;
import org.eclipse.ui.INewWizard;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.actions.NewProjectAction;

/**
 * <p>Abstract extension of the Eclipse's <code>NewWizard</code> that defines a location 
 * page and a "name and description" page.  The subclasses are supposed to add the pages
 * in the appropriate order and using the correct values for their titles and descriptions.</p>
 * 
 * <p>This implementation is based on the 
 * <code>org.eclipse.ui.wizards.newresource.BasicNewResourceWizard</code> class.</p>
 * 
 * 
 * @author  Marcelo Paternostro
 * @author  Paul E. Slauenwhite
 * @version May 4, 2010
 * @since   January 26, 2005
 */
public abstract class HyadesNewWizard extends HyadesWizard implements INewWizard {
    /*
     * Dialog Settings constants
     */
    protected static final String SET_LAST_SELECTION = "LAST_SELECTION"; //$NON-NLS-1$
    private LocationPage locationPage;
    private AttributeWizardPage attributeWizardPage;
    private IFile newFile; // added by bjiang since 1.3

    /**
     * @see org.eclipse.hyades.ui.internal.wizard.HyadesWizard#initPages()
     */
    protected void initPages() {
        
    	adjustSeletion();

    	attributeWizardPage = new AttributeWizardPage("nameDescription"); //$NON-NLS-1$
    }

    /**
     * Adjusts the {@link IStructuredSelection} passed to this wizard.
     * <p/>
     * 
     * @return The adjusted {@link IStructuredSelection}.
     * @deprecated As of TPTP 4.6.1, this method sets the adjusted {@link IStructuredSelection} to the {@link #selection} field and should not return the adjusted {@link IStructuredSelection}. 
     */
    protected IStructuredSelection adjustSeletion() {
        
		//Case 1: Workspace does not contain an open project so open the create project wizard (File >> New >> Project...): 
    	if(UIUtil.getFirstOpenProject() == null){
        	
    		new NewProjectAction().run();
    		
    		IProject newProject = UIUtil.getFirstOpenProject();
    		
    		if(newProject != null){
    			
    			selection = new StructuredSelection(newProject);
    			
    			return selection;
    		}    		
    	}

		//Case 2: Selection contains a resource (project or folder/file) so check if accessible (open for projects and exists for folders/files): 
    	if ((selection != null) && (!selection.isEmpty())) {

    		IResource resource = null;
    		Object firstSelectedElement = selection.getFirstElement();
			        	
    		if(firstSelectedElement instanceof IResource){    			
    			resource = ((IResource)(firstSelectedElement));        		
    		}
    		else if(firstSelectedElement instanceof IJavaProject){
    			resource = ((IJavaProject)(firstSelectedElement)).getProject();
    		}    		

    		if ((resource != null) && (resource.isAccessible())){
    			return selection;
    	    }
    	}

		//Case 3: Selection is null/empty so check if the last selected resource (project or folder/file) for the dialog is accessible (open for projects and exists for folders/files): 
    	if (getDialogSettings() != null) {

    		String lastSelection = getDialogSettings().get(SET_LAST_SELECTION);

    		if (lastSelection != null) {            
    			
    			IResource resource = ResourcesPlugin.getWorkspace().getRoot().findMember(lastSelection);          

    			if ((resource != null) && (resource.isAccessible())){
    	    		
    				selection = new StructuredSelection(resource);
    				
    				return selection;
    	    	}
    		}
    	}

    	//Case 4: Return an empty selection:
    	selection = StructuredSelection.EMPTY;
    	
    	return selection;
    }
    
    /**
     * Returns the location page used by this wizard.
     * @return LocationPage
     */
    protected LocationPage getLocationPage() {

    	if(locationPage == null){

    		locationPage = new LocationPage("location", selection); //$NON-NLS-1$

    		String fileExtension = getFileExtension();

    		if (fileExtension != null) {

    			locationPage.setFileExtension(fileExtension);
    			locationPage.setFileName(""); //$NON-NLS-1$
    		}
    	}

    	return locationPage;
    }

    /**
     * Returns the "name and description page" used by this wizard.
     * @return NameDescriptionWizardPage
     */
    protected AttributeWizardPage getAttributeWizardPage() {
        return attributeWizardPage;
    }

    /**
     * @see org.eclipse.jface.wizard.IWizard#performFinish()
     */
    public boolean performFinish() {
        //TODO Add Progress bar

        IWorkspaceRoot root = ResourcesPlugin.getWorkspace().getRoot();
        IPath path = getLocationPage().getContainerFullPath();
        if (root.findMember(path) == null) {
            try {
                ResourceUtil.createContainer(root.getFolder(path), null);
            } catch (CoreException e) {
                handleException(path.toString(), e);
                return false;
            }
        }

        String fileName = getLocationPage().getFileName();
        String fileExtension = getLocationPage().getFileExtension();
        if (!fileName.endsWith("." + fileExtension)) { //$NON-NLS-1$
            fileName = fileName + "." + fileExtension; //$NON-NLS-1$
        }

        path = path.append(fileName);
        IFile file = root.getFile(path);

        try {
            if (!createObject(file)) return false;
        } catch (Exception e) {
            handleException(file.getFullPath().toString(), e);
            return false;
        }

        this.newFile = file;
        selectAndReveal(file);
        try {
            openEditor(file);
        } catch (PartInitException e) {
            handleException(file.getFullPath().toString(), e);
        }

        //Save the last selected resource (project or folder/file) for the dialog:
        getDialogSettings().put(SET_LAST_SELECTION, getLocationPage().getContainerFullPath().toString());
        
        return true;
    }

    /**
     * Selects and reveals the newly added resource in all parts
     * of the active workbench window's active page.
     * @param newResource
     */
    protected void selectAndReveal(IResource newResource) {
        ResourceUtil.selectAndReveal(newResource, getWorkbench().getActiveWorkbenchWindow());
    }

    /**
     * Returns the the required extension of the file to be created or 
     * null if there is no such constraint.
     * @return int
     */
    protected String getFileExtension() {
        return null;
    }

    /**
     * Opens the editor associated to a file.  Clents can overwrite.
     * @param file
     * @throws PartInitException
     */
    protected void openEditor(IFile file) throws PartInitException {
        UIUtil.openEditor(file, null, false);
    }

    /**
     * Handles any exception thrown during the 
     * {@link #performFinish()} execution.
     * @param exception
     */
    protected void handleException(String filePath, Exception exception) {
        HyadesUIPlugin.logError(exception);
        UIUtil.openSaveFileErrorDialog(getShell(), filePath, exception);
    }

    /**
     * Creates the object.  The file argument indicates the location and the
     * file name selected in the location page.
     * @param file
     * @return <code>true</code> if the object was properly created or 
     * <code>false</code> otherwise.
     * @throws Exception any exception thrown by this method is handled
     * by the {@link #handleException(Exception)} method.
     */
    abstract protected boolean createObject(IFile file) throws Exception;

    /**
     * Returns the newly created file if this wizard is completed successfully, otherwise
     * returns <code>null</code>.
     * 
     * @return the new file object or <code>null</code>
     * @author bjiang
     */
    public IFile getNewFile() {
        return this.newFile;
    }

    /**
     * @param attributeWizardPage The attributeWizardPage to set.
     */
    protected void setAttributeWizardPage(AttributeWizardPage attributeWizardPage) {
        this.attributeWizardPage = attributeWizardPage;
    }

    /**
     * @param locationPage The locationPage to set.
     */
    protected void setLocationPage(LocationPage locationPage) {
        this.locationPage = locationPage;
    }
}