/*******************************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NameAndDescriptionWizardPage.java,v 1.7 2006/10/30 18:08:50 amehregani Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

import org.eclipse.hyades.ui.internal.util.GridDataUtil;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Listener;
import org.eclipse.swt.widgets.Text;

/**
 * Wizard page that allows the user to define the name and the description
 * of the object to be created.
 * 
 * <p>Liteners can be added to instances of this class so they are notified when
 * the events defined by the <code>EVENT*</code> constants occurs.
 * 
 * <p>Subclasses may overwrite the {@link #addControl(Composite)} to insert controls
 * between the name and the description texts.
 * 
 * @author pnedelec
 * @since 3.0.3
 */
public class NameAndDescriptionWizardPage extends WizardPage implements Listener {
    /**
     * Event type fired before the page is activated.
     */
    public final static int EVENT_BEFORE_ACTIVATION = 3;

    private Text itemNameText;
    private Text itemDescriptionText;

    private String itemName;
    private String itemDescription;

    private Collection activationListeners;
    private boolean firstActivation = true;

    /**
     * Constructor for NameDescriptionWizardPage
     * @param pageName
     */
    public NameAndDescriptionWizardPage(String pageName) {
        super(pageName);
        activationListeners = new ArrayList();
    }

    /**
     * @see org.eclipse.jface.dialogs.IDialogPage#dispose()
     */
    public void dispose() {
        activationListeners.clear();

        super.dispose();
    }

    /**
     * Adds an activation listener.  Each listener is notified when this page
     * is presented to the user.
     * 
     * <p>The notification event has the <code>data</code> attribute equals to this 
     * page and the <code>type</code> attributes equals to one of the "event type" 
     * constants defined in this class. 
     *  
     * @param activationListener
     */
    public void addActivationListener(Listener activationListener) {
        activationListeners.add(activationListener);
    }

    /**
     * Removes an activation listener.
     * @param activationListener
     */
    public void removeActivationListener(Listener activationListener) {
        activationListeners.remove(activationListener);
    }

    /**
     * Removes all the activation listeners.
     */
    public void removeAllActivationListeners() {
        activationListeners.clear();
    }

    /**
     * Notifies all the activation listeners.
     * @param eventType
     */
    protected void invokeActivationListeners(int eventType) {
        for (Iterator i = activationListeners.iterator(); i.hasNext();) {
            //Each listener is supposed receive its own event
            Event e = new Event();
            e.data = this;
            e.type = eventType;

            Listener listener = (Listener) i.next();
            listener.handleEvent(e);
        }
    }

    /**
     * @see org.eclipse.jface.dialogs.IDialogPage#createControl(org.eclipse.swt.widgets.Composite)
     */
    public void createControl(Composite parent) {
        Composite composite = new Composite(parent, SWT.NONE);
        composite.setLayout(new GridLayout());
        composite.setLayoutData(GridDataUtil.createFill());

        Label label = new Label(composite, SWT.NONE);
        label.setText(UIMessages._2); 
        itemNameText = new Text(composite, SWT.BORDER);
        itemNameText.setLayoutData(GridDataUtil.createHorizontalFill());
        if (itemName != null) {
            itemNameText.setText(itemName);
            itemNameText.selectAll();
        }

        addControl(composite);

        label = new Label(composite, SWT.NONE);
        label.setText(UIMessages._1); 
        itemDescriptionText = new Text(composite, SWT.BORDER | SWT.MULTI);
        itemDescriptionText.setLayoutData(GridDataUtil.createFill());
        if (itemDescription != null) itemDescriptionText.setText(itemDescription);

        setControl(composite);


    }

    /**
     * Subclasses may overwrite to insert aditional controls between the 
     * name and the description texts.
     * @param parent
     */
    protected void addControl(Composite parent) {
    }

    /**
     * @see org.eclipse.swt.widgets.Listener#handleEvent(org.eclipse.swt.widgets.Event)
     */
    public void handleEvent(Event event) {
        if (event.type == SWT.Modify) {
            if (event.widget == itemNameText) {
                itemName = itemNameText.getText();
                setPageComplete(validatePage());
            } else if (event.widget == itemDescriptionText) {
                itemDescription = itemDescriptionText.getText();
            }
        }
    }

    /**
     * @see org.eclipse.jface.dialogs.IDialogPage#setVisible(boolean)
     */
    public void setVisible(boolean visible) {
        if (visible) invokeActivationListeners(EVENT_BEFORE_ACTIVATION);

        super.setVisible(visible);

        if (visible && firstActivation) {
            itemNameText.setFocus();
            firstActivation = false;
            itemNameText.setFocus();
            itemNameText.addListener(SWT.Modify, this);
            setPageComplete(validatePage());
            setErrorMessage(null);
        }
    }

    /**
     * Returns whether this page's controls currently all contain valid 
     * values.
     * @return <code>true</code> if all controls are valid, and
     * <code>false</code> if at least one is invalid
     */
    protected boolean validatePage() {
        setErrorMessage(null);

        //see setVisible method
        if (firstActivation) return true;

        if ((itemNameText.getText() == null) || (itemNameText.getText().trim().length() == 0)) {
            setErrorMessage(UIMessages._22); 
            return false;
        }

        return true;
    }

    /**
     * Sets the name to be presented by this wizard page.  If the name's text
     * wizard is not created then the argument is stored in a class variable.
     * @param name
     */
    public void setItemName(String name) {
        if ((itemNameText == null) || itemNameText.isDisposed()) {
            this.itemName = name;
        } else {
            itemNameText.setText(name);
            itemNameText.selectAll();
        }
    }

    /**
     * Returns the name entered by the user or set by the 
     * <code>setName</code> method.
     * @return String
     */
    public String getItemName() {
        if ((itemNameText == null) || itemNameText.isDisposed()) {
            return itemName;
        }
        return itemNameText.getText();
    }

    /**
     * Sets the description to be presented by this wizard page.  If the description's 
     * text wizard is not created then the argument is stored in a class variable.
     * @param description
     */
    public void setItemDescription(String description) {
        if ((itemDescriptionText == null) || itemDescriptionText.isDisposed())
            this.itemDescription = description;
        else
            itemDescriptionText.setText(description);
    }

    /**
     * Returns the description entered by the user or set by the 
     * <code>setDescription</code> method.
     * @return
     */
    public String getItemDescription() {
        if ((itemDescriptionText == null) || itemDescriptionText.isDisposed()) {
            return itemDescription;
        }
        return itemDescriptionText.getText();
    }

    /**
     * @return Returns the firstActivation.
     */
    protected boolean isFirstActivation() {
        return firstActivation;
    }
}