/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: CreationOperation.java,v 1.8 2005/08/16 15:30:35 popescu Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.internal.wizard.exampleproject;

import java.io.File;
import java.io.FileNotFoundException;
import java.lang.reflect.InvocationTargetException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipFile;

import org.eclipse.core.resources.ICommand;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IProjectDescription;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Path;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubProgressMonitor;
import org.eclipse.hyades.ui.HyadesUIPlugin;
import org.eclipse.hyades.ui.internal.util.StringUtil;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.hyades.ui.internal.util.UIUtil;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.IOverwriteQuery;
import org.eclipse.ui.wizards.datatransfer.ImportOperation;
import org.eclipse.ui.wizards.datatransfer.ZipFileStructureProvider;

/**
 * @author marcelop
 * @since 0.0.1
 */
public class CreationOperation 
implements IRunnableWithProgress
{
	private Map editorIdByResourceToOpen;

	private CreationWizard creationWizard;
	private IOverwriteQuery overwriteQuery;
	private IProjectAdjuster adjuster;
	
	/**
	 * Constructor for ExampleProjectCreationOperation
	 */
	public CreationOperation(CreationWizard creationWizard, IOverwriteQuery overwriteQuery)
	{
		this.creationWizard = creationWizard;
		this.overwriteQuery = overwriteQuery;
		editorIdByResourceToOpen = new HashMap();
	}
	
	public void dispose()
	{
		editorIdByResourceToOpen.clear();
		creationWizard = null;
		overwriteQuery = null;
		adjuster = null;
	}
	
	/**
	 * Returns the zipAdjuster.
	 * @return IBaseExampleProjectCreationAdjuster
	 */
	public IProjectAdjuster getAdjuster()
	{
		return adjuster;
	}

	/**
	 * Sets the adjuster.
	 * @param adjuster The adjuster to set
	 */
	public void setAdjuster(IProjectAdjuster adjuster)
	{
		this.adjuster = adjuster;
	}

	/*
	 * @see IRunnableWithProgress#run(IProgressMonitor)
	 */
	public void run(IProgressMonitor monitor)
	throws InvocationTargetException, InterruptedException
	{
		if (monitor == null)
		{
			monitor = new NullProgressMonitor();
		}
		try
		{
			CreationWizardPage[] pages = creationWizard.getCreationWizardPages();
			monitor.beginTask(UIMessages._37, pages.length);
			for (int i = 0; i < pages.length; i++)
				createProject(pages[i], new SubProgressMonitor(monitor, 1));
		}
		finally
		{
			monitor.done();
		}
	}

	public IResource[] getResourcesToOpen()
	{
		return (IResource[])editorIdByResourceToOpen.keySet().toArray(new IResource[editorIdByResourceToOpen.keySet().size()]);
	}
	
	public String getEditorId(IResource resource)
	{
		return (String)editorIdByResourceToOpen.get(resource);
	}

	private void createProject(CreationWizardPage page, IProgressMonitor monitor) 
	throws InvocationTargetException, InterruptedException
	{
		IConfigurationElement desc = page.getConfigurationElement();

		IConfigurationElement[] imports = desc.getChildren("import");
		IConfigurationElement[] natures = desc.getChildren("nature");
		IConfigurationElement[] references = desc.getChildren("references");
		int nImports = (imports == null) ? 0 : imports.length;
		int nNatures = (natures == null) ? 0 : natures.length;
		int nReferences = (references == null) ? 0 : references.length;
		
		monitor.beginTask(UIMessages._38, nImports + 1);

		String[] natureIds = new String[nNatures];
		for (int i = 0; i < nNatures; i++)
			natureIds[i] = natures[i].getAttribute("id");

		IProject[] referencedProjects = new IProject[nReferences];
		for (int i = 0; i < nReferences; i++)
			referencedProjects[i] = ResourcesPlugin.getWorkspace().getRoot().getProject(references[i].getAttribute("id"));

		IProject proj = configNewProject(page, natureIds, referencedProjects, monitor);

		for (int i = 0; i < nImports; i++)
			doImports(page, proj, imports[i], new SubProgressMonitor(monitor, 1));

		
		IConfigurationElement[] filesToOpen = desc.getChildren("open");
		
		boolean helpFile;						// Indicates whether the file to be opened is a help 
												// file or not
		
		for(int i = 0; i < filesToOpen.length; i++)
		{
			helpFile = false;
			String baseName = filesToOpen[i].getAttribute("file");
			
			/* Defect 97581 -- Ali M.: Get the value of the editorID */
			String editorId = filesToOpen[i].getAttribute("editorId");
			if (editorId == null || editorId.equals(""))
				helpFile = true;
			
			if((baseName != null) && (baseName.length() > 0))
			{
				/* Defect 97581 -- Ali M.:  We're following the policy described in the description for the 
				 * description of the editorId attribute of the open element. See sampleWizard.exsd */
				if (helpFile)
					PlatformUI.getWorkbench().getHelpSystem().displayHelpResource(baseName);
				else
				{
					/* If the editorId is specified, then the path is assumed to be relative
					 * to the project that is imported in the workspace */
					IProject project = ResourcesPlugin.getWorkspace().getRoot().getProject(page.getProjectName());
					IFile fileToOpen = project.getFile(new Path (baseName));
					UIUtil.openEditor(fileToOpen, editorId, false);
				}
			}
		}
	} 

	private IProject configNewProject(CreationWizardPage page, String[] natureIds, IProject[] referencedProjects, IProgressMonitor monitor)
	throws InvocationTargetException
	{
		try
		{
			final IProject project = page.getProjectHandle();
			IWorkspace workspace = ResourcesPlugin.getWorkspace();
			IProjectDescription description = workspace.newProjectDescription(project.getName());
			
			if(Platform.getLocation().equals(page.getLocationPath()))
				description.setLocation(null);
			else
				description.setLocation(page.getLocationPath());
				
			IConfigurationElement[] buildCommands = page.getConfigurationElement().getChildren("buildCommand");
			if(buildCommands.length > 0)
			{
				ICommand[] commands = new ICommand[buildCommands.length];
				for (int i = 0; i < buildCommands.length; i++)
				{
					String name = buildCommands[i].getAttribute("name");
					commands[i] = description.newCommand();
					commands[i].setBuilderName(name);
				}
					
				description.setBuildSpec(commands);
			}
				
			description.setNatureIds(natureIds);
			description.setReferencedProjects(referencedProjects);
			
			if(!project.exists())
			{
				project.create(description, new SubProgressMonitor(monitor, 1));
			}
			if(!project.isOpen())
			{
				project.open(new SubProgressMonitor(monitor, 1));
				project.setDescription(description, new SubProgressMonitor(monitor, 1));
			}

			return project;
		}
		catch (CoreException e)
		{
			throw new InvocationTargetException(e);
		}
	}

	private void doImports(CreationWizardPage page, IProject project, IConfigurationElement curr, IProgressMonitor monitor)
	throws InvocationTargetException, InterruptedException
	{
		try
		{
			boolean nl1Exists = false;
			
			IPath destPath;
			String name = curr.getAttribute("dest");
			if (name == null || name.length() == 0)
			{
				destPath = project.getFullPath();
			}
			else
			{
				IFolder folder = project.getFolder(name);
				if (!folder.exists())
				{
					folder.create(true, true, null);
				}
				destPath = folder.getFullPath();
			}
			
			String importPath = curr.getAttribute("src");
			if (importPath == null)
			{
				importPath = "";
				HyadesUIPlugin.logError(UIMessages._36);
				return;
			}
			
			File file = null;
			File nl1Files = null;
			String toChange = (File.separator.equals("/")?"\\":"/");
								  
			try
			{			
				URL starterURL = new URL(Platform.getBundle(curr.getDeclaringExtension().getNamespace()).getEntry("/"), importPath);
				file = new File(Platform.asLocalURL(starterURL).getFile());
				
				String moduleString = Platform.asLocalURL(starterURL).getFile();
				moduleString = moduleString.substring(0, moduleString.indexOf(importPath));
				
				if (moduleString.endsWith("/") || moduleString.endsWith("\\"))
				{
					moduleString = moduleString.substring(0,moduleString.length() - 1); 
				}
				
				String lastSegment = moduleString.substring(moduleString.lastIndexOf('/'));
				int lastDash = lastSegment.lastIndexOf('_');
				
				String version = "";
				if (lastDash != -1)
				{
					version = lastSegment.substring(lastDash);
					moduleString = moduleString.substring(0, moduleString.indexOf(version));
				}
				
				String nl1ModuleName = moduleString + ".nl1" + version;
				String nl1FullFileName = StringUtil.replace(nl1ModuleName + "/" + importPath, toChange, File.separator);
				
				nl1Files = new File(nl1FullFileName);
				if (nl1Files.exists())
				{
					nl1Exists = true;
				}
								
			}
			catch(Exception e)
			{
				String message = importPath + ":" + e.getMessage();
				Status status = new Status(IStatus.ERROR, creationWizard.getPluginId(), IStatus.ERROR, message, e);
				throw new CoreException(status);
			}
			
			
			importPath = StringUtil.replace(importPath, toChange, File.separator) + File.separator;

			if(file.isFile())
			{
				importFilesFromZip(page, file, destPath, new SubProgressMonitor(monitor, 1));				
				
				/* If the corresponding NL1 file also exists, import it */
				if (nl1Exists)
				{
					importFilesFromZip(page, nl1Files, destPath, new SubProgressMonitor(monitor, 1));
				}
			}
			else if(file.isDirectory())
			{
				importFilesFromDir(page, file, destPath, importPath, new SubProgressMonitor(monitor, 1));
				
				/* If the corresponding NL1 file also exists, import it */
				if (nl1Exists)
				{
					importFilesFromDir(page, nl1Files, destPath, importPath, new SubProgressMonitor(monitor, 1));
				}
			}
			else
			{
				Exception e = new FileNotFoundException(file.getAbsolutePath());
				String message = importPath + ":" + e.getMessage();
				Status status = new Status(IStatus.ERROR, creationWizard.getPluginId(), IStatus.ERROR, message, e);
				throw new CoreException(status);
			}
		}
		catch(CoreException e)
		{
			throw new InvocationTargetException(e);
		}
	}

	private void importFilesFromDir(CreationWizardPage page, File srcDir, IPath destPath, String importPath, IProgressMonitor monitor) 
	throws InvocationTargetException, InterruptedException
	{
		if(adjuster != null)
		{
			File auxDir = adjuster.adjust(creationWizard.getCreationWizardPages(), page, srcDir);
			if((auxDir != null) && (!srcDir.equals(auxDir)))
			{
				srcDir = auxDir;
				importPath = srcDir.getAbsolutePath();
				String toChange = (File.separator.equals("/")?"\\":"/");
				importPath = StringUtil.replace(importPath, toChange, File.separator) + File.separator;
			}
		}
		
		try
		{
			ImportOperation op = new ImportOperation(destPath, srcDir, new FileSystemStructureProvider(importPath), overwriteQuery);
			op.run(monitor);
		}
		finally
		{
			if(adjuster != null)
				adjuster.finished(page, srcDir);
		}
	}

	private void importFilesFromZip(CreationWizardPage page, File srcFile, IPath destPath, IProgressMonitor monitor) 
	throws InvocationTargetException, InterruptedException, CoreException
	{
		ZipFile zipFile = null;
		
		try
		{
			zipFile = new ZipFile(srcFile);
		}
		catch(Exception e)
		{
			String message = e.getMessage();
			Status status = new Status(IStatus.ERROR, creationWizard.getPluginId(), IStatus.ERROR, message, e);
			throw new CoreException(status);
		}

		if(adjuster != null)
		{
			ZipFile auxFile = adjuster.adjust(creationWizard.getCreationWizardPages(), page, zipFile);
			if(auxFile != null)
				zipFile = auxFile;
		}
			
		try
		{
			ZipFileStructureProvider structureProvider = new ZipFileStructureProvider(zipFile);
			ImportOperation op = new ImportOperation(destPath, structureProvider.getRoot(), structureProvider, overwriteQuery);
			op.run(monitor);
		}
		finally
		{
			if(adjuster != null)
				adjuster.finished(page, zipFile);
		}
	}
}