/********************************************************************** 
 * Copyright (c) 2005, 2008 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0 
 * which accompanies this distribution, and is available at 
 * http://www.eclipse.org/legal/epl-v10.html 
 * $Id: ConnectorService.java,v 1.7 2008/02/29 19:19:58 jkubasta Exp $ 
 * 
 * Contributors: 
 * IBM - Initial API and implementation 
 **********************************************************************/

package org.eclipse.hyades.ui.services;

import java.util.Properties;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Status;
import org.eclipse.hyades.automation.server.AbstractService;
import org.eclipse.hyades.execution.core.task.ProgressiveTask;
import org.eclipse.hyades.execution.core.util.MutableObject;
import org.eclipse.hyades.internal.execution.local.control.Application;
import org.eclipse.hyades.security.internal.util.BaseConnectUtil;
import org.eclipse.hyades.security.internal.util.ConnectUtil;
import org.eclipse.tptp.platform.common.internal.services.HeadlessConnectorService;

/**
 * The UI connector service will establish a connection with the Agent Controller
 * in the appropriate security mode (secure or unsecure) and return back the
 * node object that was connected through (by updating the mutable node in out
 * parameter passed in). If credentials are required for a secure connection
 * and the credentials (user and password) are not passed in as service-related 
 * properties in the service properties state, then a UI will appear and query 
 * the user for the credentials.
 * <p/>
 * It is important than a mutable node is passed in, this is where the node
 * return value is set and read once the service has completed and returned back
 * to the client.
 * <p/>
 * Support properties (default values) include:
 * <ul>
 * <li>host (<code>localhost</code>)</li>
 * <li>port (<code>10002</code>)</li>
 * <li>user* (<code>null</code>)</li>
 * <li>password* (<code>null</code>)</li>
 * <li>showErrors (<code>false</code>)</li>
 * <li>mutableNode (<code>null</code>)</li>
 * </ul>
 * <p/>
 * *User and password are optional if connecting in a unsecure mode.
 * <p/>
 * <b>Implementation Note:</b> It is desirable for the dependency on test and profile to 
 * go away and move this service into the execution framework (as soon as the 
 * functionality supplied by the profile connect utility is migrated into the 
 * execution framework.
 * <p/>
 * This class requires the workbench to be running.  For example:  
 * <p/>
 * <code>if(PlatformUI.isWorkbenchRunning())</code>
 * <p/>
 * For establishing a connection with the Agent Controller in a headless environment, 
 * use the {@link HeadlessConnectorService}.
 * <p/>
 * 
 * 
 * @author  Paul E. Slauenwhite
 * @author  Scott E. Schneider
 * @version February 28, 2008
 * @since   September 26, 2005
 * @see     HeadlessConnectorService
 */
public class ConnectorService extends AbstractService implements Application {

	/**
	 * Used for serialization purposes
	 */
	private static final long serialVersionUID = 8355004980554497424L;	

	/**
	 * The connector service will retrieve the various service-related
	 * properties from the service properties state and then use the profile
	 * connect utilities to make a connection to the Agent Controller (using
	 * secure or unsecure mode).
	 * <p/>
	 * Support properties (default values) include:
	 * <ul>
	 * <li>host (<code>localhost</code>)</li>
	 * <li>port (<code>10002</code>)</li>
	 * <li>user* (<code>null</code>)</li>
	 * <li>password* (<code>null</code>)</li>
     * <li>showErrors (<code>false</code>)</li>
	 * <li>mutableNode (<code>null</code>)</li>
	 * </ul>
     * *User and password are optional if connecting in a unsecure mode.
	 * <p/>
	 * 
	 * @return The status (IStatus) of the service.
	 * @see org.eclipse.hyades.automation.core.Service#execute()
	 */
	public Object execute() {
		
		Properties properties = this.getProperties();
		String host = properties.getProperty("host");
		String port = properties.getProperty("port");
		String user = properties.getProperty("user");
		String password = properties.getProperty("password");
		Boolean showErrors = ((Boolean) (properties.get("showErrors")));
		MutableObject mutableNode = ((MutableObject) (properties.get("mutableNode")));
		
		return (this.connect((host == null ? "localhost" : host), (port == null ? "10002" : port), user, password, showErrors
				.booleanValue(), mutableNode));
	}

	/**
	 * Resolves the name of the service.
	 * <p/>
	 * 
	 * @return Name of the service
	 * @see java.security.Principal#getName()
	 */
	public String getName() {
		return "Agent Controller connector service";
	}
	
	/**
	 * Underlying connect method delegated on to connect utility in the 
	 * security package, used for unsecure and secure connections.
	 * 
	 * @param host
	 *            host to connect on
	 * @param port
	 *            port to use for the connection
	 * @param user
	 *            user to be used for a secure connection
	 * @param password
	 *            password to be used for a secure connection
	 * @param password
	 *            password to be used for a secure connection
	 * @param showErrors
	 *            true allows the service to display UI dialogs
	 * @return The status (IStatus) of the service.
	 */
	private IStatus connect(final String host, final String port, final String user, final String password,
			final boolean showErrors, final MutableObject mutableNode) {

		// A mutable object to hold the found agent
		final MutableObject mutableStatus = new MutableObject();

		// Progressive task to connect to agent controller
		ProgressiveTask connector = new ProgressiveTask("Agent Controller Connect", new Runnable() {
			
			public void run() {
			
				try {

					// Use the internal connection utility
					BaseConnectUtil connectUtil = new ConnectUtil(host, port, user, ConnectorService.this);

					int result = connectUtil.connect(password, showErrors);

					// Handle results, success and exceptions
					switch (result) {
				
					case ConnectUtil.CONNECTION_SUCCESS: {
						mutableStatus.set(new Status(IStatus.OK, ConnectorService.this.getName(), result, "", null));
						mutableNode.set(connectUtil.getNode());
						break;
					}
					case ConnectUtil.CONNECTION_CONTROLLER_ERROR:
					case ConnectUtil.CONNECTION_HOST_ERROR:
					case ConnectUtil.CONNECTION_PORT_ERROR:
					case ConnectUtil.CONNECTION_SECURITY_NOT_SUPPORTED:
					default: {
						mutableStatus.set(new Status(IStatus.ERROR, ConnectorService.this.getName(), result, "", null));
					}
					}

				} catch (Throwable t) {
					//Ignore.
				}
			}
		}, new NullProgressMonitor(), 10000);

		// Blocks until runnable completes normally or is canceled
		connector.execute(ProgressiveTask.Synchronicity.SYNCHRONOUS);

		// Return status (and node) to service consumer
		return (IStatus) mutableStatus.getAndClear();
	}
}