/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: TreeSelector.java,v 1.1 2005/04/29 09:22:09 dguilbaud Exp $
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.hyades.ui.util;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;

import org.eclipse.jface.viewers.CheckStateChangedEvent;
import org.eclipse.jface.viewers.CheckboxTreeViewer;
import org.eclipse.jface.viewers.ICheckStateListener;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.viewers.ITreeViewerListener;
import org.eclipse.jface.viewers.TreeExpansionEvent;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.viewers.ViewerFilter;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.BusyIndicator;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Tree;

/**
 * Created on 30 oct. 2003 by pnedelec
 *
 */
public class TreeSelector extends Container implements ICheckStateListener, ITreeViewerListener {
    private Tree tree;
    protected CheckboxTreeViewer treeViewer;
    protected Object input;
    protected ITreeContentProvider contentProvider;
    private final ILabelProvider labelProvider;
    protected List alreadyExpandedItems;
    private HashMap numberOfCheckedChildren;
    private HashMap numberOfGrayedChildren;

    /**
     * 
     * @param parent
     * @param input
     * @param contentProvider
     * @param labelProvider
     * @param style
     */
    public TreeSelector(Composite parent, Object input, ITreeContentProvider contentProvider, ILabelProvider labelProvider, int style) {
        super();

        this.input = input;
        this.contentProvider = contentProvider;
        this.labelProvider = labelProvider;
        init();
        initMap();
        createContents(parent, style);
    }

    private void init() {
        alreadyExpandedItems = new ArrayList();
    }

    private void initMap() {
        numberOfCheckedChildren = new HashMap();
        numberOfGrayedChildren = new HashMap();
        Object[] children = contentProvider.getChildren(input);
        for (int i = 0; i < children.length; ++i) {
            numberOfCheckedChildren.put(children[i], new Integer(0));
            numberOfGrayedChildren.put(children[i], new Integer(0));
        }
    }

    /**
     * Use this method when the TreeSelector contents has changed.
     * @param the new input
     */
    public void setInput(Object input) {
        this.input = input;
        init();
        initMap();
        if (treeViewer != null) {
            treeViewer.setInput(input);
        }
    }

    /**
     * Use this method to change the current content provider.
     * Call the setInput(Object ) method to make this change available.
     * @param contentProvider
     */
    public void setContentProvider(ITreeContentProvider contentProvider) {
        this.contentProvider = contentProvider;
        if (treeViewer != null) {
            treeViewer.setContentProvider(contentProvider);
        }
    }

    /**
     * Clears the selector.
     *
     */
    public void clear() {
        if (treeViewer == null) return;
        ViewerFilter eraser = new ViewerFilter() {
            public boolean select(Viewer viewer, Object parentElement, Object element) {
                return false;
            }
        };
        treeViewer.addFilter(eraser);
        init();
        initMap();
        treeViewer.removeFilter(eraser);
    }

    /**
     * 
     * @param parent
     * @param style
     */
    protected void createContents(Composite parent, int style) {
        tree = new Tree(parent, SWT.CHECK | style);
        GridData data = new GridData(GridData.FILL_BOTH);
        tree.setLayoutData(data);
        tree.setFont(parent.getFont());

        treeViewer = new CheckboxTreeViewer(tree);
        treeViewer.setContentProvider(contentProvider);
        treeViewer.setLabelProvider(labelProvider);
        treeViewer.setInput(input);
        treeViewer.addCheckStateListener(this);
        treeViewer.addTreeListener(this);
    }

    /**
     * @see org.eclipse.jface.viewers.ICheckStateListener#checkStateChanged(org.eclipse.jface.viewers.CheckStateChangedEvent)
     */
    public void checkStateChanged(final CheckStateChangedEvent event) {
        //May be a long operation due to the update of the parent...
        BusyIndicator.showWhile(treeViewer.getControl().getDisplay(), new Runnable() {
            public void run() {
                checkElement(event.getElement(), event.getChecked());
            }
        });
        contentsChanged();
    }

    protected void checkElement(Object element, boolean isChecked) {
        //- update the number of checked elements of the direct parent (only)
        Integer nb = (Integer) numberOfCheckedChildren.get(contentProvider.getParent(element));
        if (contentProvider.getParent(element) != input) {
            numberOfCheckedChildren.put(contentProvider.getParent(element), nb == null ? new Integer(isChecked ? 1 : 0) : new Integer(isChecked ? nb.intValue() + 1 : ((nb
                    .intValue() >= 1) ? nb.intValue() - 1 : 0)));
        }

        //- changing the check state of an item makes it white
        //- update the map only if the item was grayed (-1 on the associated value)
        setGrayed(element, false, treeViewer.getGrayed(element));

        updateTreeItems(element, isChecked);
    }

    /**
     * 
     * @param element
     * @param state
     */
    private void updateTreeItems(Object element, boolean state) {
        setChildrenChecked(element, state);
        updateParent(element, state);
    }

    /**
     * 
     * @param element
     * @param state
     */
    private void updateParent(Object element, boolean state) {
        Object parentElem = contentProvider.getParent(element);
        while (parentElem != input && parentElem != null) {
            Integer nbChecked = (Integer) numberOfCheckedChildren.get(parentElem);
            if (nbChecked == null) {
                nbChecked = new Integer(0);
            }
            boolean oldIsCkecked = treeViewer.getChecked(parentElem);
            boolean oldIsGrayed = treeViewer.getGrayed(parentElem);
            Integer nbGrayed = (Integer) numberOfGrayedChildren.get(parentElem);
            if (nbGrayed == null) {
                nbGrayed = new Integer(0);
            }
            boolean isChecked = (nbChecked.intValue() > 0);
            boolean isGrayed = (nbGrayed.intValue() > 0 || (nbChecked.intValue() > 0 && nbChecked.intValue() < contentProvider.getChildren(parentElem).length));
            //- check the element and update the number of checked items of its direct parent		
            //- do not add a checked children to the parent's parent if the first parent keep the same state				
            setWhiteChecked(parentElem, isChecked, isChecked != oldIsCkecked);

            //- Update the gray state
            setGrayed(parentElem, isGrayed, isGrayed != oldIsGrayed);

            parentElem = contentProvider.getParent(parentElem);
        }
    }

    /**
     * 
     * @param item
     */
    private void expandTreeItem(final Object item) {
        //- expanding a tree item creates its children
        //- Those are stored in a list to optimize the update of the tree
        BusyIndicator.showWhile(treeViewer.getControl().getDisplay(), new Runnable() {
            public void run() {
                if (!alreadyExpandedItems.contains(item)) {
                    //- Upon first display, check children if parent are checked
                    alreadyExpandedItems.add(item);
                    Object[] children = contentProvider.getChildren(item);
                    for (int i = 0; i < children.length; ++i) {
                        Object child = children[i];
                        //- add the children to the list of already displayed tree items
                        if (treeViewer.getChecked(item) == true) {
                            //- it's time to check the item and to update the map of the parent
                            setWhiteChecked(child, true, true);
                        }
                    }
                }

            }
        });
    }

    /**
     * 
     * @param treeElement
     * @param isWhiteChecked
     * @param withMapUpdate
     */
    protected void setWhiteChecked(Object treeElement, boolean isWhiteChecked, boolean withMapUpdate) {
        //- Changing the state of an item ungrayed it and its children
        setGrayed(treeElement, false, false);
        treeViewer.setChecked(treeElement, isWhiteChecked);
        if (!withMapUpdate || contentProvider.getParent(treeElement) == input) {
            return;
        }
        Integer nb = (Integer) numberOfCheckedChildren.get(contentProvider.getParent(treeElement));
        if (isWhiteChecked) {
            numberOfCheckedChildren.put(contentProvider.getParent(treeElement), nb == null ? new Integer(1) : new Integer(nb.intValue() + 1));
        } else {
            numberOfCheckedChildren.put(contentProvider.getParent(treeElement), (nb != null) && (nb.intValue() > 0) ? new Integer(nb.intValue() - 1) : new Integer(0));
        }
    }

    /**
     * 
     * @param elem
     * @param isGrayed
     * @param withMapUpdate
     */
    private void setGrayed(Object elem, boolean isGrayed, boolean withMapUpdate) {
        treeViewer.setGrayed(elem, isGrayed);
        if (!withMapUpdate || contentProvider.getParent(elem) == input) {
            return;
        }
        Integer nb = (Integer) numberOfGrayedChildren.get(contentProvider.getParent(elem));
        if (isGrayed) {
            //Because a grayed item is always checked 
            treeViewer.setChecked(elem, true);
            numberOfGrayedChildren.put(contentProvider.getParent(elem), nb == null ? new Integer(1) : new Integer(nb.intValue() + 1));
        } else {
            numberOfGrayedChildren.put(contentProvider.getParent(elem), (nb != null) && (nb.intValue() > 0) ? new Integer(nb.intValue() - 1) : new Integer(0));
        }
    }

    /**
     * 
     * @param b
     */
    public void setAllSelections(final boolean b) {
        //- It may be long for big project
        BusyIndicator.showWhile(treeViewer.getControl().getDisplay(), new Runnable() {
            public void run() {
                //- set the state of all the tree items to b
                //- for the highest level tree items:
                Object[] elements = contentProvider.getElements(input);
                for (int i = 0; i < elements.length; ++i) {
                    setWhiteChecked(elements[i], b, false);
                    setChildrenChecked(elements[i], b);
                }
            }
        });
        contentsChanged();
    }

    /**
     * 
     * @param element
     * @param state
     */
    protected void setChildrenChecked(Object element, boolean state) {
        //- Check the children of already expanded items to the state state 
        if (!alreadyExpandedItems.contains(element)) return;

        //- the element has been expanded, the associated maps should be updated
        Object[] children = contentProvider.getChildren(element);
        //- update of the maps 
        if (element != input) {
            numberOfCheckedChildren.put(element, state ? new Integer(children.length) : new Integer(0));
            numberOfGrayedChildren.put(element, new Integer(0));
        }
        //- cascade to the children
        for (int i = 0; i < children.length; ++i) {
            setWhiteChecked(children[i], state, false);
            setChildrenChecked(children[i], state);
        }
    }

    /**
     * Select only the top level tree items
     * @param objects
     */
    public void setInitialCheckedRootElements(Object[] objects) {
        for (int i = 0; i < objects.length; ++i) {
            treeViewer.setChecked(objects[i], true);
        }
    }

    /**
     * Select tree items even even if they are not at the top level
     * @param elements
     */
    public void setInitialCheckedElements(List elements) {
        if (elements == null) {
            return;
        }
        for (Iterator it = elements.iterator(); it.hasNext();) {
            //- For each item, you need to update first the data and then the graphical state
            Object obj = it.next();
            Object parentElem = contentProvider.getParent(obj);
            //- The idea is to simulate a user click...
            while (parentElem != input && parentElem != null) {
                //- update of the maps
                expandTreeItem(parentElem);
                parentElem = contentProvider.getParent(parentElem);
            }
            treeViewer.expandToLevel(obj, 0);
            checkElement(obj, true);
            treeViewer.setChecked(obj, true);
        }
    }

    /**
     * 
     * @return the list of selected elements without their children if they are all checked
     */
    public List getSelectedElements() {
        return getChildrenChecked(input);
    }

    /**
     * This method returns the list of selected elements that are instances of the class 'filter'.
     * e.g : This will return the list of checked IMethods displayed in myTreeSelector:
     * 		<code>Class filter = IMethod.class;</code>
     * 		<code>myTreeSelector.getSelectedElements(filter);</code>
     * @param filter a class
     * @return the list of selected elements that are instances of the class 'filter'
     */
    public List getSelectedElements(Class filter) {
        return getChildrenChecked(input, filter, false);
    }

    private List getChildrenChecked(Object element, Class filter, boolean whiteChecked) {
        List ret = new ArrayList();
        if (element != input) {
            if (!whiteChecked && !treeViewer.getChecked(element)) {
                // whiteChecked is the state of the parent of element
                return ret;
            }
            if (!treeViewer.getGrayed(element) && treeViewer.isExpandable(element)) {
                // If the current element is checked and not greyed, then all the 
                // children are checked ...
                whiteChecked = true;
            }
            if (filter.isAssignableFrom(element.getClass())) {
                ret.add(element);
            }
        }
        Object[] children = contentProvider.getChildren(element);
        for (int i = 0; i < children.length; ++i) {
            ret.addAll(getChildrenChecked(children[i], filter, whiteChecked));
        }
        return ret;
    }

    /**
     * @param input
     * @return
     */
    private List getChildrenChecked(Object element) {
        List ret = new ArrayList();
        if (element != input) {
            if (!treeViewer.getGrayed(element) && treeViewer.getChecked(element)) {
                //- a parent white and checked -> STOP
                ret.add(element);
                return ret;
            }
            if (!treeViewer.getGrayed(element) && !treeViewer.getChecked(element)) {
                //- a parent not grayed and not checked -> STOP
                return new ArrayList();
            }
        }
        Object[] children = contentProvider.getChildren(element);
        for (int i = 0; i < children.length; ++i) {
            ret.addAll(getChildrenChecked(children[i]));
        }
        return ret;
    }

    /**
     * 
     * @return the tree viewer
     */
    public CheckboxTreeViewer getTreeViewer() {
        return treeViewer;
    }

    /**
     * 
     * @return
     */
    public Control getControl() {
        return tree;
    }

    /*
     * @see org.eclipse.jface.viewers.ITreeViewerListener#treeCollapsed(org.eclipse.jface.viewers.TreeExpansionEvent)
     */
    public void treeCollapsed(TreeExpansionEvent event) {
    }

    /*
     * @see org.eclipse.jface.viewers.ITreeViewerListener#treeExpanded(org.eclipse.jface.viewers.TreeExpansionEvent)
     */
    public void treeExpanded(TreeExpansionEvent event) {
        expandTreeItem(event.getElement());
    }
}