/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/
package org.eclipse.hyades.ui.widgets.grapher;

import org.eclipse.hyades.ui.widgets.grapher.graphs.LineGraph;
import org.eclipse.hyades.ui.widgets.grapher.sources.TestGraphSource;
import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderEvent;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderListener;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.layout.FillLayout;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Shell;

/**
 * A sample implementation of the <code>grapher</code> and <code>zoomslider</code> apis.
 * The sample displays a cosine and a sine line graph on a graph canvas, along with 
 * x and y sliders. 
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class APIExample implements ZoomSliderListener  
{
    //The SINE and COSINE line graphs
    private LineGraph sinLineGraph;
    private LineGraph cosLineGraph;

    // The respective graph sources. The line graphs are responsible for displaying
    // data gathered from these graph sources
    private GraphSource sinGraphSource;
    private GraphSource cosGraphSource;    
    
    // The line colours of the line graphs
    private Color redGraphColor;
    private Color blueGraphColor;
    
    // The graph composite, which contains the GraphCanvas and two ZoomSliders
    private Composite graphComposite;
        private ZoomSlider ySlider;    
        private GraphCanvas canvas;
        private Composite empty;
        private TimeZoomSlider xSlider;           
    
    /**
     * Run this Graphing example.
     * 
     * @param args standard command line parameters, not currently used
     * @throws Exception if the graphing fails
     */
    public static void main(String [] args) throws Exception
    {
        APIExample grapher = new APIExample();
        grapher.newGraphShell();
    }
    
    /**
     * Setup the graph composite.
     * 
     * @param parent the parent composite
     * @throws Exception if the setup fails
     */        
    public void setupComposite(Composite parent) throws Exception
    {
        GridLayout gridLayout;
        GridData gridData;
        
            graphComposite = new Composite(parent, SWT.NONE);
            gridLayout = new GridLayout(2,false);
            gridLayout.horizontalSpacing = 0;
            gridLayout.verticalSpacing = 0;            
            graphComposite.setLayout(gridLayout);
            
                //Create the Y Slider            
                ySlider = new ZoomSlider(graphComposite);
                gridData = new GridData(SWT.FILL, SWT.FILL,false,true);
                ySlider.setLayoutData(gridData);
                ySlider.addZoomSliderListener(this);
                // Set the initial configuration for the y slider
                ySlider.configure(0,200,50,150,2);
            
                //Create the graph canvas
                canvas = new GraphCanvas(graphComposite, SWT.NONE);
                gridData = new GridData(SWT.FILL, SWT.FILL,true,true);                
                canvas.setLayoutData(gridData);                              
                                
                empty = new Composite(graphComposite,SWT.NONE);
                gridData = new GridData(SWT.FILL, SWT.FILL,false,false);
                empty.setLayoutData(gridData);

                //Create the X Slider
                xSlider = new TimeZoomSlider(graphComposite);
                gridData = new GridData(SWT.FILL, SWT.FILL,true,false);                
                // Set the orientational to HORIZONTAL and direction decreasing (right to left)
                xSlider.setDirection(ZoomSlider.DECREASING);                
                xSlider.setOrientation(ZoomSlider.HORIZONTAL);
                xSlider.setLayoutData(gridData);                
                xSlider.addZoomSliderListener(this);
                
                // Configure the initial settings for this slider
                xSlider.configure(-10000,10000,-200,200,2);                
    }
        
    /**
     * Intialize the Graphing. Create the graph colours, graph sources, line graphs
     * and add these to the canvas.
     */
    public void createGrapher()
    {
        // Set up the red and blue graph colors
        redGraphColor = new Color(Display.getCurrent(),255,0,0);
        blueGraphColor = new Color(Display.getCurrent(),0,0,255);
        
        // Create the sin and cosine graph sources
        sinGraphSource = createGraphSource(TestGraphSource.SINE_WAVE);       
        cosGraphSource = createGraphSource(TestGraphSource.COSINE_WAVE);
        
        // Construct a line graph to plot the points of sin data source.
        // Set the PLOTTING_TYPE_AVERAGE to plot average results. These results are obtained
        // from the graph source via the getAv
        sinLineGraph = new LineGraph(LineGraph.PLOTTING_TYPE_AVERAGE, sinGraphSource, xSlider, ySlider, redGraphColor);
        
        cosLineGraph = new LineGraph(LineGraph.PLOTTING_TYPE_AVERAGE, cosGraphSource, xSlider, ySlider, blueGraphColor);

        // Add the linegraphs to the graph canvas
        canvas.addGraph(sinLineGraph);
        canvas.addGraph(cosLineGraph);                
    }
    
    /**
     * Create a new Test Graph Source. This graph source provides the data that 
     * will be graphed by a particular graph.
     * 
     * @param type the type of graph source TestGraphSource.SINE_WAVE or TestGraphSource.COSINE_WAVE
     * @return the test Graph Source
     */
    public GraphSource createGraphSource(int type)
    {
        // The bounds of data that we require from the graph source
        int min = -10000;
        int max = 10000;
                
        TestGraphSource testGraphSource = new TestGraphSource(min,max,type);
        
        return testGraphSource;
    }
    
    /**
     * Create the Graph Shell
     * 
     * @throws Exception if the graphing
     */
    public void newGraphShell() throws Exception
    {
        Display display = new Display();
        Shell shell = new Shell(display);
        shell.setLayout(new FillLayout());
        
        setupComposite(shell);
        createGrapher();
        
        shell.setText("ZoomSlider and Grapher Example");
        shell.setSize(400,400);
        shell.open();
        while( !shell.isDisposed())
        {
            if(!display.readAndDispatch()) 
                display.sleep();
        }
        
        // Dispose system resources
        sinLineGraph.dispose();
        cosLineGraph.dispose();
        redGraphColor.dispose();
        blueGraphColor.dispose();
        
        display.dispose();
    }
        
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderListener#zoomSliderChanged(org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderEvent)
     */
    public void zoomSliderChanged(ZoomSliderEvent event)
    {        
        // Redraw the canvas on a zoom slider change
        canvas.redraw();
    }   
}
