/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher;

import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;


/**
 * The Graph interface describes any graph entity that is to be rendered
 * and the plotting settings to be used to plot the graph. A graph implementation
 * contains references to a valid GraphSource and Indicator Source.<br><br>
 * 
 * Graphs are rendered by adding implementations to the Graph Canvas. The 
 * <code>paintGraph</code> method is responsible for rendering the graph on the graphing
 * canvas. This method is invoked by a call to <code>redraw</code> in the Graph Canvas.
 * 
 * @author gchristelis
 * @since 4.0.0
 * @see org.eclipse.hyades.ui.widgets.grapher.GraphCanvas
 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource
 * @see org.eclipse.hyades.ui.widgets.grapher.IndicatorSource
 */
public interface Graph 
{
    /**
     * Average plotting type constant.<br><br>
     * 
     * Plot the average value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getAverageBetween(double, double) 
     */
	public static final int PLOTTING_TYPE_AVERAGE = 0;

    /**
     * Minimum plotting type constant.<br><br>
     * 
     * Plot the minimum value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMinBetween(double, double)
     */    
	public static final int PLOTTING_TYPE_MIN = 1;
    
    /**
     * Maximum plotting type constant.<br><br>
     * 
     * Plot the maximum value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMax()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMaxBetween(double, double)
     */        
	public static final int PLOTTING_TYPE_MAX = 2;
    
    /**
     * Maximum and Minimum plotting type constant.<br><br>
     * 
     * Plot the maximum and minimum value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMax()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMaxBetween(double, double)
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMinBetween(double, double)
     */      
	public static final int PLOTTING_TYPE_MIN_MAX = 3;
    
    /**
     * Maximum and Minimum plotting type constant.<br><br>
     * 
     * Plot the maximum, minimum and average value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMax()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMaxBetween(double, double)
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMinBetween(double, double)
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getAverageBetween(double, double)       
     */   
	public static final int PLOTTING_TYPE_MIN_MAX_AVERAGE = 4;
    
	/**
     * Aggregation plotting type constant.<br><br>
     * 
     * Plot the sum of value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getSumBetween(double, double)
     */    
	public static final int PLOTTING_TYPE_SUM = 5;
    
    /**
     * Nearest plotting type constant.<br><br>
     * 
     * Plot the nearest previous value to the current index from the graph source.<br>
     * 
     */     
	public static final int PLOTTING_TYPE_NEAREST = 6;
    
    /**
     * Count plotting type constant.<br><br>
     * 
     * Plot the number of data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getCountBetween(double, double)
     */     
	public static final int PLOTTING_TYPE_COUNT = 7;
    
    /**
     * Standard deviation plotting type constant.<br><br>
     * 
     * Plot the standard deviation of data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getStandardDeviationBetween(double, double)
     */         
	public static final int PLOTTING_TYPE_STDDEV = 8;
    
    /**
     * Standard deviation and average plotting type constant.<br><br>
     * 
     * Plot the standard deviation and average of value data points from the graph source.<br>
     * 
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getAverageBetween(double, double)
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getStandardDeviationBetween(double, double)      
     */         
	public static final int PLOTTING_TYPE_STDDEV_AVERAGE = 9;
    
    /**
     * Gradient plotting type constant.<br><br>
     * 
     * Plot the estimated gradient of value data points from the graph source.<br>
     */         
	public static final int PLOTTING_TYPE_GRADIENT = 10;
	
    /**
     * Tick period type constant.<br><br>
     * 
     * The type of data sampling between data points is set to the slider tick width associated with this graph. 
     */             
    public static final int PERIOD_TYPE_TICKS = 0;

    /**
     * Millisecond period type constant.<br><br>
     * 
     * The data sampling period between data points is set milliseconds (each time frame in milliseconds will have a value associated with it). 
     */                
	public static final int PERIOD_TYPE_MILLIS = 1;
    
	/**
     * Pixel period type constant.<br><br>
     * 
     * The data sampling period between data points is set to pixels (each x pixel will have a value associated with it). 
     */                    
	public static final int PERIOD_TYPE_PIXELS = 2;

    /**
     * Pixel period type constant.<br><br>
     * 
     * The data sampling period between data points is set to unaveraged. 
     */                    
	public static final int PERIOD_TYPE_UNAVERAGED = 3;
	
	/**
     * Do Nothing on No Data constant.<br><br>
     * 
     * Plot no values if there is no data between data points
     */             
	public static final int NODATA_DO_NOTHING = 0;
    
    /**
     * Plot zero on No Data constant.<br><br>
     * 
     * Plot a zero value for all missing values between data points
     */                 
	public static final int NODATA_DRAW_ZERO = 1;
    
    /**
     * Draw Previous on No Data constant.<br><br>
     * 
     * On missing data draw the previous data value that was calculated
     */       
	public static final int NODATA_DRAW_PREVIOUS = 2;
	
	/**
	 * Ignore no data, and draw interpolated lines between data points
	 * 
	 * 
	 */
	public static final int NODATA_INTERPOLATE = 3;
    
    /**
     * Get the GraphSource associated with this Graph.<br>
     * 
     * @return a BasicGraphSource associated with this graph 
     */
	public BasicGraphSource getGraphSource();
    
    /**
     * Get the minimum X value from the data source.
     * 
     * @return the minimum X value
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
     */
	public double getXMin();
    
    /**
     * Get the minimum X value from the data source.
     * 
     * @return the minimum X value
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
     */
	public double getXMax();
    
    /**
     * Get the minimum Y value from the data source.
     * 
     * @return the minimum Y value
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getValueMin()
     */    
	public double getYMin();
    
    /**
     * Get the maximum Y value from the data source.
     * 
     * @return the maximum Y value
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getValueMax()
     */    
	public double getYMax();
    
    /**
     * Paint this graph on the provided GC, within the given bounds. 
     * The GC uses a coordinate system increasing from LEFT to RIGHT and 
     * TOP to BOTTOM.<br><br>
     * 
     * The GraphCanvas is responsible for calling the painGraph method on
     * a redraw.
     * 
     * @param gc an SWT GC on which the implemented graph should be drawn
     * @param x the x coordinate bound, in pixels
     * @param y the y coordinate bound, in pixels
     * @param w the width of the drawing canvas, in pixels
     * @param h the height of the drawing canvas, in pixels
     * 
     * @see org.eclipse.swt.graphics.GC
     */
	public void paintGraph(GC gc, int x, int y, int w, int h);
    
     /**
      * Set the X TimeZoomSlider that this graph will use for horizontal point indexing
      * 
      * @param slider The TimeZoomSlider to be used as the X coordinate slider
      */
	public void setXSlider(TimeZoomSlider slider);
    
    /**
     * Set the Y ZoomSlider that this graph will use for vertical point indexing
     * 
     * @param slider The ZoomSlider to be used as the Y coordinate slider
     */    
	public void setYSlider(ZoomSlider slider);
    
    /** 
     * Get this graphs X slider
     * 
     * @return the TimeZoomSlider associated with this graph's Y coordinates
     */
	public TimeZoomSlider getXSlider();
    
    /** 
     * Get this graphs Y slider
     * 
     * @return the ZoomSlider associated with this graph's Y coordinates
     */    
	public ZoomSlider getYSlider();
    
    /**
     * Set the foreground color to be used for drawing this graph
     * @param col the Color for this graph to be drawn with
     */
	public void setForeground(Color col);
    
    /**
     * Return the current color associated with the foreground drawing color of this graph
     * @return the currently assigned foreground Color
     */
	public Color getForeground();
    
    /**
     * Set the line width of this graph
     * @param w the new line width value
     */
	public void setLineWidth(int w);
    
    /**
     * Get the line width of this graph
     * @return an integer representing the current line width of this graph
     * @see #getLineWidth()
     */    
	public int getLineWidth();
    
    /**
     * Set the line style of this graph. It is common practice to use the line style SWT constants similar to those for the GC. Since the line style is an integer constant the underlying implementation can interpret this in any desired way.
     * @see org.eclipse.swt.graphics.GC#setLineStyle(int)
     * @see org.eclipse.swt.SWT
     * @param swt_line_style the SWT line style constant
     */
	public void setLineStyle(int swt_line_style);
    
    /**
     * Return this graph's line style
     * @return an integer representing this graphs line style, as set by the setLineStyle(int swt_line_style) method.
     * @see #setLineStyle(int)
     */
	public int getLineStyle();
    
    /**
     * Set the static scaling to be used when plotting data points. The static scaling is the multiplier applied to the data point values.
     * @param mult the double multiplier value
     */
	public void setStaticScaling(double mult);
    
    /**
     * Get the static scaling constant.
     * 
     * @return mult the double multiplier value
     * @see #setStaticScaling(double) 
     */
	public double getStaticScaling(); 
	
    /** 
     * 
     * Return the current Graph Plotting Type constant. Valid constant values are 
     * <li>PLOTTING_TYPE_AVERAGE</li> 
     * <li>PLOTTING_TYPE_MIN</li> 
     * <li>PLOTTING_TYPE_MAX</li>
     * <li>PLOTTING_TYPE_MIN_MAX</li> 
     * <li>PLOTTING_TYPE_MIN_MAX_AVERAGE</li>  
     * <li>PLOTTING_TYPE_SUM </li>
     * <li>PLOTTING_TYPE_NEAREST </li>
     * <li>PLOTTING_TYPE_COUNT </li>
     * <li>PLOTTING_TYPE_STDDEV</li>
     * <li>PLOTTING_TYPE_STDDEV_AVERAGE</li> 
     * <li>PLOTTING_TYPE_GRADIENT</li>
     * 
     * @return an integer representing the current plotting type
    **/
	public int getPlottingType();
    
    /** 
     * Set the current Graph Plotting Type constant. Valid constant values are 
     * <li>PLOTTING_TYPE_AVERAGE</li> 
     * <li>PLOTTING_TYPE_MIN</li> 
     * <li>PLOTTING_TYPE_MAX</li>
     * <li>PLOTTING_TYPE_MIN_MAX</li> 
     * <li>PLOTTING_TYPE_MIN_MAX_AVERAGE</li>  
     * <li>PLOTTING_TYPE_SUM </li>
     * <li>PLOTTING_TYPE_NEAREST </li>
     * <li>PLOTTING_TYPE_COUNT </li>
     * <li>PLOTTING_TYPE_STDDEV</li>
     * <li>PLOTTING_TYPE_STDDEV_AVERAGE</li> 
     * <li>PLOTTING_TYPE_GRADIENT</li>
     * 
     * @param type an integer constant representing the desired plotting type
     */    
	public void setPlottingType(int type);

    /**
     * Return the no data behaviour constant. This behaviour dictates the points plotted when no data is available. Possible values are
     * <li>NODATA_DO_NOTHING</li>
     * <li>NODATA_DRAW_ZERO</li>
     * <li>NODATA_DRAW_PREVIOUS</li>
     * @return the integer no data constant
     */
	public int getNoDataBehaviour();
    
    /**
     * Set the no data behaviour constant. This behaviour dictates the points plotted when no data is available. Possible values are
     * <li>NODATA_DO_NOTHING</li>
     * <li>NODATA_DRAW_ZERO</li>
     * <li>NODATA_DRAW_PREVIOUS</li>
     * @param behaviour the No Data behaviour constant to be used when plotting this graph
     */    
	public void setNoDataBehaviour(int behaviour);
	
    /**
     * Set the plotting period type and period value. The plotting period dictates how often the data source is 
     * queried for new values to be plotted. Values are averaged over this period.
     * The value is dependent on the plotting type constant provided. 
     * Possible constants for the plotting type and their associated values are 
     * <li>PERIOD_TYPE_TICKS - period value is in slider ticks</li>
     * <li>PERIOD_TYPE_MILLIS - period value is in milliseconds</li>
     * <li>PERIOD_TYPE_PIXELS - period value is in pixels</li> 
     * <li>PERIOD_TYPE_UNAVERAGED</li>
     * @param type the plotting period type constants
     * @param value the plotting period value
     */
	public void setPlottingPeriod(int type, double value);
    
    /**
     * Get the plotting period type constant for this graph. 
     * Possible constants for the plotting type and their associated values are 
     * <li>PERIOD_TYPE_TICKS - period value is in slider ticks</li>
     * <li>PERIOD_TYPE_MILLIS - period value is in milliseconds</li>
     * <li>PERIOD_TYPE_PIXELS - period value is in pixels</li> 
     * <li>PERIOD_TYPE_UNAVERAGED</li>
     * @return the plotting period type constant
     * @see #setPlottingPeriod(int, double)
     */	
    public int getPlottingPeriodType();
    
    /**
     * Get the plotting period value. 
     * @return the plotting period double value
     * @see #setPlottingPeriod(int, double)
     */ 
    public double getPlottingPeriodValue();
	
    /**
     * Set the X coordinate time offset. All data points from the data source are offset by 
     * this offset before being plotted. This is equivalent to transposing all data points along the x axis.
     * @param millis the offset in milliseconds
     */
	public void setXOffset(double millis);
    
    /**
     * Get the X coordinate time offset. All data points from the data source are offset by 
     * this offset before being plotted. This is equivalent to transposing all data points along the x axis.
     * @return the current x coordinate offset in milliseconds
     */    
	public double getXOffset();

    /**
     * Set the Y coordinate offset. All data points from the data source are offset by 
     * this offset before being plotted. This is equivalent to transposing all data points along the y axis.
     * @param offset the offset in milliseconds
     */	
	public void setYOffset(double offset);
    
	/**
     * Get the Y coordinate time offset. All data points from the data source are offset by 
     * this offset before being plotted. This is equivalent to transposing all data points along the y axis.
     * @return the current y coordinate offset
     */    
	public double getYOffset();
    
    /**
     * Set the indicator source to be used for plotting slider indicators for this graph.
     * @see org.eclipse.hyades.ui.widgets.grapher.IndicatorSource
     * @param isource the IndicatorSource implementation to be used for plotting the indicators of this graph
     */    	
	public void setIndicatorSource(IndicatorSource isource);
    
    /** 
     * Set the name for this graph. The graph name can be used by other graphs / graping UI to distinguish between graphs.
     * 
     * @param name a String name to be assigned to this graph
     */
    public void setName(String name);
    
    /** 
     * Get the name for this graph. The graph name can be used by other graphs / graping UI to distinguish between graphs.
     * 
     * @return String name for this graph
     */
    public String getName();
    
    /** 
     * Set the description for this graph. The graph description can be used by other graphs / graping UI 
     * to distinguish between graphs.
     * 
     * @param description String descirption to be assigned to this graph
     */    
    public void setDescription(String description);
    
    /** 
     * Get the description for this graph. The graph description can be used by other graphs / graping UI 
     * to distinguish between graphs.
     * 
     * @return String description for this graph
     */    
    public String getDescription();
    
    /** 
     * Set the miscellaneous string for this graph. The miscelaneous string is used to provide additional 
     * detail about this graph in addition to the name and description. The graph name can be used 
     * by other graphs or graping UI to distinguish between graphs.
     * 
     * @param misc String descirption to be assigned to this graph
     */  
    public void setMisc(String misc);
    
    /** 
     * Get a miscellaneous string for this graph. The miscelaneous string is used to provide additional 
     * detail about this graph in addition to the name and description. The graph name can be used 
     * by other graphs or graping UI to distinguish between graphs.
     * 
     * @return misc String for this graph
     */    
    public String getMisc();
    
    /** 
     * Set the graph canvas that this graph is to be rendered on. The Graph Canvas allows for graphs
     * to query the state of other graphs and allows for more interaction with the graphing UI.
     *      
     * @param graphCanvas the Graph Canvas that this graph is assigned to
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphCanvas
     */
    public void setGraphCanvas(GraphCanvas graphCanvas);
    
    /** 
     * Get the graph canvas that this graph is rendered on.
     * 
     * @return the graph's GraphCanvas
     * @see org.eclipse.hyades.ui.widgets.grapher.GraphCanvas
     * @see #setGraphCanvas(GraphCanvas gc)
     */
    public GraphCanvas getGraphCanvas();            
    
    /**
     * Dispose of all system resources that will not be garbage collected
     * that are associated with this graph. This method is called when a graph is removed from 
     * its parent graph canvas.     
     */
    public void dispose();

    /**
     * Set the delta value for the x axis of this graph. The delta differs
     * from the offset in that it is used to represent offsets that are not
     * only used on a visual basis. For instance, a delta might be used 
     * to represent a timezone change, where the offset is used to represent
     * only UI shifts.  
     */
	public void setXDelta(double millis);
    
    /**
     * Get the delta value for the x axis of this graph. The delta differs
     * from the offset in that it is used to represent offsets that are not
     * only used on a visual basis. For instance, a delta might be used 
     * to represent a timezone change, where the offset is used to represent
     * only UI shifts.  
     */
	public double getXDelta();
	
    /**
     * Set the delta value for the x axis of this graph. The delta differs
     * from the offset in that it is used to represent offsets that are not
     * only used on a visual basis. For instance, a delta might be used 
     * to represent a timezone change, where the offset is used to represent
     * only UI shifts.  
     */
	public void setYDelta(double millis);
    
    /**
     * Get the delta value for the x axis of this graph. The delta differs
     * from the offset in that it is used to represent offsets that are not
     * only used on a visual basis. For instance, a delta might be used 
     * to represent a timezone change, where the offset is used to represent
     * only UI shifts.  
     */
	public double getYDelta();

}