/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher;

import java.util.ArrayList;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;

/**
 * Graph Canvas is a canvas extension responsible for invoking the rendering and 
 * redrawing of Graphimplementations that have been assigned to it on the canvas.  
 * 
 * @author gchristelis
 * @since 4.0.0
 * @see org.eclipse.hyades.ui.widgets.grapher.Graph
 */
public class GraphCanvas extends Canvas implements PaintListener 
{
    private ArrayList graphs = new ArrayList();
    private ArrayList postGraphs = new ArrayList();
    private ArrayList preGraphs = new ArrayList();

    private static final boolean PAINT_BUFFERED = true;
    
    private int buffer_w = -1;
    private int buffer_h = -1;
    private Image buffer = null;
    private GC buffer_gc = null;

    private int xoffset, yoffset;

    private Color bgcol;

    private GraphHighlighter highlighter = null;
    private GraphSelectionSource selection_source = null;

    /**
     * Graph Canvas constructor. The style constants are the same as those for the Canvas widget except
     * that the Graph Canvas enforces the use of the <code>SWT.NO_BACKGROUND</code> constant.
     * 
     * @param parent the parent Composite of this canvas
     * @param style the style constant for this canvas. 
     * @see org.eclipse.swt.widgets.Canvas#Canvas(org.eclipse.swt.widgets.Composite, int)
     */
	public GraphCanvas(Composite parent, int style) 
    {
		super(parent,style|SWT.NO_BACKGROUND);
		addPaintListener(this);

		setBackground(getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND));
	}
	
    /**
     * Add an implementation of the <code>Graph</code> interface to the list of graphs to be rendered
     * on this canvas. This method ensures that the graph will be drawn before any post graphs 
     * and after and pre graphs in this canvas. 
     * 
     * @param graph the <code>Graph</code> implementation to be added
     */
	public void addGraph(Graph graph) 
    {
		if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(true);
		graphs.add(graph);
        graph.setGraphCanvas(this);
	}
    
	/**
     * Add an implementation of the <code>Graph</code> interface to the list of graphs to be rendered
     * last on this canvas. This method ensures that the graph will be drawn after any post graphs 
     * and after any standard graphs in this canvas. 
     * 
     * @param graph the <code>Graph</code> implementation to be added
     */    
    public void addPostGraph(Graph graph)
    {
        if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(true);
        postGraphs.add(graph);
        graph.setGraphCanvas(this);
    }   
        
    /**
     * Add an implementation of the <code>Graph</code> interface to the list of graphs to be rendered
     * first on this canvas. This method ensures that the graph is added to a list of graphs that will be drawn
     * before any of the graph or postGraph graphs.
     * 
     * @param graph the <code>Graph</code> implementation to be added
     */        
    public void addPreGraph(Graph graph)
    {
        if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(true);
        preGraphs.add(graph);
        graph.setGraphCanvas(this);
    }
    
    /**
     * Remove a graph implementation from the post graph list.
     * @param graph the <code>Graph</code> implementation to remove from the list
     */
    public void removePostGraph(Graph graph)
    {
        if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(true);
        postGraphs.remove(graph);
    }
    
    /**
     * Remove a graph implementation from the pre graph list.
     * @param graph the <code>Graph</code> implementation to remove from the list
     */    
    public void removePreGraph(Graph graph)
    {
        if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(true);
        preGraphs.remove(graph);
    }
	
    /**
     * Remove a graph implementation from the standard graph list.
     * @param graph the <code>Graph</code> implementation to remove from the list
     */
	public void removeGraph(Graph graph) 
    {
		if (graph instanceof BasicGraph) 
            ((BasicGraph)graph).setShown(false);
		graphs.remove(graph);
	}
	
    /**
     * Remove all pre/standard/post graphs from the canvas.     
     */
	public void removeAllGraphs() 
    {		
		for (int i = 0; i < graphs.size(); i++) 
        {
			Graph g = (Graph)graphs.get(i);
			if (g instanceof BasicGraph) 
                ((BasicGraph)g).setShown(false);
		}
        for (int i = 0; i < preGraphs.size(); i++) 
        {
            Graph g = (Graph)preGraphs.get(i);
            if (g instanceof BasicGraph) 
                ((BasicGraph)g).setShown(false);
        }       
        for (int i = 0; i < postGraphs.size(); i++) 
        {
            Graph g = (Graph)postGraphs.get(i);
            if (g instanceof BasicGraph) 
                ((BasicGraph)g).setShown(false);
        }       
        postGraphs.clear();
		graphs.clear();    
        preGraphs.clear();        
	}
	
    /** 
     * Dispose of the resources used by this canvas.
     */
	public void dispose() 
    {
		super.dispose();
		if (buffer != null) 
        {
			buffer.dispose();
			buffer_gc.dispose();	
			if (bgcol != null) bgcol.dispose();
		}	
	}
	
    /**
     * Set the background canvas color
     * @param rgb the background color to set
     */
	public void setBackground(RGB rgb) 
    {
		if (bgcol != null) 
            bgcol.dispose();
		bgcol = new Color(getDisplay(),rgb);
		super.setBackground(bgcol);	
	}
	
    /**
     * Set the instance of <code>GraphHighlighter</code> to use on this canvas. The
     * GraphHighlighter 
     * @param highlighter the GraphHighlighter instance to use on this canvas. 
     */
	public void setGraphHighlighter(GraphHighlighter highlighter) 
    {
		this.highlighter = highlighter;
	}
	
    /**
     * Set the GraphSelectionSource implementation to use on this canvas. The selection
     * source defines which graph is currently selected (if any).
     * 
     * @param source the instance of <code>GraphSelectionSource</code> to use as a selection source
     */
	public void setGraphSelectionSource(GraphSelectionSource source) 
    {
		selection_source = source;	
	}    
    
    /**
     * Get the current drawing offsets to use when rendering the graphs on the canvas. In certain 
     * situations it is possible that the GraphCanvas implementation should not use the entire canvas
     * for rendering. If the entire canvas should not be used then this method should be overridden.
     * 
     * @return a Point with the x and y offsets
     */
	public Point getOffsets() 
    {
        return new Point(xoffset,yoffset);
	}
    
    /**
     * Set the current drawing offsets
     * @param a Point with the x and y offsets
     */
    public void setOffsets(Point newOffsets)
    {
        xoffset = newOffsets.x;
        yoffset = newOffsets.y;
    }
	
	/**
     * The PaintListener paintControl method. Paint the buffered gc onto the canvas.
     * @param e the PaintEvent
	 */	
	public void paintControl(PaintEvent e) 
    {
		GC gc = e.gc;
		int x = e.x;
		int y = e.y;
		int w = e.width;
		int h = e.height;

		if (w < 1 || h < 1) return;

		if (PAINT_BUFFERED) {

			if ((buffer_w < w || buffer_h < h) && buffer != null) {
				buffer.dispose();
				buffer_gc.dispose();
				buffer = null;
				buffer_gc = null;
			}

			if (buffer == null) {
				if (getDisplay() == null) return; 
				buffer_w = Math.max(buffer_w,w);
				buffer_h = Math.max(buffer_h,h);
				buffer = new Image(getDisplay(),buffer_w,buffer_h);
				buffer_gc = new GC(buffer); 	
			}
			
			paintUnbuffered(buffer_gc,0,0,w,h);

			gc.drawImage(buffer,0,0);
			
		} else {
			paintUnbuffered(gc,x,y,w,h);
		}
	}

	/**
     * Render the graphs in this canvas on the GC within the given bounds.
     * 
     * @param gc the GC to paint onto
     * @param x the start x coordinate
     * @param y the start y coordinate
     * @param w the width of the drawing bound
     * @param h the height of the drawing bound
     */
	public void paintUnbuffered(GC gc, int x, int y, int w, int h) 
    {
		
		Graph highlighted = null;
		if (highlighter != null && selection_source != null) {
			highlighted = selection_source.getSelectedGraph();	
		}

		//draw background
		Color c = getBackground();
		gc.setBackground(c);
		gc.fillRectangle(x,y,w+xoffset,h+yoffset);
		gc.fillRectangle(0,0,10000,10000);
		
		int textoffset = 0;        
		
        for (int i=0; i< preGraphs.size(); i++)
        {
            Graph g = (Graph)preGraphs.get(i);
            g.paintGraph(gc,x+xoffset,y+yoffset,w,h);
        }
        
		//draw graphs
		for (int i = 0; i < graphs.size(); i++) 
        {			
			Graph g = (Graph)graphs.get(i);
			
			if (g == highlighted) {
				highlighter.setHighlight(g);	
			}

			if (g instanceof TextGraph) {
				textoffset = ((TextGraph)g).paintGraph(gc,x+xoffset,y+yoffset,w,h,textoffset);
			} else {	
				g.paintGraph(gc,x+xoffset,y+yoffset,w,h);
			}	

			if (g == highlighted) {
				highlighter.unsetHighlight(g);	
			}
		}
        for (int i=0; i< postGraphs.size(); i++)
        {
            Graph g = (Graph)postGraphs.get(i);
            g.paintGraph(gc,x+xoffset,y+yoffset,w,h);
        }
	}

    /**
     * Get the standard graph list
     * @return an ArrayList of Graph implementations
     */
    public ArrayList getGraphs()
    {
        return graphs;
    }

    /**
     * Get the pre order graph list
     * @return an ArrayList of Graph implementations
     */
    public ArrayList getPreGraphs()
    {
        return preGraphs;
    }
    
    /**
     * Get the post order graph list
     * @return an ArrayList of Graph implementations
     */    
    public ArrayList getPostGraphs()
    {
        return postGraphs;
    }
}