/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher;

/**
 * The VisiblePointsCache keeps a cache of all the currently visible data source
 * points in the graph canvas for this graph. Commonly, the graph canvas will only
 * show a small subset of the data points in the graph source. When rendering its
 * visible data points, a graph can populate the visible points cache so that a 
 * reference is kept as to exactly which points were used.
 * 
 * @author gchristelis
 * @since 4.0.0
 *
 */
public class VisiblePointsCache
{
    
    private double [] x_values;   
    private double [] y_values;
    // Bug 98615
    private boolean [] is_valid;
    // End
    private int pointLen;    
    private int length = 0;    
    private Graph graph;
    
    /**
     * Visible Points Cache constructor. Initializes the cache size and stores a 
     * reference to the current Graph implementation
     * 
     * @param graph the Graph implementation which populates this cache
     */
    public VisiblePointsCache(Graph graph)
    {
        pointLen = 100;
        x_values = new double[pointLen];
        y_values = new double[pointLen];
        is_valid = new boolean[pointLen];
        this.graph = graph;
    }
    
    /**
     * Get the current number of valid points in the point cache
     * @return the number of valid points in the point cache 
     */
    public int getLength()
    {
        return length;
    }
    
    /**
     * Get the current Graph implemetation that is populating this points cache
     * @return the Graph implementation
     */
    public Graph getGraph()
    {
        return graph;
    }
    
    private void doubleBuffer()
    {
        pointLen *= 2;
        double [] new_x = new double[pointLen];
        double [] new_y = new double[pointLen];
                
        System.arraycopy(x_values,0,new_x,0,length);
        System.arraycopy(y_values,0,new_y,0,length);
        
        x_values = new_x;
        y_values = new_y;
        
        //Bug 98615
        boolean [] new_valid = new boolean[pointLen];
        System.arraycopy(is_valid, 0,new_valid,0,length);
        is_valid = new_valid;        
        //End
    }
    
    /**
     * Get the x values of valid points in this points cache. The array size must
     * be computed through a call to <code>getLength</code> as the buffer array might
     * not be full
     * 
     * @return a double array of x values
     */
    public double[] getXValues()
    {
        return x_values;
    }

    /**
     * Get the y values of valid points in this points cache. The array size must
     * be computed through a call to <code>getLength</code> as the buffer array might
     * not be full
     * 
     * @return a double array of y values
     */
    public double[] getYValues()
    {
        return y_values;
    }
    
    /**
     * Returns an array of booleans, depicting whether data points are valid or not.
     * @return a boolean array
     */
    public boolean[] getValidValues()
    {
        return is_valid;
    }
    
    /**
     * Reset the buffer array. The number of valid points in the cache is set to 0.     
     */
    public void reset()
    {
        length = 0;
    }

    /**
     * Add an (x,y) coordinate to the points cache. The point is assumed to be a valid value
     * @param x the x coordinate value
     * @param y the y coordinate value
     */
    public void addPoint(double x, double y)
    {
        addPoint(x,y,true);
    }
    
    /**
     * Add an (x,y) coordinate to the points cache, and specify whether the point is a valid point. An invalid point is not part of the data, but drawn by the graphers to facilitate the interpretation of the data. 
     * @param x the x coordinate value
     * @param y the y coordinate value
     * @param isValid a boolean of true if the point is valid, and false otherwise
     */
    public void addPoint(double x, double y, boolean isValid)
    {
        if (length == pointLen)
        {
            doubleBuffer();
        }
        x_values[length] = x;
        y_values[length] = y;
        is_valid[length] = isValid;
        length++;
    }
}