/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher.graphs;

import java.util.ArrayList;
import java.util.Date;

import org.eclipse.hyades.ui.widgets.grapher.BasicGraph;
import org.eclipse.hyades.ui.widgets.grapher.BasicGraphSource;
import org.eclipse.hyades.ui.widgets.grapher.Graph;
import org.eclipse.hyades.ui.widgets.grapher.GraphCanvas;
import org.eclipse.hyades.ui.widgets.grapher.IndicatorSource;
import org.eclipse.hyades.ui.widgets.grapher.VisiblePointsCache;
import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.text.DecimalFormat;
import com.ibm.icu.text.NumberFormat;

/**
 * The HighlighterGraph is a graph and graph source implementation that highlights
 * data points on line graphs that are currently rendered on the Graph Canvas. The graph 
 * is set active on a held down left click on the canvas, and displays summary information
 * about the nearest data point and its associated graph.
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class HighlighterGraph extends BasicGraph implements Graph, BasicGraphSource, MouseListener, MouseMoveListener
{
    private boolean highlight = false;
    private int currentx, currenty;
    private double nearestx;
    private double nearesty;
    private boolean nearestvalid;
    private double truex;
    private double truey;
    private DecimalFormat decFormat;
            
    private GraphCanvas gw;
    private TimeZoomSlider xslider;
    private ZoomSlider yslider;
    
    private ArrayList disposables = new ArrayList();
    private static Color WHITE = null;
    
    /**
     * The HighligtherGraph constructor
     */
    public HighlighterGraph()
    {
        getDecFormat();
    }
    
    private void getDecFormat()
    {
        NumberFormat format = DecimalFormat.getNumberInstance();
        if (format instanceof DecimalFormat)
            decFormat = (DecimalFormat) format;
        decFormat.setMaximumFractionDigits(4);   
    }
    
    /**
     * Set the active status of the highlighter graph
     * @param b the active state of the highlighter graph
     */
    protected void setActive(boolean b)
    {
        highlight = b;
    }
    
    /**
     * Is the highlighter graph is active (visible and processing)
     * @return whether the highlighter graph is active 
     */
    protected boolean getActive()
    {
        return highlight;
    }
    
    /**
     * Set the new coordinates of the mouse on the canvas
     * @param x the new x coordinate
     * @param y the new y coordinate
     */
    public void setCoordinates(int x, int y)
    {
        this.currentx = x;
        this.currenty = y;
    }
    
/*    /**
     * The HighlighterGraph constructor.
     * @param GraphCanvas to render the highlighter graph on
     */
    /*
    public HighlighterGraph(GraphCanvas graphCanvas)
    {
        this.gw = graphCanvas;        
    }*/
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphSource()
     */
    public BasicGraphSource getGraphSource()
    {
        return this;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMin()
     */
    public double getXMin()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMax()
     */
    public double getXMax()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMin()
     */
    public double getYMin()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMax()
     */
    public double getYMax()
    {
        return 0;
    }
    
    /**
     * The rounded rectangle corner radius
     */
    protected int radius = 10;
    
    /**
     * The target color shade
     */
    protected int target = 130;
    
    private Color adjust(Color c)
    {
        int max = c.getRed();
        max = Math.max(c.getBlue(),max);
        max = Math.max(c.getGreen(),max);
        
        float multi = ((float) (target))/((float) max);
        return adjust(c, multi);
    }
    
    private Color adjust(Color c, float multi)
    {
        int r = (int) (c.getRed() * multi);
        int b = (int) (c.getBlue() * multi);
        int g = (int) (c.getGreen() * multi);
        Color newCol = new Color(gw.getDisplay(),r,g,b);
        return newCol;
    }
    
/*    private Color adjust(Color c, int diff)
    {
        int r = c.getRed() + diff;
        if (r < 0)
            r = 0;
        else if (r > 255)
            r = 255;
                            
        int b = c.getBlue() + diff;;
        if (b < 0)
            b = 0;
        else if (b > 255)
            b = 255;
        
        int g = c.getGreen() + diff;;
        if (g < 0)
            g = 0;
        else if (g > 255)
            g = 255;
        
        Color newCol = new Color(gw.getDisplay(),r,g,b);
        return newCol;
    }
*/    
    private static final int roundBuffer = 3;
    private static final int minRadius = 6;
    
    private Font setBold(boolean bold, GC gc)
    {
        FontData [] data = gc.getFont().getFontData();
        for (int i=0; i<data.length; i++)
        {
            int style = data[i].getStyle();
            if (bold)
            {                
                data[i].setStyle(style|SWT.BOLD);
            }
            else
            {
                data[i].setStyle(style&(~SWT.BOLD));
            }
        }
        Font f = new Font(gw.getDisplay(),data);
        gc.setFont(f);
        return f;
    }
    
/*    private void setAlpha(GC gc, int alpha)
    {
        try
        {
            gc.setAlpha(alpha);
        }
        catch (Exception e)
        {
            e.printStackTrace();
        }
    }
*/    
    // The buffer space between the top and bottom of the exmaple line.
    private static final int LINE_BUFF = 3;
    private static final int LINE_SIDE_BUF = 10;
    private static final int ARC_HEIGHT = 8;
    private static final int ARC_WIDTH = 8;    
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#paintGraph(org.eclipse.swt.graphics.GC, int, int, int, int)
     */
    public void paintGraph(GC gc, int x, int y, int w, int h)
    {
        if (highlight)
        {
            if (decFormat == null)
                getDecFormat();
            
            ArrayList graphs = gw.getGraphs();
            LineGraph g = locateNearestGraph(graphs,this.currentx, this.currenty);
                             
            if (g == null)
                return;
            
            int width = gw.getSize().x;
            int height = gw.getSize().y;
            truex = g.getXSlider().pixel2Value(nearestx);
            String valString;    
            // Bug 98610
            if (nearestvalid)
            {
                truey = g.getYSlider().pixel2Value(nearesty-15); // -15 to take the ^ button into account
                valString = decFormat.format(truey);
            }
            else
            {
                truey = Double.NaN;
                valString = ""+truey;
            }
            // End Bug 98610
 
            // Calculate the date to display
            Date date = new Date((long)(truex));            
            DateFormat dateFormat = DateFormat.getDateTimeInstance();
            String dateString = " ("+dateFormat.format(date)+")";
            
            String nameString = g.getName();
            String miscString = g.getMisc();
            
            String misc = miscString;
            
            Font f = setBold(true,gc);
            Point valExtent = gc.textExtent(valString,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            Font f2 = setBold(false,gc);            
            Point nameExtent = gc.textExtent(nameString,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            Point dateExtent = gc.textExtent(dateString,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            Point miscExtent = gc.textExtent(misc,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            Point lineExtent = new Point(0,2*LINE_BUFF+g.getLineWidth());           
            
            Point extent = new Point(Math.max(Math.max(valExtent.x+dateExtent.x,miscExtent.x),nameExtent.x),Math.max(nameExtent.y,dateExtent.y)+miscExtent.y+nameExtent.y+lineExtent.y);

            int drawx = (int) (nearestx) + 2*radius;
            int drawy = (int) (nearesty) - extent.y - 2*radius;
            
            if (drawx + extent.x > width)
                drawx = Math.max(0,drawx-extent.x-4*radius);            
            if (drawy < 0)
                drawy = Math.min(height-extent.y,drawy+extent.y+4*radius);
            
            radius = (int) (2*g.getLineWidth());
            if (radius < minRadius)
                radius = minRadius;
            int linewidth = g.getLineWidth();
                
            Color newCol = adjust(g.getForeground());
            gc.setForeground(newCol);
            
            gc.setBackground(WHITE);
            gc.setLineStyle(SWT.LINE_SOLID);            
            
            int tempLineWidth = 2;
            gc.setLineWidth(tempLineWidth);
            gc.drawRoundRectangle(drawx-roundBuffer,drawy-roundBuffer,extent.x+2*roundBuffer,extent.y+2*roundBuffer,ARC_HEIGHT,ARC_WIDTH);
            // Not sure why this only works for 1 and not 2 considering 2 is the width...?- rather odd.
            tempLineWidth=1;
            gc.fillRoundRectangle(drawx-roundBuffer+tempLineWidth,drawy-roundBuffer+tempLineWidth,extent.x+2*roundBuffer-2*tempLineWidth,extent.y+2*roundBuffer-2*tempLineWidth,ARC_HEIGHT,ARC_WIDTH);                       
            gc.setLineWidth(Math.max(linewidth-1,1));            
                        
            gc.drawOval((int)(nearestx-radius),(int)(nearesty-radius),2*radius,2*radius);
            setBold(true,gc);
            gc.drawText(valString,drawx,drawy,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            setBold(false,gc);            
            gc.drawText(dateString,drawx+valExtent.x,drawy,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            gc.drawText(nameString,drawx,drawy+Math.max(valExtent.y,dateExtent.y),SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
                        
            gc.drawText(misc,drawx,drawy+Math.max(valExtent.y,dateExtent.y)+nameExtent.y+lineExtent.y,SWT.DRAW_DELIMITER|SWT.DRAW_TRANSPARENT);
            
            gc.drawLine(0,(int) (nearesty),(int) (nearestx-radius),(int)(nearesty));
            gc.drawLine((int) (nearestx),(int) (nearesty+radius),(int) (nearestx),height);

            // Draw the example line here
            gc.setLineStyle(g.getLineStyle());
            gc.setLineWidth(g.getLineWidth());
            gc.setForeground(g.getForeground());
            
            gc.drawLine(drawx+LINE_SIDE_BUF,drawy+Math.max(valExtent.y,dateExtent.y)+(lineExtent.y/2)+nameExtent.y,drawx+extent.x-LINE_SIDE_BUF,drawy+Math.max(valExtent.y,dateExtent.y)+(lineExtent.y/2)+nameExtent.y);
                        
            gc.setForeground(g.getForeground());
            gc.setLineStyle(g.getLineStyle());
            gc.setLineWidth(g.getLineWidth());
            
            f.dispose();
            f2.dispose();
            newCol.dispose();
        }        
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXSlider(org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider)
     */
    public void setXSlider(TimeZoomSlider slider)
    {
        xslider = slider;        
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYSlider(org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider)
     */
    public void setYSlider(ZoomSlider slider)
    {
        yslider = slider;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXSlider()
     */
    public TimeZoomSlider getXSlider()
    {
        return xslider;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYSlider()
     */
    public ZoomSlider getYSlider()
    {
        return yslider;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setForeground(org.eclipse.swt.graphics.Color)
     */
    public void setForeground(Color col)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getForeground()
     */
    public Color getForeground()
    {
        return null;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineWidth(int)
     */
    public void setLineWidth(int w)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineWidth()
     */
    public int getLineWidth()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineStyle(int)
     */
    public void setLineStyle(int swt_line_style)
    {        
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineStyle()
     */
    public int getLineStyle()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setStaticScaling(double)
     */
    public void setStaticScaling(double mult)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getStaticScaling()
     */
    public double getStaticScaling()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingType()
     */
    public int getPlottingType()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingType(int)
     */
    public void setPlottingType(int type)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getNoDataBehaviour()
     */
    public int getNoDataBehaviour()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setNoDataBehaviour(int)
     */
    public void setNoDataBehaviour(int behaviour)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingPeriod(int, double)
     */
    public void setPlottingPeriod(int type, double value)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodType()
     */
    public int getPlottingPeriodType()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodValue()
     */
    public double getPlottingPeriodValue()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXOffset(double)
     */
    public void setXOffset(double millis)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXOffset()
     */
    public double getXOffset()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYOffset(double)
     */
    public void setYOffset(double millis)
    {
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYOffset()
     */
    public double getYOffset()
    {
        return 0;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setIndicatorSource(org.eclipse.hyades.ui.widgets.grapher.IndicatorSource)
     */
    public void setIndicatorSource(IndicatorSource isource)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.BasicGraphSource#getMin()
     */
    public double getMin()
    {
        return 0;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.BasicGraphSource#getMax()
     */
    public double getMax()
    {
        return 0;
    }
    
/*    private static int binarySearch(double[] a, double key, int low,int high) 
    {
        while (low <= high) {
            int mid = (low + high) >> 1;
            double midVal = a[mid];

                int cmp;
                if (midVal < key) {
                    cmp = -1;   // Neither val is NaN, thisVal is smaller
                } else if (midVal > key) {
                    cmp = 1;    // Neither val is NaN, thisVal is larger
                } else {
                    long midBits = Double.doubleToLongBits(midVal);
                    long keyBits = Double.doubleToLongBits(key);
                    cmp = (midBits == keyBits ?  0 : // Values are equal
                           (midBits < keyBits ? -1 : // (-0.0, 0.0) or (!NaN, NaN)
                            1));                     // (0.0, -0.0) or (NaN, !NaN)
                }

            if (cmp < 0)
            low = mid + 1;
            else if (cmp > 0)
            high = mid - 1;
            else
            return mid; // key found
        }
        return -(low + 1);  // key not found.
        }*/
        
    private LineGraph locateNearestGraph(ArrayList graphList, int mousex, int mousey)
    {
        LineGraph nearest = null;
        double dist = -1;
        double [] xvalues;
        double [] yvalues;
        boolean [] isvalid;
        double newdist,x,y;
        boolean valid;
                
        for (int i=0; i<graphList.size(); i++)
        {
            Object obj = graphList.get(i);
            if (obj instanceof LineGraph)
            {
                LineGraph tempGraph = (LineGraph) obj;
                VisiblePointsCache pc = tempGraph.getPointsCache();           
                xvalues = pc.getXValues(); 
                yvalues = pc.getYValues();
                isvalid = pc.getValidValues();
                
                for (int index = 0; index < pc.getLength(); index++)
                {
                    x = xvalues[index];
                    y = yvalues[index];
                    valid = isvalid[index];
                    
                    newdist = (mousex-x)*(mousex-x) + (mousey-y)*(mousey-y);                        
                    if (dist == -1 || newdist < dist)
                    {
                        nearestx = x;
                        nearesty = y;
                        nearestvalid = valid;
                        nearest = tempGraph;
                        dist = newdist;
                    }                    
                }
            }
        }
        return nearest;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setName(java.lang.String)
     */
    public void setName(String name)
    { 
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getName()
     */
    public String getName()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setDescription(java.lang.String)
     */
    public void setDescription(String description)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getDescription()
     */
    public String getDescription()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setMisc(java.lang.String)
     */
    public void setMisc(String misc)
    {

    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getMisc()
     */
    public String getMisc()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setGraphCanvas(org.eclipse.hyades.ui.widgets.grapher.GraphCanvas)
     */
    public void setGraphCanvas(GraphCanvas gc)
    {
        gw = gc;
        
        if (WHITE == null)
        {
            WHITE = new Color(gw.getDisplay(),255,255,255);
            disposables.add(WHITE);
        }
        
        if (!gw.isDisposed()) 
            gw.addMouseListener(this);
        if (!gw.isDisposed())        
            gw.addMouseMoveListener(this);
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphCanvas()
     */
    public GraphCanvas getGraphCanvas()
    {
        return gw;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.swt.events.MouseListener#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
     */
    public void mouseDoubleClick(MouseEvent e)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
     */
    public void mouseDown(MouseEvent e)
    {
        // We should only show the highlighter on left clicks
        if (e.button == 1)
        {
            setActive(true);
            setCoordinates(e.x, e.y);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.events.MouseListener#mouseUp(org.eclipse.swt.events.MouseEvent)
     */
    public void mouseUp(MouseEvent e)
    {
        // We should only show the highlighter on left clicks
        if (e.button == 1)
        {
            setActive(false);
        }
    }

    /* (non-Javadoc)
     * @see org.eclipse.swt.events.MouseMoveListener#mouseMove(org.eclipse.swt.events.MouseEvent)
     */
    public void mouseMove(MouseEvent e)
    {
        setCoordinates(e.x, e.y);
        if (getActive())
            gw.redraw();
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#dispose()
     */
    public void dispose()
    {
        for (int i=0; i<disposables.size(); i++)
        {
            Object o =disposables.get(i); 
            if (o instanceof Color)
            {
                ((Color) o).dispose();
            }
        }        
    }

	public void setXDelta(double millis) {
		// TODO Auto-generated method stub
		
	}

	public double getXDelta() {
		// TODO Auto-generated method stub
		return 0;
	}

	public void setYDelta(double millis) {
		// TODO Auto-generated method stub
		
	}

	public double getYDelta() {
		// TODO Auto-generated method stub
		return 0;
	}
    
}