/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher.graphs;

import java.util.Date;

import org.eclipse.hyades.ui.widgets.grapher.BasicGraphSource;
import org.eclipse.hyades.ui.widgets.grapher.Graph;
import org.eclipse.hyades.ui.widgets.grapher.GraphCanvas;
import org.eclipse.hyades.ui.widgets.grapher.IndicatorSource;
import org.eclipse.hyades.ui.widgets.grapher.TextGraph;
import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;

import com.ibm.icu.text.DateFormat;

/**
 * An implementation of textgraph and indicator source. The IndicatorLineGraph
 * draws a vertical line at the current indicator location, and displays the current
 * time next to the line.
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class IndicatorLineGraph implements TextGraph, IndicatorSource 
{	
    // The indicator source type    
	private int type = TYPE_TIME;
    // The slider configuration
    private TimeZoomSlider xslider;
    private ZoomSlider yslider;	
    
    private Color black_color;	
    private Color fg_col;
    
    private int line_width = 1;
    private int line_style = SWT.LINE_SOLID;
    
    private static DateFormat df = DateFormat.getDateInstance(DateFormat.LONG);
    private static DateFormat tf = DateFormat.getTimeInstance();    
    
    private double source_val = Double.MAX_VALUE;
		
	/**
	 * Constructor for an IndicatorLineGraph to graph an indicator line on the canvas in the provided color.
	 * @param xslider the associated x slider
	 * @param yslider the associated y slider
	 * @param col the foreground line color
	 */
	public IndicatorLineGraph(TimeZoomSlider xslider, ZoomSlider yslider, Color col) {
		this.xslider = xslider;
		this.yslider = yslider;
		
		this.fg_col = col;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphSource()
	 */
	public BasicGraphSource getGraphSource() 
    {
		return null;	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMin()
	 */
	public double getXMin() 
    {
		return Double.NEGATIVE_INFINITY;		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMax()
	 */
	public double getXMax() 
    {
		return Double.POSITIVE_INFINITY;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMin()
	 */
	public double getYMin() 
    {
		return Double.NEGATIVE_INFINITY;		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMax()
	 */
	public double getYMax() 
    {
		return Double.POSITIVE_INFINITY;		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXSlider()
	 */
	public TimeZoomSlider getXSlider() {
		return xslider;	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYSlider()
	 */
	public ZoomSlider getYSlider() 
    {
		return yslider;	
	}	
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXSlider(org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider)
	 */
	public void setXSlider(TimeZoomSlider slider) 
    {
		xslider = slider;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYSlider(org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider)
	 */
	public void setYSlider(ZoomSlider slider) 
    {
		yslider = slider;
	}	
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setForeground(org.eclipse.swt.graphics.Color)
	 */
	public void setForeground(Color col) 
    {
		fg_col = col;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getForeground()
	 */
	public Color getForeground() 
    {
		return fg_col;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineWidth(int)
	 */
	public void setLineWidth(int w) 
    {
		line_width = w;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineWidth()
	 */
	public int getLineWidth() 
    {
		return line_width;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineStyle(int)
	 */
	public void setLineStyle(int swt_line_style) 
    {
		line_style = swt_line_style;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineStyle()
	 */
	public int getLineStyle() 
    {
		return line_style;	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.IndicatorSource#getSourceType()
	 */
	public int getSourceType() 
    {
		return type;
	}

	/**
     * Set the current indicator source type
	 * @param type the indicator source type constant 
     * @see org.eclipse.hyades.ui.widgets.grapher.IndicatorSource
	 */
	public void setSourceType(int type) 
    {
		this.type = type;	
	}

	/**
	 * The the current indicator source location to the provided time in milliseconds.
     * @param time the time to set the indicator source location to
	 */
	public void setIndicatorLocation(double time) 
    {
		source_val = time;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#paintGraph(org.eclipse.swt.graphics.GC, int, int, int, int)
	 */
	public void paintGraph(GC gc, int x, int y, int w, int h) 
    {
		paintGraph(gc,x,y,w,h,0);
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.TextGraph#paintGraph(org.eclipse.swt.graphics.GC, int, int, int, int, int)
	 */
	public int paintGraph(GC gc, int x, int y, int w, int h, int yoffset) 
    {

		int fheight = gc.getFontMetrics().getHeight();

		gc.setForeground(fg_col);
		gc.setLineWidth(line_width);	
		gc.setLineStyle(line_style);

		int xval = (int)source_val;
		long t = (long)xslider.pixel2Value(xval);
		if (type == TYPE_TIME) {
			xval = (int)xslider.value2Pixel(source_val);
			t = (long)source_val;
		}

		//simple bounds check
		if (xval < w) 
        {
		
			gc.drawLine(xval,y,xval,y+h);
	
			//SimpleDateFormat sdf = new SimpleDateFormat("HH:mm:ss");
			Date d = new Date(t);
		    
	
			String dateString = df.format(d);
			String timeString = tf.format(d);

			int dateExtent = gc.textExtent(dateString).y;
			
			gc.setForeground(yslider.getDisplay().getSystemColor(SWT.COLOR_GRAY));
			gc.drawString(dateString.toString(),xval+4,yoffset + y+1,true);
			gc.drawString(timeString.toString(),xval+4,yoffset + y+1 + dateExtent,true);			

			try 
            {
				gc.setForeground(yslider.getDisplay().getSystemColor(SWT.COLOR_BLACK));
			} 
            catch (Throwable xx) 
            {
				if (black_color == null) 
                { 
				    black_color = new Color(yslider.getDisplay(),0,0,0);
				}
				gc.setForeground(black_color);
			}
			gc.drawString(dateString.toString(),xval+3,yoffset + y,true);
			gc.drawString(timeString.toString(),xval+3,yoffset + y + dateExtent,true);			
	
			yoffset += fheight;
		
			if ((yoffset + fheight) > h) 
            {
				yoffset = 0;
			}	
		}

		return yoffset;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setStaticScaling(double)
	 */
	public void setStaticScaling(double mult) 
    {
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getStaticScaling()
	 */
	public double getStaticScaling() 
    {
		return 1.0;
	} 
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingType()
	 */
	public int getPlottingType() 
    {
		return Graph.PLOTTING_TYPE_AVERAGE;
	}
        
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingType(int)
	 */
	public void setPlottingType(int type) 
    {
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingPeriod(int, double)
	 */
	public void setPlottingPeriod(int type, double value) 
    {
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodType()
	 */
	public int getPlottingPeriodType() 
    {
		return Graph.PERIOD_TYPE_TICKS;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodValue()
	 */
	public double getPlottingPeriodValue() {
		return 1.0;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXOffset(double)
	 */
	public void setXOffset(double millis) 
    {
	}
    
	public double getXOffset() 
    {
		return 0;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYOffset(double)
	 */
	public void setYOffset(double millis) 
    {
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYOffset()
	 */
	public double getYOffset() 
    {
		return 0;
	}	
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getNoDataBehaviour()
	 */
	public int getNoDataBehaviour() 
    {
		return Graph.NODATA_DO_NOTHING;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setNoDataBehaviour(int)
	 */
	public void setNoDataBehaviour(int i) 
    {
	}
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setName(java.lang.String)
     */
    public void setName(String name)
    { 
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getName()
     */
    public String getName()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setDescription(java.lang.String)
     */
    public void setDescription(String description)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getDescription()
     */
    public String getDescription()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setMisc(java.lang.String)
     */
    public void setMisc(String misc)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getMisc()
     */
    public String getMisc()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setGraphCanvas(org.eclipse.hyades.ui.widgets.grapher.GraphCanvas)
     */
    public void setGraphCanvas(GraphCanvas gc)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphCanvas()
     */
    public GraphCanvas getGraphCanvas()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#dispose()
     */
    public void dispose()
    {        
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.IndicatorSource#getIndicatorLocation()
     */
    public double getIndicatorLocation()     
    {
        return source_val;  
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setIndicatorSource(org.eclipse.hyades.ui.widgets.grapher.IndicatorSource)
     */
    public void setIndicatorSource(IndicatorSource isource) 
    {
    }

	public void setXDelta(double millis) {
		// TODO Auto-generated method stub
		
	}

	public double getXDelta() {
		// TODO Auto-generated method stub
		return 0;
	}

	public void setYDelta(double millis) {
		// TODO Auto-generated method stub
		
	}

	public double getYDelta() {
		// TODO Auto-generated method stub
		return 0;
	}
}