/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher.graphs;

import org.eclipse.hyades.ui.widgets.grapher.BasicGraph;
import org.eclipse.hyades.ui.widgets.grapher.BasicGraphSource;
import org.eclipse.hyades.ui.widgets.grapher.Graph;
import org.eclipse.hyades.ui.widgets.grapher.GraphCanvas;
import org.eclipse.hyades.ui.widgets.grapher.GraphSource;
import org.eclipse.hyades.ui.widgets.grapher.IndicatorSource;
import org.eclipse.hyades.ui.widgets.grapher.VisiblePointsCache;
import org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomIndicator;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider;
import org.eclipse.hyades.ui.widgets.zoomslider.ZoomSliderTick;
import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;


/**
 * A Graph implementation used to draw line graphs on a Graph Canvas. Data points are joined by straight lines
 * and the line style, width, plotting period and type are all defined through the Graph interface.
 * 
 * @author gchristelis
 * @since 4.0.0
 * 
 */
public class LineGraph extends BasicGraph implements Graph 
{
    private static final boolean DRAW_ONLY_IF_DATA = true;

    	
    private static final int PLOTTING_TYPE_STDDEV_AVERAGE_PLUS = -9990;
    private static final int PLOTTING_TYPE_STDDEV_AVERAGE_MINUS = -9991;
    
    private String name, description, misc;
    
    private Color fg_col;
    private int line_width = 2;
    private int line_style= SWT.LINE_SOLID;
    
    private int nodata = Graph.NODATA_DRAW_ZERO;
    
    private int plotting_type = Graph.PLOTTING_TYPE_AVERAGE;
    private int plotting_period_type = Graph.PERIOD_TYPE_TICKS;
    private double plotting_period_value = 1.0;
    private double xoffset = 0;
    private double yoffset = 0;
    private double xdelta = 0;
    private double ydelta = 0;
    private double scaling = 1.0;
    private double last_drawn_value = 0.0d;
    
    private double recentMaxValue = Double.MIN_VALUE;   
    private double tmpMaxValue = 0;
    
    /**
     * A boolean to decide whether to draw indicators on sliders
     */
    protected boolean use_indicators = true;
    
    /**
     * This graph's current indicator
     */
    protected ZoomIndicator indicator;
    
    /**
     * This graph's current indicator source
     */
    protected IndicatorSource isource;
    
    /**
     * This graph's GraphSource
     */
    protected GraphSource source;
    
    /**
     * This graphs y slider
     */
    protected ZoomSlider yslider;
    
    /**
     * This graphs x slider
     */
    protected TimeZoomSlider xslider;
    
    /**
     * A points cache of all the plotted points on a canvas
     */
    protected VisiblePointsCache pointsCache;

	/**
     * LineGraph constructor. Create a new LineGraph instance referencing a GraphSource by specifying the 
     * plotting period type, slider references and plotting colour. Valid plotting constants are defined in 
     * <code>Graph</code>
	 * @param type the plotting period type
	 * @param source the GraphSource
	 * @param xslider the current xslider
	 * @param yslider the current yslider
	 * @param col the graphing color
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph
	 */
	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col) 
    {
		plotting_period_type = type;
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
        pointsCache = new VisiblePointsCache(this);
		init();
		
	}
	
	/**
     * LineGraph constructor. Create a new LineGraph instance referencing a GraphSource by specifying the 
     * plotting period type, slider references and plotting colour. Valid plotting constants are defined in 
     * <code>Graph</code>
     * @param type the plotting period type
     * @param source the GraphSource
     * @param xslider the current xslider
     * @param yslider the current yslider
     * @param col the graphing color
	 * @param pixel_perfect boolean to graph with per pixel accuracy
	 * @param use_indicators boolean to display indicator lines
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph
     */
	public LineGraph(int type, GraphSource source, TimeZoomSlider xslider, ZoomSlider yslider, Color col, boolean pixel_perfect, boolean use_indicators) 
    {
	    plotting_period_type = type;
		this.source = source;
		this.fg_col = col;
		this.xslider = xslider;
		this.yslider = yslider;
		if (pixel_perfect) {
			plotting_period_type = Graph.PERIOD_TYPE_PIXELS;
			plotting_period_value = 1.0;
		} else {
			plotting_period_type = Graph.PERIOD_TYPE_TICKS;
			plotting_period_value = 1.0;
		}
		this.use_indicators = use_indicators;
        pointsCache = new VisiblePointsCache(this);        
		init();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphSource()
	 */
	public BasicGraphSource getGraphSource() 
    {
		return source;	
	}
		
	private void init() 
    {
		if (use_indicators) 
        {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setIndicatorSource(org.eclipse.hyades.ui.widgets.grapher.IndicatorSource)
	 */
	public void setIndicatorSource(IndicatorSource source) {
		isource = source;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.BasicGraph#setShown(boolean)
	 */
	public void setShown(boolean b) 
	{
		if (use_indicators && !b) 
		{
			if (!yslider.isDisposed())
                yslider.removeIndicator(indicator);
			indicator.dispose();			
		} 
		else if (use_indicators && b) 
		{
            if (!yslider.isDisposed())
            {
                indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);            
                yslider.addZoomIndicator(indicator);
            }            
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMin()
	 */
	public double getXMin() 
    {
		return source.getMin();		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXMax()
	 */
	public double getXMax() 
    {
		return source.getMax();
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMin()
	 */
	public double getYMin() 
    {
		return source.getValueMin();		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYMax()
	 */
	public double getYMax() 
    {
		return source.getValueMax();		
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setForeground(org.eclipse.swt.graphics.Color)
	 */
	public void setForeground(Color col) 
    {
		fg_col = col;
		if (use_indicators) 
        {
			indicator.setColor(col);	
		}
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getForeground()
	 */
	public Color getForeground() 
    {
		return fg_col;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineWidth(int)
	 */
	public void setLineWidth(int w) 
    {
		line_width = w;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineWidth()
	 */
	public int getLineWidth() 
    {
		return line_width;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setLineStyle(int)
	 */
	public void setLineStyle(int swt_line_style) 
    {
		line_style = swt_line_style;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getLineStyle()
	 */
	public int getLineStyle() 
    {
		return line_style;	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getNoDataBehaviour()
	 */
	public int getNoDataBehaviour() 
    {
		return nodata;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setNoDataBehaviour(int)
	 */
	public void setNoDataBehaviour(int behaviour) 
    {
		nodata = behaviour;
	}
	
	/**
     * Get the most recent maximum value, as opposed to the overall max
     * 
	 * @return the most recent maximum value
	 */
	public double getRecentMaxValue() 
    {
		return recentMaxValue;
	}
    
/*    private void setAntiAlias(GC gc, boolean on)
    {
        try
        {
            if (on)
                gc.setAntialias(SWT.ON);
            else 
                gc.setAntialias(SWT.OFF);                
        }
        catch (Exception e)
        {
            // We will hit this if we do not have a capable graphics library
        }
    }
*/   	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#paintGraph(org.eclipse.swt.graphics.GC, int, int, int, int)
	 */
	public void paintGraph(GC gc, int x, int y, int w, int h) 
    {
        gc.setLineCap(SWT.CAP_ROUND);
//        setAntiAlias(gc, true);

		tmpMaxValue = Double.MIN_VALUE;

        gc.setForeground(fg_col);
		gc.setLineWidth(line_width);	
		gc.setLineStyle(line_style);
		
//we always draw on the ticks, this way the graphs stay constant 
//since we are averaging over the same periods and thus the same
//data all the time
		ZoomSliderTick tick_start = xslider.getFirstTick();
		ZoomSliderTick tick_end = xslider.getLastTick();

		if (tick_start == null || tick_end == null) 
            return;

		double res = xslider.getIncrement();

		double start = tick_start.getValue() - res;
		double end = tick_end.getValue() + res;

		if (plotting_period_type == Graph.PERIOD_TYPE_TICKS) 
        {
			res = xslider.getIncrement();
			res *= plotting_period_value;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_PIXELS) 
        {
			res = xslider.pixel2Value(1)-xslider.pixel2Value(0);
			res *= plotting_period_value;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_MILLIS) 
        {
			res = plotting_period_value;
			if (res < 10) res = 10;
		} else if (plotting_period_type == Graph.PERIOD_TYPE_UNAVERAGED) 
        {
			//special case, need to iterate through the data points
			//just use 10ms periods for now though, but this is NOT CORRECT in reality
			res = 10;
		}
		
		//stops the averaging jumping around because of different start points
		start -= start % res;
		
		if (plotting_type == Graph.PLOTTING_TYPE_MIN_MAX) 
        {
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MIN,w,h);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MAX,w,h);
			
		} 
        else if (plotting_type == Graph.PLOTTING_TYPE_MIN_MAX_AVERAGE) 
        {
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MIN,w,h);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_MAX,w,h);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_AVERAGE,w,h);
			
		} 
        else if (plotting_type == Graph.PLOTTING_TYPE_STDDEV_AVERAGE) 
        {
			paintGraph(gc,x,y,start,end,res,PLOTTING_TYPE_STDDEV_AVERAGE_PLUS,w,h);
			paintGraph(gc,x,y,start,end,res,PLOTTING_TYPE_STDDEV_AVERAGE_MINUS,w,h);
			paintGraph(gc,x,y,start,end,res,Graph.PLOTTING_TYPE_AVERAGE,w,h);
		} 
        else 
        {
			paintGraph(gc,x,y,start,end,res,plotting_type,w,h);			
		}
//        setAntiAlias(gc, false);
	}

	/**
     * Paint the graph on the given GC
     * 
	 * @param gc the GC to draw on
	 * @param xorigin the xorigin offset to draw on the canvas from 
	 * @param yorigin the yorigin offset to draw on the canvas from
	 * @param start the start range bound
	 * @param end  the end range bound
	 * @param res the resolution
	 * @param plot_type the plotting type
     * @param width the width of the canvas
     * @param height the height of the canvas
	 */
	public void paintGraph(GC gc, int xorigin, int yorigin, double start, double end, double res, int plot_type, int width, int height) 
    {
		boolean found_indicator = false;
        double indicator_loc = 0;
        int indicator_type = 0;        
        if (isource != null)
        {
            indicator_loc = isource.getIndicatorLocation();
            indicator_type = isource.getSourceType();
        }
		double indicator_val = last_drawn_value;       

		boolean first = true;

		double px = 0;
		double py = 0;
		
		double last_clean_cx = 0;
		double last_clean_cy = 0;
		double last_cy = 0;
		        
		// Reset the points cache
        
        pointsCache.reset();
        
		boolean empty = false;
		
		for (double i = start; i <= end; i+= res) {

			double pixel_x;
			double pixel_y;
			double cx = i;
			
			//
			// X offset
			//
			double cx_shifted = cx - xoffset - xdelta;
			
			if (DRAW_ONLY_IF_DATA) 
            {
				//dont draw past the maximum extent of the data
				if (cx_shifted > (source.getMax()+res)) 
                    continue;
				if (cx_shifted < (source.getMin()-res)) 
                    continue;
			}

			double cy = 0;
			
			if (plot_type == Graph.PLOTTING_TYPE_NEAREST) 
            {
				//plot the closest (previous) value
				cy = source.getValueAt(cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_AVERAGE) 
            {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_SUM) 
            {
				cy = source.getSumBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_MIN) 
            {
				cy = source.getMinBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_MAX) 
            {
				cy = source.getMaxBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_COUNT) 
            {
				cy = source.getCountBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == Graph.PLOTTING_TYPE_STDDEV) 
            {
				cy = source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == PLOTTING_TYPE_STDDEV_AVERAGE_PLUS) 
            {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted) + source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == PLOTTING_TYPE_STDDEV_AVERAGE_MINUS) 
            {
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted) - source.getStandardDeviationBetween(cx_shifted-res,cx_shifted);
			} 
            else if (plot_type == PLOTTING_TYPE_GRADIENT) 
            {
				//get average
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			} 
            else 
            {
				//unrecognised type, default to average
				cy = source.getAverageBetween(cx_shifted-res,cx_shifted);
			}			
			
			//dont draw lines on the first run, we wont have valid previous pixel data
			if (first) 
            {
				first = false;	
				empty = Double.isNaN(cy);

				if (empty && (nodata == Graph.NODATA_INTERPOLATE))
				{
					// Do not draw anything on the first data point.
					first = true;
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = py;										

				}
				else if (empty && (nodata == Graph.NODATA_DO_NOTHING)) 
                {
					first = true;
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = py;										
				} 
                else 
                {
					if (empty && (nodata == Graph.NODATA_DRAW_PREVIOUS)) 
                    {
						cy = last_clean_cy;                        
					} 
                    else if (empty && (nodata == Graph.NODATA_DRAW_ZERO)) 
                    {
						cy = 0;
					}	                     
                    
					last_clean_cx = cx;
					
					//store this for DRAW_PREVIOUS
					last_clean_cy = cy;
					
					//Y Offset
					cy += yoffset+ydelta;
					
					//apply static scaling
					cy = cy * scaling;
		
					last_cy = cy;
		
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = yslider.value2Pixel(cy);
				
					if (cy > tmpMaxValue) 
                        tmpMaxValue = cy;					
                    drawLine(gc,xorigin+(int)pixel_x,yorigin+(int)pixel_y,xorigin+(int)pixel_x,yorigin+(int)pixel_y);
				}								
			} 
            else 
            {
				boolean emptyPrevious = empty;
				empty = Double.isNaN(cy);

				if (empty && (nodata == Graph.NODATA_DO_NOTHING)) 
                {
					//draw nothing, need to start the graph all over again
					first = true;
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = py;

                    if (isource != null)
                    {
    					//Indicator source type TIME - specifies the time value to show values at 
    					if (indicator_type == IndicatorSource.TYPE_TIME) 
                        {
    						if (indicator_loc > cx-res && indicator_loc < cx) 
                            { 	
    							found_indicator = true;
    							indicator_val = 0;
    						}
    					}                        
    					//Indicator source type PIXEL - specifies the pixel to show values at 
    					if (indicator_type == IndicatorSource.TYPE_PIXEL) 
                        {
    						if (indicator_loc > px && indicator_loc < pixel_x) 
                            { 	
    							found_indicator = true;
    							indicator_val = 0;
    						}
    					}					
                    }					
					//we had a data point just before this, but now we have none
					if (!emptyPrevious) 
                    {
						//draw a horizontal line rather than a dot for individual data values                        
                        drawLine(gc,xorigin+(int)px,yorigin+(int)py,xorigin+(int)pixel_x,yorigin+(int)py);                        
					}					
				} 
                else 
                {                       	
                	boolean valid = true;

					if (empty && (nodata == Graph.NODATA_DRAW_PREVIOUS)) 
                    {
						cy = last_clean_cy;
					} 
                    else if (empty && (nodata == Graph.NODATA_DRAW_ZERO)) 
                    {
                        valid = false;
                        // Compensate for any y offset so that result is 0
						cy = -yoffset-ydelta;
					}
                    else if (nodata == Graph.NODATA_INTERPOLATE)
                    {
                    	if (!empty)
                    	{                    		
							// We need to remember the last clean x point so as to draw an interpolated straight
	                    	// line between valid data points
							last_clean_cx = cx;
							last_clean_cy = cy;
                    	}
                    	else
                    	{
                    		cx = last_clean_cx;
                    		cy = last_clean_cy;
                    		valid = false;
                    	}
                    	
					}
					//special case for gradient
					if (plot_type == PLOTTING_TYPE_GRADIENT) 
                    {
						double deltay = cy - last_clean_cy;
						double deltax = (cx - last_clean_cx)/1000.0d;

						last_clean_cx = cx;
						
						//store this for DRAW_PREVIOUS
						last_clean_cy = cy;

						//gradient (y/x)
						cy = deltay/deltax;					
					} 
                    else
                    {					
						//store this for DRAW_PREVIOUS
						last_clean_cy = cy;
					}
					
					//Y Offset
					cy += yoffset;
					cy += ydelta;
					
					//apply static scaling
					cy = cy * scaling;
	
					//Indicator source type TIME - specifies the time value to show values at
                    if (isource != null)
                    {
    					if (indicator_type == IndicatorSource.TYPE_TIME) 
                        {
    						if (indicator_loc > cx-res && indicator_loc < cx) 
                            { 	
    							found_indicator = true;
    							indicator_val = cy;
    						}
    					}
                    }
					last_cy = cy;
		
					pixel_x = xslider.value2Pixel(cx);
					pixel_y = yslider.value2Pixel(cy);
	
					if (cy > tmpMaxValue) 
                    {
						tmpMaxValue = cy;
					}
					
					//Indicator source type PIXEL - specifies the pixel to show values at
                    if (isource != null)
                    {
    					if (indicator_type == IndicatorSource.TYPE_PIXEL) 
                        {
    						if (indicator_loc > px && indicator_loc < pixel_x) 
                            { 	
    							found_indicator = true;
    							indicator_val = cy;
    						}
    					}
                    }
                    if (!(empty && nodata == Graph.NODATA_INTERPOLATE))
                    	drawLine(gc,xorigin+(int)px,yorigin+(int)py,xorigin+(int)pixel_x,yorigin+(int)pixel_y,valid);				
                }
			}
			px = pixel_x;
			py = pixel_y;		
		}        
        
        // Bug # 98610
		double startpx = px;
		double endpx = Math.min(xslider.value2Pixel(source.getSessionMax()+xoffset+xdelta), width);		
        if (nodata == Graph.NODATA_DRAW_PREVIOUS && last_clean_cx != 0)
        {
            drawLine(gc,xorigin+(int)px, yorigin+(int)py,xorigin+(int)endpx,yorigin+(int)py);
            startpx = endpx;
        }
        //if (nodata == Graph.NODATA_DRAW_ZERO)// && last_clean_cx != 0)
        { 
//            endpx = Math.min(xslider.value2Pixel(max),width);
            if (endpx > px)
            {
                double endy = yslider.value2Pixel(0);
                int oldStyle = gc.getLineStyle();
                gc.setLineStyle(SWT.LINE_DOT);
                drawLine(gc,xorigin+(int)startpx, yorigin+(int)endy, xorigin+(int)endpx,yorigin+(int)endy, false);
                gc.setLineStyle(oldStyle);
            }
        }
        // End
		
		//this will be the value that appears on the indicator
		last_drawn_value = last_cy;
		
		//this is the most recent max value
		recentMaxValue = tmpMaxValue;

        if (use_indicators == true)
        {
    		if (isource == null || !found_indicator) 
            {
    			indicator.setBaseValue(yoffset+ydelta);
    			indicator.setValue(last_drawn_value);
    		} 
            else 
            {
    			indicator.setBaseValue(yoffset+ydelta);
    			indicator.setValue(indicator_val);
    		}
        }
		yslider.updateIndicators();        
	}
    
    private void drawLine(GC gc,int startx, int starty, int endx, int endy)    
    {
        drawLine(gc,startx, starty, endx, endy,true);
    }
    
    private void drawLine(GC gc,int startx, int starty, int endx, int endy, boolean valid)
    {
        gc.drawLine(startx,starty,endx,endy);
        pointsCache.addPoint(endx, endy,valid);
    }
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingType()
	 */
	public int getPlottingType() 
    {
		return plotting_type;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingType(int)
	 */
	public void setPlottingType(int type) 
    {
		plotting_type = type;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setPlottingPeriod(int, double)
	 */
	public void setPlottingPeriod(int type, double value) 
    {
		plotting_period_type = type;
		plotting_period_value = value;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodType()
	 */
	public int getPlottingPeriodType() 
    {
		return plotting_period_type;
	}
 
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getPlottingPeriodValue()
     */      
	public double getPlottingPeriodValue() 
    {
		return plotting_period_value;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXOffset(double)
	 */
	public void setXOffset(double millis) 
    {
		xoffset = millis;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXOffset()
	 */
	public double getXOffset() 
    {
		return xoffset;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYOffset(double)
	 */
	public void setYOffset(double millis) 
    {
		yoffset = millis;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYOffset()
	 */
	public double getYOffset() 
    {
		return yoffset;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getXSlider()
	 */
	public TimeZoomSlider getXSlider() 
    {
		return xslider;	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getYSlider()
	 */
	public ZoomSlider getYSlider() 
    {
		return yslider;	
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setXSlider(org.eclipse.hyades.ui.widgets.zoomslider.TimeZoomSlider)
	 */
	public void setXSlider(TimeZoomSlider slider) 
    {
		xslider = slider;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setYSlider(org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider)
	 */
	public void setYSlider(ZoomSlider slider) 
    {
		if (use_indicators) 
        {
			yslider.removeIndicator(indicator);
			indicator.dispose();
		}
		yslider = slider;
		if (use_indicators) 
        {
			indicator = new ZoomIndicator(yslider,last_drawn_value,fg_col);
			yslider.addZoomIndicator(indicator);
		}
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setStaticScaling(double)
	 */
	public void setStaticScaling(double mult) 
    {
		scaling = mult;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getStaticScaling()
	 */
	public double getStaticScaling() 
    {
		return scaling;	
	} 

    /**
     * Return this line graph's cache of the currently rendered points. 
     * 
     * @return the current points cache
     */
    public VisiblePointsCache getPointsCache()
    {
        return pointsCache;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setName(java.lang.String)
     */
    public void setName(String name)
    { 
        this.name = name;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getName()
     */
    public String getName()
    {
        return name;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setDescription(java.lang.String)
     */
    public void setDescription(String description)
    {
        this.description = description;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getDescription()
     */
    public String getDescription()
    {
        return description;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setMisc(java.lang.String)
     */
    public void setMisc(String misc)
    {
        this.misc = misc;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getMisc()
     */
    public String getMisc()
    {
        return misc;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#setGraphCanvas(org.eclipse.hyades.ui.widgets.grapher.GraphCanvas)
     */
    public void setGraphCanvas(GraphCanvas gc)
    {
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#getGraphCanvas()
     */
    public GraphCanvas getGraphCanvas()
    {
        return null;
    }

    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.grapher.Graph#dispose()
     */
    public void dispose()
    {        
    }   
        
    /**
     * Boolean to show indicator values on sliders or not 
     * @param useIndicators true to display indicators on sliders, false otherwise
     */
    public void setUseIndicators(boolean useIndicators)
    {
        use_indicators = useIndicators;
    }
    
    /**
     * Boolean to show indicator values on sliders or not 
     * @return true to display indicators on sliders, false otherwise
     */
    public boolean getUseIndicators()
    {
        return use_indicators;
    }

	public void setXDelta(double millis) 
	{
		xdelta = millis;
	}

	public double getXDelta() 
	{
		return xdelta;	
	}

	public void setYDelta(double millis) 
	{
		ydelta = millis;
	}

	public double getYDelta() 
	{
		return ydelta;
	}
}