/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.grapher.sources;

import org.eclipse.hyades.ui.widgets.grapher.GraphSource;

/**
 * A Test Graph Source implementation of SINE or COSINE wave data.
 * 
 * @author gchristelis
 *
 */
public class TestGraphSource implements GraphSource 
{
    /**
     * The Sine Wave Constant. Sets the graph source to return sine wave point data.
     */
    public static final int SINE_WAVE = 0;
    
    /**
     * The Cosine Wave Constant. Sets the graph source to return cosine wave point data.
     */
    public static final int COSINE_WAVE = 1;
        
    private int type;
    private double min;
    private double max;

    /**
     * TestGraphSource constructor. Sets the data point bounds between min and max and
     * the graph type. Possible constants are <code>SINE_WAVE</code> and <code>COSINE_WAVE</code>.
     *  
     * @param min the minimum time x value 
     * @param max the maximum time x value
     * @param type the graph type
     */
	public TestGraphSource(double min, double max, int type) {
		this.min = min;
		this.max = max;
		this.type = type;
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMin()
	 */
	public double getMin() {
		return min;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMax()
	 */
	public double getMax() {
		return max;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getValueMin()
	 */
	public double getValueMin() {
		return 0;
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getValueMax()
	 */
	public double getValueMax() 
    {
		return 100;
	}	
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getValueAt(double)
	 */
	public double getValueAt(double x) 
    {
        // return a scaled and transposed y value for the given x coordinate. 
		if (type == SINE_WAVE) 
        {
			return 100d + (50d * (double)Math.sin(Math.toRadians(x)));
		} 
        else if (type == COSINE_WAVE) 
        {
			return 100d + (50d * (double)Math.cos(Math.toRadians(x)));
		} 
        else 
        {
			return 0;
		}        	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getAverageBetween(double, double)
	 */
	public double getAverageBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getSumBetween(double, double)
	 */
	public double getSumBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMaxBetween(double, double)
	 */
	public double getMaxBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}

	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getMinBetween(double, double)
	 */
	public double getMinBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}
	
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getCountBetween(double, double)
	 */
	public double getCountBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.grapher.GraphSource#getStandardDeviationBetween(double, double)
	 */
	public double getStandardDeviationBetween(double x, double x2) 
    {
		return getValueAt(x);	
	}

    public double getSessionMin()
    {
        // TODO Auto-generated method stub
        return 0;
    }

    public double getSessionMax()
    {
        // TODO Auto-generated method stub
        return 0;
    }
}