/**********************************************************************
 * Copyright (c) 2005, 2008 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Date;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.widgets.Composite;

import com.ibm.icu.util.GregorianCalendar;

/**
 * The TimeZoomSlider is an instance of ZoomSlider that represents times on the scale, as 
 * opposed to generic linear data.
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class TimeZoomSlider extends ZoomSlider
{
    private static final double [] LINEAR_PROGRESSION = {1, 2, 5};
    private static final double [] OOM4 = {1, 2, 3, 6}; // 10 secs > 1 min
    private static final double [] OOM5 = {1.2, 3, 6}; // 2 mins > 10 mins
    private static final double [] OOM6 = {1.2, 1.8, 3.6, 7.2}; // 20 mins > 2 hrs
    private static final double [] OOM7 = {1.44, 2.16, 4.32, 8.64};
    private static final double [] GTOOM7 = {1*0.864, 2*0.864, 6*0.864};
        
    private GregorianCalendar calendar = new GregorianCalendar();        
    private boolean wallTime = true;
    
    /**
     * TimeZoomSlider constructor
     * @param parent
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#ZoomSlider(Composite)
     */
    public TimeZoomSlider(Composite parent)
    {
    	super(parent);
    }
    
    /**
     * Return the Gregorian Calendar used by this time slider
     * @return the GregorianCalendar
     */
    public GregorianCalendar getGregorianCalendar()
    {
        return calendar;
    }
    
    // CREATE TICKS
    /** Create the vector of ticks */
    protected void createTicks()
    {
        calculateIncrement();
        
        // Empty the tick vector
        tickVector.removeAllElements();
        
        // Refill the tick vector
        double tickValue = ZoomSliderUtilities.round(visible[MIN] - increment, increment);
        
        //EditorPlugin.DBG.info(tickValue + " + " + increment + " = " + (tickValue + increment));
        
        while (tickValue <= visible[MAX] + increment)
        {
            //EditorPlugin.DBG.info((long)tickValue);
            
            tickVector.addElement(new TimeZoomSliderTick(this, tickValue));

            tickValue += increment;
            
            tickValue = ZoomSliderUtilities.round(tickValue, increment);
        }
        
        // Depending on the pixel spacing between ticks find a reasonable number of 
        // intermediate selecteable points (5 or 10)
		int selectablePoints = 10;
    	int tickSpacing = (int)(getPixelRange()/(getVisibleValueRange()/increment));
		if (tickSpacing < 20)
		{
		    selectablePoints = 5;
		}
		
        // Set the unit increment based on the intermediate selectable points
		unitIncrement = increment/selectablePoints;
		if (unitIncrement < resolution)
		{
		    unitIncrement = resolution;
		}
    }

    /**
     * Set the TimeZoomSlider wall time boolean. 
     * @param wall true if the SliderTicks use HH:MM:SS, false for possible concise representing (MM:SS)
     */
    public void setWallTime(boolean wall)
    {
        wallTime = wall;
    }
    
    /** 
     * The TimeZoomSlider wall time boolean
     * @return true if the SliderTicks use HH:MM:SS, false for possible concise representing (MM:SS)
     */
    public boolean isWallTime()
    {
        return wallTime;
    }
    
    /* (non-Javadoc)
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#findDecimalPointPosition()
     */
    public void findDecimalPointPosition()
    {
    	GC gcTemp = new GC(this);
    	
        decimalPointPosition = getBounds().width - 10;
        
        gcTemp.dispose();
    }
    
	/* (non-Javadoc)
	 * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#computeSize(int, int, boolean)
	 */
	public Point computeSize(int wHint, int hHint, boolean changed)
	{
		Point standard = super.computeSize(wHint, hHint, changed);
		Point size = new Point(standard.x, standard.y); 
		
   	    if (orientation == VERTICAL)
	    {
			if (wHint == SWT.DEFAULT)
			{
				GC gcTemp = new GC(this);				
				size.x = gcTemp.stringExtent("99:99:99").x + 20;
				gcTemp.dispose();				
			}
			else
			{
				size.x = wHint;
			}
			
			if (hHint != SWT.DEFAULT)
			{
				size.y = hHint;
			}
	    }
        else
	    {
	    	if (hHint == SWT.DEFAULT)
	    	{
	    		size.y = getFontHeight() + 20;
	    	}
	    	else
	    	{
	    		size.y = hHint;
	    	}
	    	
	    	if (wHint != SWT.DEFAULT)
	    	{
	    		size.x = wHint;
	    	}
	    }
	    
	    return size;
	}
	
    /** 
     * Calculate the value spacing between consecutive ticks 
     */
    protected void calculateIncrement()
    {
        TickValueRange tvr = new TickValueRange();
        
        if (tvr.outside())
        {
            int maxTickCount = getPixelRange() / MIN_TICKWIDTH;

            increment = getVisibleValueRange() / maxTickCount;

            int orderOfMagnitude = ZoomSliderUtilities.calculateOOM(increment);
            
            double [] progression = getProgression(orderOfMagnitude);
            
            increment = progression[0] * Math.pow(10, orderOfMagnitude);
            
            while (increment < tvr.min)
            {
                for (int i=0; i<progression.length; i++)
                {
                    increment = progression[i] * Math.pow(10, orderOfMagnitude);
                    
                    if (increment >= tvr.min)
                    {
                        increment = (double)((long)increment);
                        
                        break;
                    }
                }
                orderOfMagnitude++;
                
                progression = getProgression(orderOfMagnitude);
            }
            
            incrementOOM = ZoomSliderUtilities.calculateOOM(increment);
        }        
    }   
    
    protected double [] getProgression(int oom)
    {
        switch (oom)
        {
        	case 1:
        	case 2:
        	case 3: return LINEAR_PROGRESSION;
            case 4:
                return OOM4;
            case 5:
                return OOM5;
            case 6:
                return OOM6;
            case 7:            	
                return OOM7;
            default:
            	return GTOOM7;
        }
    }    
    
    protected double getZero()
    {
        GregorianCalendar calendar = new GregorianCalendar();
        
        calendar.setTime(new Date());
        
        double hrs = 1 + (double)calendar.get(GregorianCalendar.HOUR_OF_DAY);
        
        double mins = calendar.get(GregorianCalendar.MINUTE);
        
        double secs = calendar.get(GregorianCalendar.SECOND);
        
        return (hrs * 60 * 60) + (mins * 60) + secs;
    }    
}        
        
