/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Date;

import com.ibm.icu.text.DateFormat;
import com.ibm.icu.util.Calendar;
import com.ibm.icu.text.NumberFormat;

/**
 * TimeZoomSliderTick represents a tick on a TimeZoomSlider
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class TimeZoomSliderTick extends ZoomSliderTick
{	
    private int hrs, mins, secs;
    private int day, month, year;
    private String hrs$, mins$, secs$;
    
    private static Object df_LOCK = new Object();
    private static DateFormat df = DateFormat.getDateInstance();
    private static DateFormat dtf = DateFormat.getDateTimeInstance();
    private static DateFormat tf = DateFormat.getTimeInstance(DateFormat.MEDIUM);
    private static NumberFormat nf = NumberFormat.getInstance();
    
    static
    {    	
    	nf.setMinimumIntegerDigits(2);
    }
    
    private String getWallTitle()
    {
//        String title = "";
        double endTime = slider.getMaxVisible();
        double startTime = slider.getMinVisible();
        Date d = new Date((long) sliderValue);
        
        synchronized(df_LOCK) {
	        if (endTime - startTime < 24 * 60 * 60 * 1000)
	        {
	            return tf.format(d);
	        }
	        else if (endTime - startTime < 24 * 60 * 60 * 1000 * 5)	// If there are less that 5 days on the slider show date and time
	        {
	            return dtf.format(d);
	        }
	        else
	        {
	            return df.format(d);
	        }
        }

    }
    
	/**
     * TimeZoomSliderTick constructor.
     * 
	 * @param slider the ZoomSlider parent for this tick
	 * @param d the double location value of this tick
	 */
	public TimeZoomSliderTick(ZoomSlider slider, double d)
	{
		this.slider = slider;
        
        sliderValue = d;
                
        calculateComponents();
        
        if (((TimeZoomSlider)slider).isWallTime())
        {
            representation = getWallTitle();
        }
        else
        {
            representation = "";
            
            if (hrs != 0)
            {
                representation += hrs$ + ":";
            }
            if (mins != 0 || hrs != 0)
            {
                representation += mins$ + ":";
            }
            
            representation += secs$;
        }    
        
        sliderPixel = (int)slider.value2Pixel(sliderValue);
        
        textVisible = isTextVisible();
    }
	
    private void calculateComponents()
    {
        long millis = (long)sliderValue;
        
        if (!((TimeZoomSlider)slider).isWallTime())
        {
            millis = (long)sliderValue - slider.rawOffset;
        }    
        
        ((TimeZoomSlider)slider).getGregorianCalendar().setTime(new Date(millis));
        
        day = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.DAY_OF_YEAR);
        month = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.MONTH);
        year = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.YEAR);
        
        hrs = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.HOUR_OF_DAY);
        hrs$ = nf.format(hrs);        

        mins = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.MINUTE);
        mins$ = nf.format(mins);

        secs = ((TimeZoomSlider)slider).getGregorianCalendar().get(Calendar.SECOND);
        secs$ = nf.format(secs);
    }
    
    private boolean isTextVisible(int increment)
    {
        switch (increment)
        {
        	case 0: 	return true;
        	
            case 1:     return (secs%5 == 0);
                    
            case 2:     return (secs%10 == 0);
        
            case 5:     return (secs%15 == 0);
            
            case 10:    return (secs%30 == 0);

            case 20:    return (secs%60 == 0);
            
            case 30:    return (secs%60 == 0 && mins%2 == 0);
            
            case 60:    return (mins%5 == 0);
            
            case 120:   return (mins%10 == 0);

            case 300:   return (mins%20 == 0);

            case 600:   return (mins%60 == 0);
            
            case 1200:  return (mins%60 == 0 && hrs%2 == 0);

            case 1800:  return (mins%60 == 0 && hrs%2 == 0);
            
            case 3600:  return (hrs%12 == 0);

            case 7200:  return (hrs%12 == 0);        
            
            case 14400: return (hrs%12 == 0);

            case 21600: return (hrs%24 == 0);
            
            case 43200: return (hrs%24 == 0);
            
            case 86400: return (day % 4 == 0);
                                             
            default:
            {
            	int dayIncrement = increment / 86400;
            	if (dayIncrement <= 20)
            	{
	        		int dayOfYear = (day / dayIncrement)*dayIncrement;
	        		return (dayOfYear % (4*dayIncrement) == 0); 
            	}
            	else if (dayIncrement < 380)
            	{            		
            		int monthIncrement = dayIncrement / 30;
	        		int monthOfYear = (month / monthIncrement) * monthIncrement;
	        		return (monthOfYear % (2*monthIncrement) == 0);
            	}
            	else
            	{
            		int yearIncrement = dayIncrement / 360;
	        		int thisYear = (year / yearIncrement) * yearIncrement;
	        		thisYear = (thisYear + (thisYear % 2));
	        		return (thisYear % (4*yearIncrement) == 0);            		
            	}
            }
        }        	
    }        

    protected boolean isTextVisible()
    {
        int inc = (int)((long)(slider.getIncrement())/1000);
        boolean res = isTextVisible(inc);
        return res;
    }
}
