/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Vector;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.graphics.Rectangle;

/**
 * A zoom control bar is a user adjustable entity placed on a slider that enables the user to 
 * dynamically alter a generic control by adjusting the position of the entity on the slider
 *  
 * @author gchristelis
 * @since 4.0.0
 */
public class ZoomControlBar extends Object
{
    /**
     * The slider this bar appears on
     */
    protected ZoomSlider slider;
    /**
     * This control bars current value
     */
    protected double value;
    protected Rectangle	area;
    /**
     * This control bars color
     */
    protected Color	color;
    
    /**
     * The size of this control bar
     */
    protected int size = 11;
    
    /**
     * The enabled state of this control bar
     */
    protected boolean enabled = true;
    
    /**
     * The image to display on the control bar
     */
    protected Image image;
    
    /**
     * The tooltip to show on hovering over the control bar
     */
    protected String tooltip = null;
    
    
    private int indent = 3;    
    private double	oldValue;    
    private Vector listeners = new Vector();
    
    /**
     * Construct a new ZoomControlBar with the default color and initial value and size
     * @param slider the ZoomSlider to create a ZoomControlBar on
     */
    public ZoomControlBar(ZoomSlider slider)
    {
        this.slider = slider;
        
		setColor(slider.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        
        value = slider.getMinVisible();
        
        slider.addZoomControlBar(this);
    }
    
    /**
     * Construct a new ZoomControlBar with default initial color and size
     * @param slider the parent ZoomSlider
     * @param value the initial control value
     */
    public ZoomControlBar(ZoomSlider slider, double value)
    {
        this(slider);
        
        setValue(value);
        
        slider.updateScale();
    }
    
    /**
     * Construct a new ZoomControlBar with default initial value and size
     * @param slider the parent ZoomSlider
     * @param color the initial color of the control bar
     */
    public ZoomControlBar(ZoomSlider slider, Color color)
    {
        this(slider);
        
        setColor(color);
        
        slider.updateScale();
    }
    
    /**
     * Construct a new ZoomControlBar with default size
     * @param slider the parent ZoomSlider
     * @param value the initial value
     * @param color the initial color of the control bar
     */
    public ZoomControlBar(ZoomSlider slider, double value, Color color)
    {
        this(slider, value);
        
        setColor(color);
        
        slider.updateScale();
    }
    
    /**
     * Contruct a new ZoomControlBar with default color and value
     * @param slider the parent ZoomSlider
     * @param size the control bar size
     */
    public ZoomControlBar(ZoomSlider slider, int size)
    {
        this(slider);
        
        setSize(size);
        
        slider.updateScale();
    }
    
    /**
     * Construct a new ZoomControlBar with default color
     * @param slider the parent ZoomSlider
     * @param value the initial value
     * @param size the initial control bar size
     */
    public ZoomControlBar(ZoomSlider slider, double value, int size)
    {
        this(slider, value);
        
        setSize(size);
        
        slider.updateScale();
    }    
    
    /**
     * Construct a new ZoomControlBar with default initial value
     * @param slider the parent ZoomSlider
     * @param color the initial control bar color
     * @param size the initial control bar size
     */
    public ZoomControlBar(ZoomSlider slider, Color color, int size)
    {
        this(slider, color);
        
        setSize(size);
        
        slider.updateScale();
    }
    
    /**
     * Construct a new ZoomControlBar
     * @param slider the parent ZoomSlider
     * @param color the initial control bar color
     * @param size the initial control bar size
     * @param value the intial control bar value
     */
    public ZoomControlBar(ZoomSlider slider, double value, Color color, int size)
    {
        this(slider, value, color);
        
        setSize(size);
        
        slider.updateScale();
    }    

    /**
     * Set the parent zoomslider for this control bar
     * @param slider the parent slider
     */
    public void setZoomSlider(ZoomSlider slider)
    {
        this.slider = slider;
    }
    
    /**
     * Get the parent zoomslider
     * @return the parent zoomslider
     */
    public ZoomSlider getZoomSlider()
    {
        return slider;
    }    

    /**
     * Set this control's tooltip text
     * @param text the tooltip text to set
     */
    public void setToolTipText(String text) 
    {
    	tooltip = text;
    }
    
    /**
     * Get this control's tooltip text
     * @return the current tooltip text, and null if none has been set
     */
    public String getToolTipText() 
    {
    	return tooltip;	
    }
    
    /**
     * Set this control bar's control value
     * @param val the value to set this control bar control to 
     */
    public void setValue(double val)
    {
        oldValue = value;
        
        if (val > slider.getMaxLimit()) 
            val = slider.getMaxLimit();
		if (val < slider.getMinLimit()) 
            val = slider.getMinLimit();
        
        value = val;
    }
    
    /**
     * Get the current control bar's value
     * @return the current value
     */
    public double getValue()
    {
        return value;
    }
    
    /**
     * Set the control bar's color
     * @param color the color to set
     */
    public void setColor(Color color)
    {
        this.color = color;
    }    
    
    /**
     * Get the control bar's color
     * @return the current control bar color
     */
    public Color getColor()
    {
        return color;
    }        
        
    /**
     * Set the enabled state of this control bar
     * @param en a boolean depicting the enabled state of this control
     */
    public void setEnabled(boolean en)
    {
        enabled = en;
    }
    
    /**
     * Check whether this control is enabled
     * @return a boolean representing the enabled state of the control bar
     */
    public boolean isEnabled()
    {
        return enabled;
    }    

    /**
     * Set the rectangle height size of the control bar
     * @param size the size of the bar in pixels
     */
    public void setSize(int size)
    {
        this.size = size;
    }
    
    /**
     * Get the rectangle height size of the control bar
     * @return the height in pixels
     */
    public int getSize()
    {
        return size;
    }    
        
    /**
     * Hide the control
     * @param backBuffer a reverse image to paint
     * @param g the parent GC
     */
    public void hide(Image backBuffer, GC g)
    {
        g.setClipping(getDrawingArea(oldValue));
        g.drawImage(backBuffer, 0, 0);
        g.setClipping(0, 0, slider.getBounds().width, slider.getBounds().height);
    }    
    
    /**
     * Paint this zoom control bar on the provided GC
     * @param g the GC to paint this control bar on
     */
    public void paint(GC g)
    {
        area = getDrawingArea(value);
        
        // Create the ZoomControlBar's Image
        if (image == null)
        {
        	// Draw the ZoomControlBar on a temporary Image, and get its ImageData
        	Image tempImage = new Image(slider.getDisplay(), area.width, area.height);
			draw3DBox(new GC(tempImage), 0, 0, area.width, area.height, color);
        	ImageData imageData = tempImage.getImageData();
        	tempImage.dispose();
        	
	        // Set the Alpha blending on the ImageData
	        for (int i = 0; i < area.width; i++)
	        { 
                for (int j = 0; j < area.height; j++) 
                { 
                        imageData.setAlpha(i, j, 180); 
                } 
	        } 

			// Use the ImageData to create the required transparent Image
        	image = new Image(slider.getDisplay(), imageData);
        }

		// Draw the ZoomControlBar's Image on the Graphics Context        
        g.drawImage(image, area.x, area.y);

		// Draw 2 little lines from the edges of the Image to the edges of the ZoomSlider        
        g.setForeground(slider.getForeground());
        if (slider.getOrientation() == ZoomSlider.HORIZONTAL)
        {
            g.drawLine(area.x + area.width/2, 1, area.x + area.width/2, indent);
            g.drawLine(area.x + area.width/2, slider.getBounds().height - indent, area.x + area.width/2, slider.getBounds().height - 1);
        }
        else
        {
            g.drawLine(1, area.y + area.height/2, indent, area.y + area.height/2);
            g.drawLine(slider.getBounds().width - indent, area.y + area.height/2, slider.getBounds().width - 1, area.y + area.height/2);
        }            
    }
    
    protected Rectangle getDrawingArea(double d)
    {
        if (slider.getOrientation() == ZoomSlider.VERTICAL)
        {
            return new Rectangle(indent, (int)(slider.value2Pixel(d)) - size/2, slider.getBounds().width - indent * 2, size);
        }
        else
        {
            return new Rectangle((int)(slider.value2Pixel(d)) - size/2, indent, size, slider.getBounds().height - indent * 2);
        }    
    }
    
    /**
     * Returns whether an arbitrary coordinate is within this control bar control
     * @param x the x coordinate
     * @param y the y coordinate
     * @return true if (x,y) is contained within this control, false otherwise
     */
    public boolean inside(int x, int y)
    {
        if (area == null)
        {
            return false;
        }
        
        if (area.contains(x,y) && enabled)
        {
            return true;
        }
        return false;
    }
    
    /**
     * Send a ZoomControlBarEvent to all listeners
     * @param finished true if the event shows completion of a change
     */
	public void sprayZoomControlBarEvent(boolean finished)
	{
        for (int i=0; i<listeners.size(); i++)
        {
            ((ZoomControlBarListener)(listeners.elementAt(i))).zoomControlBarChanged(new ZoomControlBarEvent(this, value, finished));
        }
	}
	
    /**
     * Add a ZoomControlBarListener to this control bar's list of event listeners
     * @param listener an implementation of ZoomControlBarListener
     */
    public void addZoomControlBarListener(ZoomControlBarListener listener)
    {
        listeners.addElement(listener);
    }
    
    /**
     * Remove a ZoomControlBarListener from this control bar's list of listeners
     * @param listener an implementation of ZoomControlBarListener
     */
    public void removeZoomControlBarListener(ZoomControlBarListener listener)
    {
        listeners.removeElement(listener);
    }

    protected void draw3DBox(GC g, int x, int y, int w, int h, Color boxcolor)
    {
        Color darkColor = null, darkerColor = null, lightColor = null;
        double dark = 0.5, darker = 0.1, light = 3;
        
		//if (!enabled) boxcolor = slider.getBackground();


        RGB rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * dark);
		rgb.blue = (int)(rgb.blue * dark);
		rgb.green = (int)(rgb.green * dark);				
		darkColor = new Color(slider.getDisplay(), rgb);
		
        rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * darker);
		rgb.blue = (int)(rgb.blue * darker);
		rgb.green = (int)(rgb.green * darker);				
		darkerColor = new Color(slider.getDisplay(), rgb);

        rgb = boxcolor.getRGB();
		rgb.red = (int)(rgb.red * light);
		rgb.blue = (int)(rgb.blue * light);
		rgb.green = (int)(rgb.green * light);				
		if (rgb.red > 255) rgb.red = 255;
		if (rgb.blue > 255) rgb.blue = 255;
		if (rgb.green > 255) rgb.green = 255;		
		lightColor = new Color(slider.getDisplay(), rgb);

        g.setBackground(darkerColor);
        g.fillRectangle(x, y, w, h);
        
        g.setBackground(boxcolor);
        g.fillRectangle(x, y, w-1, h-1);
        
        g.setBackground(darkColor);
        g.fillRectangle(x+1, y+1, w-2, h-2);
        
        g.setBackground(lightColor);
        g.fillRectangle(x+1, y+1, w-3, h-3);
        
        g.setBackground(boxcolor);
        g.fillRectangle(x+2, y+2, w-4, h-4);
        
        darkColor.dispose();
        darkerColor.dispose();
        lightColor.dispose();
   }
   
    /**
     * Dispose of system resources
     */
   public void dispose()
   {
		if (image != null) 
            image.dispose();
		if (color != null) 
            color.dispose();
   }
}


            
            
