/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Vector;

import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;

/**
 * A collection of ZoomControlBars for a particular ZoomSlider
 * 
 * @author gchristelis
 * @since 4.0.0
 *
 */
public class ZoomControlBarsCollection
{
	private ZoomSlider slider;	
	private Vector bars = new Vector();	    
	private boolean selected;	
	private boolean primed;	
	private boolean moved;
	    
    /**
     * Create a new ZoomControlBarsCollection for this slider
     * @param slider the slider that the zoom controls will be associated with
     */
    public ZoomControlBarsCollection(ZoomSlider slider)
    {
        this.slider = slider;
    }    
        
    /**
     * Add this ZoomControlBar to the collection of control bars for this slider. Zoom
     * Control bars are not duplicated in the list.
     * @param bar the ZoomControlBar to add to the collection.
     */
	public void add(ZoomControlBar bar)
	{
        if (bar != null)
        {
            if (!bars.contains(bar))
            {
	            bars.addElement(bar);
	        }
	    }    
	}
	    
    /**
     * Remove a ZoomControlBar from the collection of bars.
     * @param bar the ZoomControlBar to remove from the list
     */
	public void remove(ZoomControlBar bar)
	{
	    if (bar != null)
	    {
	        if (bars.contains(bar))
	        {
	            bars.removeElement(bar);
	        }
	    }    
	    selected = false;
	}
	
    /**
     * Returns true if the collection contains at least one bar.
     * @return true if the number of bars is at least 1, false if it is 0
     */
	public boolean hasBars()
	{
	    return !(bars.isEmpty());
	}
	
    /**
     * Returns <code>true</code> if the current collection is selected. A collection is
     * selected if the user has started interacting with any one of the control bars in the collection
     * @return whether this collection is selected
     */
	public boolean isSelected()
	{
	    return selected;
	}
	
    /**
     * Primed is true if the collection has a bar that the user is interacting with, and
     * if the bar is currently being changed
     * @return the primed status
     */
	public boolean isPrimed()
	{
	    return primed;
	}
	    
    /**
     * Paint all the ZoomControlBars in this collection
     * @param buffer the zoom control image to paint on the control
     * @param gBuffer a buffer GC
     * @param g GC
     * @param indicatorsChanged true if the indicators for this slider have changed, false otherwise 
     */
	public void paint(Image buffer, GC gBuffer, GC g, boolean indicatorsChanged)
	{
	    if (!selected)
	    {
	        for (int i=0; i<bars.size(); i++)
	        {
	            ((ZoomControlBar)(bars.elementAt(i))).paint(gBuffer);
	        }
	        
	        g.drawImage(buffer, 0, 0);
	    }
	    else
	    {
	        if (!primed || indicatorsChanged)
	        {
	            for (int i=0; i<bars.size() - 1; i++)
	            {
	                ((ZoomControlBar)(bars.elementAt(i))).paint(gBuffer);
	            }
	            
	            primed = true;
            }
	        
	        ((ZoomControlBar)(bars.elementAt(bars.size() - 1))).hide(buffer, gBuffer);
	            
	        g.drawImage(buffer, 0, 0);
	        
	        ((ZoomControlBar)(bars.elementAt(bars.size() - 1))).paint(g);
	    }
	}    

    /**
     * A method to handle a mouse down MouseEvent on the bars within this collection
     * @param event the mouse event
     * @return boolean if a bar within this collection was clicked on
     */
    public boolean mouseDown(MouseEvent event)
    {
        selected = false;
        primed = false;
        moved = false;
        
   	    ZoomControlBar bar = getZoomControlBarContaining(event.x, event.y);
  	        
  	    if (bar != null)
  	    {
            selected = true;
            
  	        bars.removeElement(bar);
              	        
  	        bars.addElement(bar);
  	            
            if (event.button == 1)
            {
                slider.updateScale();
                
  	            slider.redraw();
  	            
  	            primed = false;
  	        }

  	        return true;
  	    }
  	    return false;
  	}
  	 
    /**
     * A method to handle a mouse up MouseEvent on the bars within this collection
     * @param event the mouse event
     * @return boolean if a bar within this collection occured under the mouse up
     */    
  	public boolean mouseUp(MouseEvent event)
  	{
  	    if (selected)
  	    {
  	        ZoomControlBar last = (ZoomControlBar)(bars.elementAt(bars.size() - 1));
  	        
  	        last.sprayZoomControlBarEvent(true);
  	        
  	        selected = false;
  	        
  	        primed = false;

            if (!moved)
            {
                int pix = (int)(slider.value2Pixel(last.getValue()));
                
                for (int i=bars.size() - 2; i >=0; i--)
                {
                    ZoomControlBar previous = (ZoomControlBar)(bars.elementAt(i));
                    
                    int prevPix = (int)(slider.value2Pixel(previous.getValue()));
                    
                    if (prevPix > pix - 3 && prevPix < pix + 3)
                    {
                        bars.removeElement(last);
                        
                        bars.insertElementAt(last, 0);
                        
                        bars.removeElement(previous);
                        
                        bars.addElement(previous);
                        
                        break;
                    }
                }
            }    

            moved = false;
            
  	        slider.redraw();
  	        
  	        return true;
  	    }
  	    
  	    return false;
  	}
  	
    
    /**
     * A method to handle a mouse move MouseEvent on the bars within this collection
     * This is only valid if the collection has been primed (a bar was clicked on)
     * @param event the mouse move event
     * @return the boolean state of the move
     */
  	public boolean mouseMove(MouseEvent event)
  	{
  	    if (primed)
  	    {
  	        moved = true;
  	        
  	        ZoomControlBar bar = ((ZoomControlBar)(bars.elementAt(bars.size() - 1)));
  	        
  	        double oldValue = bar.getValue();
  	        
  	        if (slider.getOrientation() == ZoomSlider.VERTICAL)
  	        {
  	            bar.setValue(ZoomSliderUtilities.round(slider.pixel2Value(event.y), slider.getUnitIncrement()));
  	        }    
  	        else
  	        {
  	            bar.setValue(ZoomSliderUtilities.round(slider.pixel2Value(event.x), slider.getUnitIncrement()));
  	        }
  	        
  	        slider.redraw();
  	        
  	        if (oldValue != bar.getValue())
  	        {
  	        	bar.sprayZoomControlBarEvent(false);
  	        }
  	    }
  	    
  	    return false;
  	}

    /**
     * Get the ZoomControlBar from this collection at the given coordinates
     * @param x the x coordinate
     * @param y the y coordinate
     * @return the control bar at the coordinate specified or null if none is found
     */
    public ZoomControlBar getZoomControlBarContaining(int x, int y)
	{
	    if (bars.size() > 0)
	    {
	        int i = bars.size() - 1;
    	        
	        while (i >= 0 && !((ZoomControlBar)(bars.elementAt(i))).inside(x, y))
	        {
	            i--;
	        }
    	        
	        if (i >= 0)
	        {
	            return (ZoomControlBar)(bars.elementAt(i));
	        }
	    }    
	        
	    return null;
	}

    /**
     * Get a vector of all the controls bars within this collection
     * @return a Vector of ZoomControlBars
     */
	public Vector getZoomControlBars()
	{
	    return bars;
	}
	
    /**
     * Dispose of all system resources used by this collection and the bars within it
     */
	public void dispose()
	{
		for (int i=0; i<bars.size() - 1; i++)
		{
		    ((ZoomControlBar)(bars.elementAt(i))).dispose();
		}
	}  
} 
