/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import org.eclipse.swt.SWT;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.RGB;

/**
 * A ZoomIndicator paints a bounded, coloured line on a slider to represent a value.
 * The ZoomIndicator paints between the basevalue and value of the indicator. Typically,
 * the baseindicator represents a value that does not change, whereas the value 
 * represents the variable value that the indicator is displaying.
 * 
 * @author gchristelis
 * @since 4.0.0
 *
 */
public class ZoomIndicator extends Object
{
    /**
     * The ZoomSlider this indicator resides on
     */
    protected ZoomSlider slider = null;
    
    private double value;
    private double baseValue = 0;
            
    protected Color color;
    protected Color xorColor;
    
    /**
     * Create a ZoomIndicator for this slider, with default initial value and color
     * @param slider the ZoomSlider for this indicator 
     */
    public ZoomIndicator(ZoomSlider slider)
    {
        this.slider = slider;
        
        setColor(slider.getDisplay().getSystemColor(SWT.COLOR_WIDGET_BACKGROUND));
        
        setValue(slider.getMinLimit());
        
        slider.addZoomIndicator(this);
    }
    
    /**
     * Create a ZoomIndicator for this slider with default color
     * @param slider the ZoomSlider for this indicator
     * @param value the initial indicator value
     */    
    public ZoomIndicator(ZoomSlider slider, double value)
    {
        this(slider);
        
        setValue(value);
        
        slider.redraw();
    }    
    
     
    /**
     * Create a ZoomIndicator for this slider with default color
     * @param slider the ZoomSlider for this indicator
     * @param color the initial color of this indicator
     */    
    public ZoomIndicator(ZoomSlider slider, Color color)
    {
        this(slider);
        
        setColor(color);
        
        slider.redraw();
    }    
    
    /**
     * Create a ZoomIndicator for this slider
     * @param slider the ZoomSlider for this indicator
     * @param color the initial color of this indicator
     * @param value the initial indicator value
     */    
    public ZoomIndicator(ZoomSlider slider, double value, Color color)
    {
        this(slider, value);
        
        setColor(color);
        
        slider.redraw();
    }
    
    /**
     * Set the ZoomSlider this indicator is assigned to
     * @param slider the ZoomSlider 
     */
    public void setZoomSlider(ZoomSlider slider)
    {
        this.slider = slider;
    }

    /**
     * Get the ZoomSlider this indicator is assigned to
     * @return the ZoomSlider
     */
    public ZoomSlider getZoomSlider()
    {
        return slider;
    }    
    
    /**
     * Set the base value of this indicator. The base indicator provides the second bound of the indicator, but is typically static. 
     * @param value
     */
    public void setBaseValue(double value)
    {
        if (value > slider.getMaxLimit()) 
            value = slider.getMaxLimit();
		if (value < slider.getMinLimit()) 
            value = slider.getMinLimit();

    	baseValue = value;
    }
    
    /**
     * Set this indicators value
     * @param value the indicators value
     */
    public void setValue(double value)
    {        
        if (value > slider.getMaxLimit()) 
            value = slider.getMaxLimit();
		if (value < slider.getMinLimit()) 
            value = slider.getMinLimit();

        this.value = value;
    }
    
    /**
     * Get the indicators value
     * @return the value
     */
    public double getValue()
    {
        return value;
    }    
    
    /**
     * Set the color of this indicator
     * @param color the new Color of this indicator 
     */
    public void setColor(Color color)
    {
        this.color = color;
        
        setXORColor();
    }
    
    /**
     * Get the color of this indicator
     * @return the Color of this indicator
     */
    public Color getColor()
    {
        return color;
    }    

    /**
     * Paint this ZoomIndicator on the ZoomSlider GC 
     * @param g the GC on which to paint
     * @param backBuffer the background Image of this slider
     * @param pos the position to draw this indicator (if there are multiple indicators)
     * @param w the width of this ZoomIndicator
     */
    public void paint(GC g, Image backBuffer, int pos, int w)
    {
        int pix = 0;
        int oldPix = 0;

        pix = (int)slider.value2Pixel(value);

        oldPix = (int)slider.value2Pixel(baseValue);
        
		int orientation = slider.getOrientation();
		int direction = slider.getDirection();
		int size_height = slider.getBounds().height;
		int size_width = slider.getBounds().width;
        if (backBuffer == null)
        {
            g.setBackground(xorColor);                                    

            g.setXORMode(true);
                        
            if (orientation == ZoomSlider.VERTICAL)
            {
                if (direction == ZoomSlider.INCREASING)
                {
                    g.fillRectangle(pos, pix, w, Math.abs(pix-oldPix));
                }
                else
                {
                    g.fillRectangle(pos, oldPix, w, Math.abs(pix-oldPix));
                }    
            }
            else
            {
                if (direction == ZoomSlider.INCREASING)
                {
                    g.fillRectangle(pix, pos, Math.abs(pix-oldPix), w);
                }
                else
                {
                    g.fillRectangle(oldPix, pos, Math.abs(pix-oldPix), w);
                }
            }    
            g.setXORMode(false);
        }    
        else
        {
      	    if (orientation == ZoomSlider.VERTICAL)
            {
                g.setClipping(pos, 0, w, size_height);
            }
            else
            {
                g.setClipping(0, pos, size_width, w);
            }
                
            g.drawImage(backBuffer, 0, 0);
    
            g.setClipping(0, 0, size_width, size_height);

            g.setBackground(xorColor);

            g.setXORMode(true);

            if (orientation == ZoomSlider.VERTICAL)
            {
                if (direction == ZoomSlider.INCREASING)
                {
                    g.fillRectangle(pos, pix, w, Math.abs(pix-oldPix));
                }
                else
                {
                    g.fillRectangle(pos, oldPix, w, Math.abs(pix-oldPix));
                }    
            }
            else
            {
                if (direction == ZoomSlider.INCREASING)
                {
                    g.fillRectangle(pix, pos, Math.abs(pix-oldPix), w);
                }
                else
                {
                    g.fillRectangle(oldPix, pos, Math.abs(pix-oldPix), w);
                }
            }
            g.setXORMode(false);
        }    
    }    
    
    /**
     * Set the XOR color of this indicator
     */
    public void setXORColor()
    {
        RGB rgb1 = color.getRGB();
        RGB rgb2 = slider.getBackground().getRGB();
        
        String [] color1BinaryString = {Integer.toBinaryString(rgb1.red), Integer.toBinaryString(rgb1.green), Integer.toBinaryString(rgb1.blue)};
        String [] color2BinaryString = {Integer.toBinaryString(rgb2.red), Integer.toBinaryString(rgb2.green), Integer.toBinaryString(rgb2.blue)};
        
       	for (int i=0; i<3; i++)
       	{
       		while (color1BinaryString[i].length() < 8)
       		{
       			color1BinaryString[i] = "0" + color1BinaryString[i];
       		}
       	}
       	
       	for (int i=0; i<3; i++)
       	{
       		while (color2BinaryString[i].length() < 8)
       		{
       			color2BinaryString[i] = "0" + color2BinaryString[i];
       		}
       	}

		StringBuffer [] color3BinaryStringBuffer = new StringBuffer[3];
		
		for (int i=0; i<3; i++)
		{
			color3BinaryStringBuffer[i] = new StringBuffer();
			
			char [] color1CharArray = color1BinaryString[i].toCharArray();
			char [] color2CharArray = color2BinaryString[i].toCharArray();
			
			for (int j=0; j<color1BinaryString[i].length(); j++)
			{
				int color3BitValue = 0;
				
				if (color1CharArray[j] != color2CharArray[j]) color3BitValue = 1;
				
				color3BinaryStringBuffer[i].append(String.valueOf(color3BitValue));
			}
		}
	
        xorColor = new Color(slider.getDisplay(), new RGB(Integer.parseInt(color3BinaryStringBuffer[0].toString(), 2), Integer.parseInt(color3BinaryStringBuffer[1].toString(), 2), Integer.parseInt(color3BinaryStringBuffer[2].toString(), 2)));
    }

    /**
     * Dispose of ZoomIndicator resources
     */
    public void dispose()
    {
        //indicator shouldnt dispose of this if it didnt create it    	
    	if (xorColor != null) 
            xorColor.dispose();
    }
}    
    
