/**********************************************************************
 * Copyright (c) 2005 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Vector;

import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;

/**
 * A collection of ZoomIndicators for a particular slider. The collection is
 * able to invoke the paint call on all ZoomIndicators for a particular slider.  
 * 
 * @author gchristelis
 * @since 4.0.0
 * 
 */
public class ZoomIndicatorsCollection
{
    protected static final int IDEAL_WIDTH = 7;
    protected static final int INDENT = 6;
    
    protected ZoomSlider slider;
    protected int width = IDEAL_WIDTH;
    
    private Vector indicators = new Vector();    
    private int startPoint;
    
    /**
     * ZoomIndicatorsCollection constructor
     * 
     * @param slider the ZoomSlider that the indicators are set on  
     */
    public ZoomIndicatorsCollection(ZoomSlider slider)
    {
        this.slider = slider;
    }
    
    /**
     * Add a ZoomIndicator to this collection. Duplicate ZoomIndicators will not be added. 
     * 
     * @param indicator the ZoomIndicator to add to this collection
     */
    public void add(ZoomIndicator indicator)
	{
	    if (indicator != null)
	    {
	        if (!indicators.contains(indicator))
	        {
	            indicators.addElement(indicator);
	            
	            calculatePositions();
	        }
	    }    
	}
	 
    /**
     * Remove a ZoomIndicator from this collection
     * 
     * @param indicator the ZoomIndicator to remove from the collection
     */
	public void remove(ZoomIndicator indicator)
	{
	    if (indicator != null)
	    {
	        if (indicators.contains(indicator))
	        {
	            indicators.removeElement(indicator);
	    
	            calculatePositions();
    	    }
	    }    
	}
	
    /**
     * Recalculate the positions of the indicators in this ZoomSlider. Should be called
     * if the indicator widths or number of indicators change.
     */
	public void calculatePositions()
	{
        if (indicators.size() > 0)
        {
            int base = slider.getBounds().width;
            
            if (slider.getOrientation() == ZoomSlider.HORIZONTAL)
            {
                base = slider.getBounds().height;
            }

	        width = (base - (INDENT * 2))/indicators.size();
	        
	        if (width > IDEAL_WIDTH) width = IDEAL_WIDTH;
    	        
	        startPoint = base/2 - ((width/2) * indicators.size());
        }	    
	}    
	    
    /**
     * Paint all indicators onto the ZoomSlider GC
     * 
     * @param g the GC to paint on
     * @param backBuffer the background image
     */
	public void paint(GC g, Image backBuffer)
	{
	    for (int i=0; i<indicators.size(); i++)
	    {
	        ((ZoomIndicator)(indicators.elementAt(i))).paint(g, backBuffer, startPoint + (i * width), width);
	    }
	}
	
    /**
     * Get all ZoomIndicators in this collection
     * @return a Vector of ZoomIndicators
     */
	public Vector getZoomIndicators()
	{
	    return indicators;
	}
	
    /**
     * Set the XOR colors for each ZoomIndicator.
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomIndicator#setXORColor()
     */
	public void setXORColors()
	{
	    for (int i=0; i<indicators.size(); i++)
	    {
	        ((ZoomIndicator)(indicators.elementAt(i))).setXORColor();
	    }
	}

    /**
     * Dispose of the resources used by the ZoomIndicators in this collection
     */
	public void dispose()
	{
	    for (int i=0; i<indicators.size(); i++)
	    {
	        ((ZoomIndicator)(indicators.elementAt(i))).dispose();
	    }
	}
    
    /**
     * Clear the ZoomIndicatorCollection
     *
     */
    public void clear()
    {
        for (int i=0; i<indicators.size(); i++)
        {
            ((ZoomIndicator)(indicators.elementAt(i))).dispose();
        }        
        indicators.clear();
    }
}    
    
