/**********************************************************************
 * Copyright (c) 2005, 2006 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import java.util.Vector;

import org.eclipse.hyades.ui.HyadesUIImages;
import org.eclipse.hyades.ui.internal.util.UIMessages;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.DisposeEvent;
import org.eclipse.swt.events.DisposeListener;
import org.eclipse.swt.events.MouseEvent;
import org.eclipse.swt.events.MouseListener;
import org.eclipse.swt.events.MouseMoveListener;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.Cursor;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.graphics.FontData;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.ImageData;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Composite;

import com.ibm.icu.text.DecimalFormatSymbols;
import com.ibm.icu.text.NumberFormat;

/**
 * A ruler-like widget representing a data range. The data range that the widget
 * operates over is defined by the maxLimit and minLimit configuration settings. The
 * widget will show data that is outside this range. The visible range, that is the data
 * that is currently shown on the widget, is set by the minVisible and maxVisible 
 * configuration settings.<br><br>
 * 
 * Mouse interactions allow the widget to be zoomed or transposed. Zooming narrows the
 * currently visible data range, and transposing transposes the current visible range
 * within the maximum and minimum configuration limits.<br><br>
 * 
 * Zoomsliders also support controls on the zoomslider (ZoomControlBar) and current
 * indicator lines (ZoomIndicator). The Zoomslider holds all controls and indicators
 * in collections. Both are optional. 
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class ZoomSlider extends Canvas
{   
    /**
     * The minimum end constant for this slider
     */
    public static final int MIN = 0;
    
    /**
     * The maximum end constant for this slider
     */
    public static final int MAX = 1;

    /**
     * The constant representing the top of the slider
     */
    public static final int TOP = 0;
    
    /**
     * The constant representing the bottom of the slider
     */    
    public static final int BOTTOM = 1;        

    /**
     * The minimum tick width constant
     */
    public static final int MIN_TICKWIDTH = 10;
    
    /**
     * The maximum tick width constant
     */    
    public static final int MAX_TICKWIDTH = 25;
        
    /**
     * The vertical slider orientation constant
     */
    public static final int VERTICAL = 0;
    
    /**
     * The horizontal slider orientation constant
     */    
    public static final int HORIZONTAL = 1;
    
    /**
     * The increasing slider direction constant
     */
    public static final int INCREASING = 1;
    
    /**
     * The increasing slider direction constant
     */    
    public static final int DECREASING = -1;
    
    private static final double [] LINEAR_PROGRESSION = {1, 2, 5};
        
    /**
     * A vector of all the ZoomSliderTicks on this slider. 
     */
    protected Vector tickVector = new Vector();
    
    /** 
     * Vector of ZoomSliderListeners registered on this Slider
     */ 
    private Vector listeners = new Vector(); 
    
    /**
     * Collection of Bars 
     */
    private ZoomControlBarsCollection barsCollection = new ZoomControlBarsCollection(this);

    /** 
     * Collection of Indicators 
     */
    private ZoomIndicatorsCollection indicatorsCollection = new ZoomIndicatorsCollection(this);

    /** 
     * Stores last tick to be painted with text visible
     * (used to avoid overlapping of long text when zoomSlider 
     * orientation is HORIZONTAL)
     */
    transient ZoomSliderTick lastVisible;    
    
    
    /** Min and Max possible values */
    protected double [] limit = {0, 100};
    
    /** Min and Max visible values */
    protected double [] visible = {0, 50};
    
    /** Min and Max lock status */
    protected boolean [] locked = {false, false};
    
    /** Visual orientation */
    protected int orientation = VERTICAL;
    
    /** Visual direction */
    protected int direction = INCREASING;
    
    /** Resolution - minimum possible value difference between consecutive ticks */
    protected double resolution = 1;
    
    /** Pixels between component edge and first tick */
    protected int indent = 10;
    
    /** The default font size used for the title **/
    protected static final int MINIMIZED_WIDTH = 15;
    
    /** Image used for double buffering */
    protected Image image;
    protected GC gImage;
    
    /** Image used to buffer the scale (without the bars being drawn on it).
        This is used to cover up the indicators when their value is reduced */
    protected Image plainScale;
    protected GC gPlain;
    
    protected Image titleImage;
    protected GC gTitle;
    
    private int titWidth, titHeight;
    
    private Color backgroundColor;
    private Color foregroundColor;
    private Color titleColor;
    
    private Font titleFont;
    
    protected boolean zoomable = true;
    protected boolean transposable = true;
    
    /* Current value-difference between consecutive ticks */
    transient double increment = 1;
    
    /* Order of magnitude of increment property */
    transient int incrementOOM;
    
    /* Current difference between consecutive selectable positions on the scale */
    transient double unitIncrement;
    
    /* x-pixel value of decimal point position (used only when orientation is VERTICAL) */
    transient int decimalPointPosition;
    
    /* determines whether the image buffer should be redrawn when repainting */        
    transient boolean scaleChanged = true;
    
    transient boolean indicatorsChanged;
    
    /* Zoom state */
    transient boolean zoomPrimed;
	transient boolean zoomConfigured;
	transient int zoomStart;
	transient double zoomValue;
	transient int zoomFixed;
	transient boolean zoomed;
	transient int startZoomRange;
	
    /* Transpose state */
	transient boolean transposePrimed;
	transient boolean transposeConfigured;
	transient int transposeStart;
	transient double transposeValue;
	transient double transposeRange;
    transient boolean transposed;
    
    protected String title;
    
    protected int rawOffset;
    
   	private int fontHeight;
   	
   	private Cursor cursor_zoom_vertical;
    private Cursor cursor_zoom_horizontal;
    private Cursor cursor_transpose;
    private Cursor cursor_normal;
    
    private static DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    private static NumberFormat nf = NumberFormat.getIntegerInstance();
        
    
    /** 
     * ZoomSlider constructor with no preset limit configuration
     * @param parent the parent composite to place this ZoomSlider on 
     */
	public ZoomSlider(Composite parent)
	{
		super(parent, SWT.NO_BACKGROUND);
		
		FontData fd = getFont().getFontData()[0];
		
        indent = (fd.getHeight() + 4)/2;
        fontHeight = fd.getHeight();

		backgroundColor = parent.getBackground();
		foregroundColor = parent.getForeground();
				
		if (backgroundColor == null) 
            backgroundColor = getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		if (foregroundColor == null) 
            foregroundColor = getDisplay().getSystemColor(SWT.COLOR_LIST_FOREGROUND);		
	    titleColor = foregroundColor;
		
		rawOffset = java.util.TimeZone.getDefault().getRawOffset();
		if (rawOffset < 0)
		{
		    rawOffset += 24 * 60 * 60 * 1000;
		}    

		// Listeners
		SliderMouseListener sliderMouseListener = new SliderMouseListener();
		addMouseListener(sliderMouseListener);
		
		SliderMouseMoveListener sliderMouseMoveListener = new SliderMouseMoveListener();
		addMouseMoveListener(sliderMouseMoveListener);

		SliderPaintListener sliderPaintListener = new SliderPaintListener();
		addPaintListener(sliderPaintListener);
		
		SliderControlListener sliderControlListener = new SliderControlListener();
		addControlListener(sliderControlListener);
		
		SliderDisposeListener sliderDisposeListener = new SliderDisposeListener();
		addDisposeListener(sliderDisposeListener);
		
		//cursors
		cursor_zoom_vertical = new Cursor(getDisplay(),SWT.CURSOR_SIZENS);
		cursor_zoom_horizontal = new Cursor(getDisplay(),SWT.CURSOR_SIZEWE);
		cursor_transpose = new Cursor(getDisplay(),SWT.CURSOR_HAND);
		cursor_normal = new Cursor(getDisplay(),SWT.CURSOR_ARROW);
	}
	
    /**
     * Construct a ZoomSlider with an initial limits configuration
     * @param parent the parent composite
     * @param minLimit the minimum value limit of this slider
     * @param maxLimit the maximum value limit of this slider
     * @param minVisible the minimum visible limit of this slider
     * @param maxVisible the maximum visible limit of this slider
     * @param resolution the slider resolution
     * @throws ZoomSliderConfigurationException if the configuration limits presented are not valid for this zoomslider 
     */
	public ZoomSlider(Composite parent, double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    this(parent);
	    
	    configure(minLimit, maxLimit, minVisible, maxVisible, resolution);
	}    
	
    /**
     * Instantiate and add a ZoomControlBar to this ZoomSlider with a preset initial value
     * @param value the initial value of the control bar
     */
    public void addZoomControlBar(double value)
    {
        new ZoomControlBar(this, value);
        
        updateScale();
    }
    
    /**
     * Add a provided ZoomControlBar to this ZoomSlider
     * @param bar the ZoomControlBar to add to this slider's control collection
     */
    public void addZoomControlBar(ZoomControlBar bar)
    {
        barsCollection.add(bar);
        
        updateScale();
    }

    /**
     * Remove a provided ZoomControlBar to this ZoomSlider
     * @param bar the ZoomControlBar to remove from this slider's control collection
     */
    public void removeZoomControlBar(ZoomControlBar bar)
    {
        barsCollection.remove(bar);
        
        updateScale();
    }
    
    /**
     * Query whether this slider has any control bars
     * @return true if this slider has at least one control bar, false otherwise
     */
    public boolean hasBars()
    {
        return barsCollection.hasBars();
    }
    
    /**
     * Get the ZoomControlBarsCollection for this slider
     * @return an instance of ZoomControlBarsCollection referencing all the ZoomControlBars for this slider
     */
    public ZoomControlBarsCollection getZoomControlBarsCollection()
    {
        return barsCollection;
    }    
    
    /**
     * Add a Zoom Indicator to this slider's indicators collection
     * @param indicator ZoomIndicator to add to the collection
     */
    public void addZoomIndicator(ZoomIndicator indicator)
    {
        indicatorsCollection.add(indicator);
        
        updateScale();
    }
    
    /**
     * Remove a Zoom Indicator from this slider's indicators collection
     * @param indicator ZoomIndicator to remove
     */
    public void removeIndicator(ZoomIndicator indicator)
    {
        indicatorsCollection.remove(indicator);
        
        updateScale();
    }    
    
    
    /**
     * Get this ZoomSliders indicators collection
     * @return this slider's ZoomIndicatorsCollection instance
     */
    public ZoomIndicatorsCollection getZoomIndicatorsCollection()
    {
        return indicatorsCollection;
    }    
    
    /**
     * Get this ZoomSliders vector of ZoomSliderTicks
     * @return a vector of ZoomSliderTick instances
     */
    public Vector getTickVector()
    {
        return tickVector;
    }    
	
	/**
     * Return the first ZoomSliderTick instance in the list of ticks
	 * @return the first ZoomSliderTick, or null if the list is empty
	 */
	public ZoomSliderTick getFirstTick()
	{
		if (tickVector.size() == 0) return null;
		return (ZoomSliderTick)tickVector.get(0);
	}

	/**
     * Return the last ZoomSliderTick instance in the list of ticks
     * @return the last ZoomSliderTick or null if the list is empty
     */
	public ZoomSliderTick getLastTick()
	{
		if (tickVector.size() == 0) return null;
		return (ZoomSliderTick)tickVector.get(tickVector.size()-1);
	}
	
	private void reinitialiseGraphics() 
	{
        if (image != null) image.dispose();
        if (gImage != null) gImage.dispose();
        if (plainScale != null) plainScale.dispose();
        if (gPlain != null) gPlain.dispose();
        
        if (getDisplay() == null) return;
        if (getBounds().width < 1) return;
		if (getBounds().height < 1) return;
        
        // Recreate the double-buffering image.
        image = new Image(getDisplay(), getBounds().width, getBounds().height);
        gImage = new GC(image);

        // Recreate the plainScale image
        plainScale = new Image(getDisplay(), getBounds().width, getBounds().height);
        gPlain = new GC(plainScale);

		// Find new decimal point position
		findDecimalPointPosition();
		
        createTicks();
	}

    /**
     * Paint this ZoomSlider control
     * @param e the associated repaint event
     */
    public void paint(PaintEvent e)
    {
    	if (isDisposed()) {
    		return;	
    	}
    	
    	if (!minimized)
    	{
	        if (gImage == null) 
	        {
	            reinitialiseGraphics();
	        }
	        
	        if (scaleChanged)
	        {
	            paintTicks(gImage);
	            
	            paintTicks(gPlain);
	            
	            scaleChanged = false;
	            
	            indicatorsCollection.paint(gImage, null);
	        }
	                
	        if (indicatorsChanged)
	        {
	            indicatorsCollection.paint(gImage, plainScale);
	        }    
	        
	        barsCollection.paint(image, gImage, e.gc, indicatorsChanged);
	        
	        indicatorsChanged = false;
    	}
    	else
    	{
	        if (gImage == null) 
	        {
	            reinitialiseGraphics();
	        }
	        
	        paintMinimized(gImage);
	        paintMinimized(gPlain);
	        paintMinimizedTitle(gImage);
	        paintMinimizedTitle(gPlain);
	        e.gc.drawImage(image, 0, 0);
    	}
    }
    
    /**
     * Force a redraw of this ZoomSlider and update the scale of this slider
     */
    public void updateScale()
    {
        scaleChanged = true;
        
        redraw();
    }
    
    /**
     * Recreate the scale on this zoom slider. Reinitialize all ticks on the slider and force a redraw
     */
    public void recreateScale()
    {
        createTicks();
    
        updateScale();
    }
    
    /**
     * Update the indicators on this slider
     */
    public void updateIndicators()
    {
        indicatorsChanged = true;
        
        redraw();
    }    

    /**
     * Force a repaint of this component
     */
    public void externalRepaint()
    {
        scaleChanged = true;
        
        redraw();
    }
    
    /**
     * Resize the slider ticks and indicators of this slider
     */
    public void resizeScale()
    {
    	reinitialiseGraphics();
    	indicatorsCollection.calculatePositions();
    	recreateScale();
    }    
    
    /* (non-Javadoc)
     * @see org.eclipse.swt.widgets.Composite#layout()
     */
    public void layout()
    {
        super.layout();
        
        indicatorsCollection.calculatePositions();
        
        // Extend the range of visible values if the layout action would cause the 
        // maxTickWidth property to be exceeded
        if (isMaxTickWidthExceeded())
        {
            visible[MAX] = visible[MIN] + (getPixelRange()/MAX_TICKWIDTH * resolution);
            
            if (visible[MAX] > limit[MAX])
            {
                visible[MAX] = limit[MAX];
                
                visible[MIN] = visible[MAX] - (getPixelRange()/MAX_TICKWIDTH * resolution);
            }    
            
            updateScale();
        }    
    }
            
    /* (non-Javadoc)
     * @see org.eclipse.swt.widgets.Canvas#setFont(org.eclipse.swt.graphics.Font)
     */
    public void setFont(Font font)
    {
		FontData fd = font.getFontData()[0];
		
        indent = (fd.getHeight() + 4)/2;
        fontHeight = fd.getHeight();

        super.setFont(font);
    }    
    
    /** 
     * Get the width or height of the visible area. 
     * @return the height of the HORIZONTAL slider or the width of the VERTICAL slider
     */
    protected int getPixelRange()
    {
        if (orientation == HORIZONTAL)
        {
            return getBounds().width - (indent * 2);
        }
        else
        { 
            return getBounds().height - (indent * 2);
        }    
    }
    
    protected double getZeroPixelValue()
    {
        if (direction == DECREASING)
        {
            return visible[MIN];
        }
        else
        {
            return visible[MAX];
        }
    }    
    
    // Get range of visible values 
    protected double getVisibleValueRange()
    {
        return Math.abs(visible[MAX] - visible[MIN]);
    }    

    // GET SINGLE PIXEL RANGE , Get value range across a single pixel
    protected double getSinglePixelRange()
    {
        return getVisibleValueRange() / getPixelRange();
    }    

    /**
     * Convert a given pixel position to its corresponding slider value
     * @param pixelPos the pixel position on this slider
     * @return the corresponding value at this pixel position
     */
    public double pixel2Value(double pixelPos)
    {
        return getZeroPixelValue() - (direction * ((pixelPos - indent) * getSinglePixelRange()));
    }    
    
    /**
     * Convert a given slider value to its corresponding pixel value
     * @param value the slider position on this slider
     * @return the corresponding pixel position
     */
    public double value2Pixel(double value)
    {
        return (((getZeroPixelValue() - value) / direction) / getSinglePixelRange()) + indent;        
    }    

    /** 
     * Get the value spacing between ticks on the slider
     * @return the space between ticks 
     */
    public double getIncrement()
    {
        return increment;
    }

    /** 
     * Get the order of magnitude of the value spacing between ticks
     * @return an int representing the order of magnitude 
     */
    public int getIncrementOOM()
    {
        return incrementOOM;
    }
        
    /** 
     * Get the value spacing between consecutive selectable positions on the scale 
     * @return the value spacing
     */
    public double getUnitIncrement()
    {
        return unitIncrement;
    }    
    
    /** 
     * Get the pixel position of the decimal point
     * @return the pixel position of the decimal point 
     */
    public int getDecimalPointPosition()
    {
        return decimalPointPosition;
    }    
        
    /**
     * Get the current slider indent
     * @return the slider indent in pixels
     */
    public int getIndent()
    {
        return indent;
    }
    
    /**
     * Get the current font height that is used to render text on the slider
     * @return the fontHeight in pixels
     */
    public int getFontHeight()
    {
    	return fontHeight;
    }    
    
    /** 
     * Tells some visiting thread whether or not the scale is currently changing 
     * @return a boolean of true if the slider is currently changing
     */
    public boolean isChanging()
    {
        if (zoomConfigured)
        {
            return true;
        }
        
        if (transposeConfigured)
        {
            return true;
        }
        
        return false;
    }    
    
    /** 
     * Find the pixel position of the decimal point 
     */
    protected void findDecimalPointPosition()
    {
        if (orientation == VERTICAL)
        {
            int tail = 0;
            
            if (increment < 1)
            {
                GC gcTemp = new GC(this);
                String zeroString = nf.format(0);
                tail =  Math.abs(incrementOOM) * gcTemp.stringExtent(zeroString).x + gcTemp.stringExtent(""+dfs.getDecimalSeparator()).x;
            }    
            
            decimalPointPosition = getBounds().width - tail - 5;
        }
    }
    
    /**
     * A convenience class to store a tick value range from the slider
     */
    class TickValueRange
    {
        /**
         * The current minimum values that define this tick value range
         */
        public double min;
        
        /**
         * The current maximum values that define this tick value range
         */        
        public double max;        
        
        /**
         * Construct a new tick value range, ensuring the the range is within the current slider resolution
         */
        public TickValueRange()
        {
            min = getSinglePixelRange() * MIN_TICKWIDTH;
            if (min < resolution) 
                min = resolution;
            
            max = getSinglePixelRange() * MAX_TICKWIDTH;
            if (max < resolution) 
                max = resolution;
        }
        
        /**
         * Return a bolean indicating whether the current slider increment is within the tick value range
         * @return <code>true</code> if the increment is within the range, <code>false</code> otherwise.
         */
        public boolean outside()
        {
            return (increment < min || increment > max);
        }
    }    

    /** 
     * Calculate the value spacing between consecutive ticks 
     */
    protected void calculateIncrement()
    {
        TickValueRange tvr = new TickValueRange();
        
        if (tvr.outside())
        {
            int maxTickCount = getPixelRange() / MIN_TICKWIDTH;

            increment = getVisibleValueRange() / maxTickCount;
            
            int orderOfMagnitude = ZoomSliderUtilities.calculateOOM(increment);
            
            increment = LINEAR_PROGRESSION[0] * Math.pow(10, orderOfMagnitude);
            
            while (increment < tvr.min)
            {
                for (int i=0; i<3; i++)
                {
                    increment = LINEAR_PROGRESSION[i] * Math.pow(10, orderOfMagnitude);
                    
                    if (increment >= tvr.min) break;
                }
                orderOfMagnitude++;
            }
            
            incrementOOM = ZoomSliderUtilities.calculateOOM(increment);
            
            findDecimalPointPosition();
        }    
    }
    
    /** 
     * Create the vector of ticks 
     */
    protected void createTicks()
    {
        calculateIncrement();
        
        // Empty the tick vector
        tickVector.removeAllElements();
        
        // Refill the tick vector
        double tickValue = ZoomSliderUtilities.round(visible[MIN] - increment, increment);
        
        int loop = 0;
        
        while (tickValue <= visible[MAX] + increment)
        {            
            tickVector.addElement(new ZoomSliderTick(this, tickValue));

            tickValue += increment;
            
            tickValue = ZoomSliderUtilities.round(tickValue, increment);
            
            loop++;
            
            if (loop > 500)
            {
                break;
            }    
        }
        
        // Depending on the pixel spacing between ticks find a reasonable number of 
        // intermediate selecteable points (5 or 10)
		int selectablePoints = 10;
    	int tickSpacing = (int)(getPixelRange()/(getVisibleValueRange()/increment));
		if (tickSpacing < 20)
		{
		    selectablePoints = 5;
		}
		
        // Set the unit increment based on the intermediate selectable points
		unitIncrement = increment/selectablePoints;
		if (unitIncrement < resolution)
		{
		    unitIncrement = resolution;
		}
    }
    
    private void paintMinimized(GC g)
    {
    	// Paint background
    	g.setBackground(backgroundColor);        
        g.fillRectangle(0,0,getBounds().width, getBounds().height);

        g.setForeground(foregroundColor);
        g.drawRectangle(0, 0, getBounds().width - 1, getBounds().height - 1);
    }
    

    /** 
     * Paint the slideticks on the slider ticks
     * @param g the GC to paint on 
     */
    private void paintTicks(GC g)
    {
        g.setBackground(backgroundColor);
        
        g.fillRectangle(0,0,getBounds().width, getBounds().height);
        
        g.setForeground(foregroundColor);
        
        // initialise lastVisible
        lastVisible = null;        

        paintTitle(g);
        
        try
        {
            // Paint each tick
            for (int i=0; i<tickVector.size(); i++)
            {
                ((ZoomSliderTick)(tickVector.elementAt(i))).paintLine(g);
            }
            for (int i=0; i<tickVector.size(); i++)
            {
                ((ZoomSliderTick)(tickVector.elementAt(i))).paintText(g);
            }
        }
        catch (NullPointerException e) 
        {
//            UiPlugin.DBG.warning("tickVector elements null");
            return;
        }        
        
        /*switch (orientation)
        {
            case VERTICAL:
                g.drawLine(getBounds().width - 8, indent, getBounds().width, indent);
                g.drawLine(getBounds().width - 8, getBounds().height - indent, getBounds().width, getBounds().height - indent);
                g.drawLine(0, indent, 8, indent);
                g.drawLine(0, getBounds().height - indent, 8, getBounds().height - indent);
                break;
                
            case HORIZONTAL:
                g.drawLine(indent, 0, indent, 8);
                g.drawLine(getBounds().width - indent, 0, getBounds().width - indent, 8);
                g.drawLine(getBounds().width - indent, getBounds().height - 8, getBounds().width - indent, getBounds().height);
                g.drawLine(indent, getBounds().height - 8, indent, getBounds().height);
                break;
        }    */    

        // Draw the border 
        g.setForeground(foregroundColor);
        g.drawRectangle(0, 0, getBounds().width - 1, getBounds().height - 1);
        
        if (isLocked(MAX))
        	paintMaxLock(g);
        if (isLocked(MIN))        
        	paintMinLock(g);

    }
    
    private void paintMinimizedTitle(GC g)
    {
    	if (titleImage == null)
        {
            createTitle(g);
        }    
            
        if (titleImage != null)
        {
            if (orientation == HORIZONTAL)
            {
                g.drawImage(titleImage, getBounds().width - titWidth - (indent + 15), getBounds().height - titHeight-1);
            }
            else
            {
        	    g.drawImage(titleImage, 1, indent + 15);
            }
        }    	
    }
    
    private void paintTitle(GC g)
    {
        if (titleImage == null)
        {
            createTitle(g);
        }    
            
        if (titleImage != null)
        {
            if (orientation == HORIZONTAL)
            {
                g.drawImage(titleImage, getBounds().width - titWidth - (indent + 15), getBounds().height - (titHeight + 2));
            }
            else
            {
        	    g.drawImage(titleImage, 4, indent + 15);
            }
        }    
    }
    
    Image lockImage;
    
    private void createLock(GC g)
    {   	
    	lockImage = HyadesUIImages.INSTANCE.getImage(HyadesUIImages.IMG_ZOOMSLIDER_LOCK);    	
    }    
      
    private void paintMaxLock(GC g)
    {
        if (lockImage == null)
        {
            createLock(g);
        }    
            
        if (lockImage != null)
        {
            if (orientation == HORIZONTAL)
            {
                g.drawImage(lockImage, getBounds().width - (indent + 6), getBounds().height - 11);
            }
            else
            {
        	    g.drawImage(lockImage, 2, 2);
            }
        }    
    }
    
    private void paintMinLock(GC g)
    {
        if (lockImage == null)
        {
            createLock(g);
        }    
            
        if (lockImage != null)
        {
            if (orientation == HORIZONTAL)
            {
                //g.drawImage(titleImage, getSize().width/2 - titWidth/2, getSize().height - (titHeight + 5), null);
                g.drawImage(lockImage, 2, getBounds().height - 11);
            }
            else
            {
        	    //g.drawImage(titleImage, 5, getSize().height/2 - titWidth/2, null);
        	    
        	    g.drawImage(lockImage,  2, getBounds().height - 11);
            }
        }    
    }

    
    private void createTitle(GC g)
    {
        if (title != null)
        {
            if (!title.equals(""))
            {
				g.setFont(titleFont);
                titWidth = g.stringExtent(title).x;
                titHeight = g.stringExtent(title).y;
                g.setFont(getFont());
                                
                // Create Image Buffer for title
                titleImage = new Image(getDisplay(), titWidth, titHeight);
                gTitle = new GC(titleImage);
                                
                gTitle.setBackground(backgroundColor);
                                
                gTitle.fillRectangle(0, 0, titWidth, titHeight);
                
                gTitle.setForeground(titleColor);

				gTitle.setFont(titleFont);				
				
                gTitle.drawString(title, 0, 0);
                    
                g.setForeground(foregroundColor);
                
                if (orientation == VERTICAL)
                {
        	        titleImage = rotateImage();
                }
            }    
        }    
    }    
    
    private Image rotateImage()
    {
    	ImageData titleData = titleImage.getImageData();
    	
    	Image temp = new Image(getDisplay(), titleData.height, titleData.width);
    	
    	ImageData tempData = temp.getImageData();
    	
    	for (int h=0; h<titleData.height; h++)
    	{
    		for (int w=0; w<titleData.width; w++)
    		{
    			tempData.setPixel(h, tempData.height - 1 - w, titleData.getPixel(w,h));
    		}
    	}
    	
    	return new Image(getDisplay(), tempData);
    }
    
	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Composite#computeSize(int, int, boolean)
	 */
	public Point computeSize(int wHint, int hHint, boolean changed)
	{
		Point standard = super.computeSize(wHint, hHint, changed);
		Point size = new Point(standard.x, standard.y); 
		
		GC gcTemp = new GC(this);				
		
   	    if (orientation == VERTICAL)
	    {   	    	
   	    	if (minimized)
   	    	{
   	    		size.x = MINIMIZED_WIDTH;
   	    	}
   	    	
   	    	else
   	    	{
   	    		if (wHint == SWT.DEFAULT)
				{
		            // Find out whether the max or min value is the "longest"
		            String maxTick = (new ZoomSliderTick(this, limit[MAX] + resolution)).representation;
		            String minTick = (new ZoomSliderTick(this, limit[MIN] - resolution)).representation;
		
		            int biggestMax = gcTemp.stringExtent(maxTick).x;
					int biggestMin = gcTemp.stringExtent(minTick).x;            
		
		            int biggest = biggestMax;
		            if (biggestMin > biggest)
		            {
		                biggest = biggestMin;
		            }
		
			        size.x = biggest + 20;
				}
				else
				{
					size.x = wHint;
				}
			
				if (hHint != SWT.DEFAULT)
				{
					size.y = hHint;
				}
   	    	}
	    }
        else
	    {
        	if (!minimized)
        	{
		    	if (hHint == SWT.DEFAULT)
		    	{
		    		size.y = gcTemp.getFontMetrics().getAscent() + 20;
		    	}
		    	else
		    	{
		    		size.y = hHint;
		    	}
        	}
        	else
        		size.y = MINIMIZED_WIDTH;
	    	
	    	if (wHint != SWT.DEFAULT)
	    	{
	    		size.x = wHint;
	    	}
	    }
	    
		gcTemp.dispose();
		
	    return size;
	}
		
    /**
     * Configure a slider limits and resolution. This method superseeds the getMaxLimit, getMinLimit, getMaxVisible, getMinVisible as 
     * it allows you to configure multiple limits at a single time. Only once all the limits have been set will the configuration be checked.
     * @param minLimit the minimum limit for this slider
     * @param maxLimit the maximum limit for this slider
     * @param minVisible the minimum visible limit for this slider 
     * @param maxVisible the maximum visible limit for this slider
     * @param resolution the resolution for this slider
     * @throws ZoomSliderConfigurationException if the new configuration is invalid
     */
	public void configure(double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    checkConfiguration(minLimit, maxLimit, minVisible, maxVisible, resolution);
		
	    limit[MIN] = minLimit;
	    limit[MAX] = maxLimit;
	    visible[MIN] = minVisible;
	    visible[MAX] = maxVisible;
	    this.resolution = resolution;
	    
	    try
	    {
	        recreateScale();
	    }
	    catch (NullPointerException e){}
	} 
	
	private void checkConfiguration(double minLimit, double maxLimit, double minVisible, double maxVisible, double resolution) throws ZoomSliderConfigurationException
	{
	    if (minLimit >= maxLimit)
	    {
	    	throw new ZoomSliderConfigurationException(UIMessages._70);
	    }
	        
	    if (minVisible >= maxVisible)
	    {
	    	throw new ZoomSliderConfigurationException(UIMessages._71);
	    }
	    
	    if (minVisible < minLimit)
	    {
	    	throw new ZoomSliderConfigurationException(UIMessages._72);
	    }
	    
	    if (maxVisible > maxLimit)
	    {
	    	throw new ZoomSliderConfigurationException(UIMessages._73);
	    }
	    
		if (resolution > (maxLimit - minLimit))
		{
			throw new ZoomSliderConfigurationException(UIMessages._74);
		}
		
		if (resolution <= 0)
		{
			throw new ZoomSliderConfigurationException(UIMessages._75);
		}

		double oomZeroVersion = resolution * Math.pow(10, -1 * ZoomSliderUtilities.calculateOOM(resolution));
		if (oomZeroVersion != 1 && oomZeroVersion != 2 && oomZeroVersion != 5)
		{
			throw new ZoomSliderConfigurationException(UIMessages._76);
		}
	}
	
    /**
     * Set this slider's title
     * @param title a new string to be used as the title of this slider
     */
	public void setTitle(String title)
	{
	    this.title = title;
	    
	    titleImage = null;
	}
	
    /**
     * Get this slider's title
     * @return a string representation of this sliders title
     */
	public String getTitle()
	{
	    return title;
	}    
	
	/**
     * Set the color used for the slider title text
	 * @param color the new title color
	 */
	public void setTitleColor(Color color)
	{
	    titleColor = color;
	    
	    titleImage = null;
	}
	
	/**
     * Get the color used for the slider title text
	 * @return the title color
	 */
	public Color getTitleColor()
	{
	    return titleColor;
	}
	
	/**
     * Set the Font to be used to paint the slider title on the slider
	 * @param font the new Font to be used to pain the slider title on the slider
	 */
	public void setTitleFont(Font font)
	{
		titleFont = font;

	    titleImage = null;
	}
	
	/**
     * Get the current Font used to pain the slider title on the slider
	 * @return the Font used to paint the slider title
	 */
	public Font getTitleFont()
	{
		return titleFont;
	}    
	
	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setForeground(org.eclipse.swt.graphics.Color)
	 */
	public void setForeground(Color color)
	{
		super.setForeground(color);
		
		if (color != null)
		{
		    foregroundColor = color;
		}

	    titleImage = null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#getForeground()
	 */
	public Color getForeground()
	{
	    return foregroundColor;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#setBackground(org.eclipse.swt.graphics.Color)
	 */
	public void setBackground(Color color)
	{
		super.setBackground(color);
		
		if (color != null)
		{
		    backgroundColor = color;
		    indicatorsCollection.setXORColors();
		}

	    titleImage = null;
	}

	/* (non-Javadoc)
	 * @see org.eclipse.swt.widgets.Control#getBackground()
	 */
	public Color getBackground()
	{
		if (backgroundColor != null) {
	    	return backgroundColor;
		} else {
			return getDisplay().getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		}
	}

	/**
     * Set the maximum visible limit for this slider. The slider will data between the minimum and maximum visible limits
     * @param d the maximum visible limit of this slider
     * @throws ZoomSliderConfigurationException if the new visible maximum results in an invalid slider configuration
     */   
	public void setMaxVisible(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], visible[MIN], d, resolution);

	    visible[MAX] = d;
        
        recreateScale();
    }
    
	/**
     * Get the maximum visible for this slider. The slider will data between the minimum and maximum visible limits
     * @return the maximum limit of this slider
     */
    public double getMaxVisible()
    {
        return visible[MAX];
    }    
	
    /**
     * Set the minimum visible limit for this slider. The slider will data between the minimum and maximum visible limits
     * @param d the minimum visible limit of this slider
     * @throws ZoomSliderConfigurationException if the new visible minimum results in an invalid slider configuration
     */    
	public void setMinVisible(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], d, visible[MAX], resolution);

        visible[MIN] = d;
        
        recreateScale();
    }    

    /**
     * Get the minimum visible for this slider. The slider will data between the minimum and maximum visible limits
     * @return the minimum limit of this slider
     */
    public double getMinVisible()
    {
        return visible[MIN];
    }    

    /**
     * Set the maximum limit for this slider. The slider will display no values beyond this limit
     * @param d the maximum limit of this slider
     * @throws ZoomSliderConfigurationException if the new maximum results in an invalid slider configuration
     */
	public void setMaxLimit(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], d, visible[MIN], visible[MAX], resolution);

	    limit[MAX] = d;
        
        recreateScale();
    }    

    /**
     * Get the maximum limit for this slider. The slider will display no values below this limit
     * @return the maximum limit of this slider
     */
    public double getMaxLimit()
    {
        return limit[MAX];
    }    

    /**
     * Set the maximum limit for this slider. The slider will display no values below this limit
     * @param d the minimum limit of this slider
     * @throws ZoomSliderConfigurationException if the new minimum results in an invalid slider configuration
     */
	public void setMinLimit(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(d, limit[MAX], visible[MIN], visible[MAX], resolution);

	    limit[MIN] = d;

        recreateScale();    
    }    

    /**
     * Get the minimum limit for this slider. The slider will display no values below this limit
     * @return the minimum limit of this slider
     */
    public double getMinLimit()
    {
        return limit[MIN];
    }    

    /**
     * Set the resolution of this Zoom slider.
     * @param d the resolution value you wish to set for this slider
     * @throws ZoomSliderConfigurationException if this resolution results in an invalid Zoom Slider configuration
     */
	public void setResolution(double d) throws ZoomSliderConfigurationException
	{
		checkConfiguration(limit[MIN], limit[MAX], visible[MIN], visible[MAX], d);		
	    resolution = d;
        scaleChanged = true;
        findDecimalPointPosition();
	    redraw();       
    }
    
    /**
     * Get the current graph resolution
     * @return the resolution double
     */
    public double getResolution()
    {
        return resolution;
    }    
	
    /**
     * Set the direction of this slider
     * @param direction the direction constant, either INCREASING or DESCREASING
     * @throws ZoomSliderConfigurationException if altering the direction of the slider results in an invalid slider configuration
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#INCREASING
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#DECREASING       
     */
	public void setDirection(int direction) throws ZoomSliderConfigurationException
	{
		if (direction == INCREASING || direction == DECREASING)
		{
	    	this.direction = direction;
	        recreateScale();
		}
		else
		{
			throw new ZoomSliderConfigurationException(UIMessages._77);
		}
	}

    /**
     * Get the direction constant, either increasing or decreasing
     * @return the slider direction constant
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#INCREASING
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#DECREASING      
     */
	public int getDirection()
	{
	    return direction;
	}    
    
	/**
     * Set the orientation of this slider
	 * @param orientation the orientation constant for this slider, either VERTICAL or HORIZONTAL
	 * @throws ZoomSliderConfigurationException if this orientation causes the current slider configuration to be invalid
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#HORIZONTAL
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#VERTICAL      
	 */
	public void setOrientation(int orientation) throws ZoomSliderConfigurationException
	{
		if (orientation == HORIZONTAL || orientation == VERTICAL)
		{
	        this.orientation = orientation;        
        	recreateScale();
		}
		else
		{
			throw new ZoomSliderConfigurationException(UIMessages._78);
		}			
	}    
	    
    /** 
     * Get the current slider orientation. The orientation is either horizontal or vertical, stipulated by the HORIZONTAL or VERTICAL constants
     * @return an integer constant, either VERTICAL or HORIZONTAL
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#HORIZONTAL
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#VERTICAL      
     */
    public int getOrientation()
	{
	    return orientation;
	}
		
    /**
     * Set the locked status of an end of the slider
     * @param end a constant defining which end is queried for lock status
     * @param lock a boolean of true if the end should be locked, and false otherwise
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#MIN
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#MAX
     */
	public void lock(int end, boolean lock)
	{
	    locked[end] = lock;
        scaleChanged = true;
	}
	
    /**
     * Return whether the minimum or maximum of a slider is locked
     * @param end a constant defining which end is queried for lock status
     * @return a boolean of true if the slider end is locked, or false otherwise
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#MIN
     * @see org.eclipse.hyades.ui.widgets.zoomslider.ZoomSlider#MAX
     */
	public boolean isLocked(int end)
	{
	    return locked[end];
	}
	
    /**
     * Set whether this slider is zoomable or not
     * @param b a boolean indicating whether this slider is zoomable or not 
     */
	public void setZoomable(boolean b)
	{
	    zoomable = b;
	}
	
    /**
     * Return wheter this zoom slider is zoomable
     * @return true if the slider is zoomable or false if not
     */
	public boolean isZoomable()
	{
	    return zoomable;
	}
	
    /**
     * Set whether this slider is transposable or not
     * @param b a boolean indicating whether this slider is transposable 
     */
	public void setTransposable(boolean b)
	{
	    transposable = b;
	}
	
    /**
     * Return wheter this zoom slider is transposable
     * @return true if the slider is transposable or false if not
     */
	public boolean isTransposable()
	{
	    return transposable;
	}    

    /**
     * Set the last visible slider tick that is present on the slider
     * @param tick to set to the last ZoomSliderTick
     */
	public void setLastVisible(ZoomSliderTick tick)
	{
	    lastVisible = tick;
	}

    /**
     * Get the last visible slider tick that is present on the slider
     * @return the last ZoomSliderTick
     */
	public ZoomSliderTick getLastVisible()
	{
	    return lastVisible;
	}	
	
	private boolean isMaxTickWidthExceeded()
	{
	    if (getVisibleValueRange()/resolution < getPixelRange()/MAX_TICKWIDTH)
	    {
	        return true;
	    }
	    return false;
	}
	
    /**
     * Get the minimum pixel range relative to the resolution and current pixel range of this slider
     * @return the current minimum slider range
     */
	public double getMinRange()
	{
	    return getPixelRange()/MAX_TICKWIDTH * resolution;
	}    
	    
    // Convenience method, converts from zero to one and visa-versa 
    private int complement(int i)
    {
        if (i == 0) return 1;
        return 0;
    }
    
	private void primeZoom(MouseEvent event)
	{
	    if (orientation == VERTICAL)
	    {
	        zoomStart = event.y;
	    }
	    else
	    {
	        zoomStart = event.x;
	    }
	    
	    zoomValue = pixel2Value(zoomStart);
    	    
	    zoomPrimed = true;	    
	}
	
	private void configureZoom(MouseEvent event)
	{
        int zoomPosition = 0;
        
        if (orientation == VERTICAL)
        {
            zoomPosition = event.y;
        }
        else
        {
            zoomPosition = event.x;
        }
        
        if (Math.abs(zoomPosition - zoomStart) > 2)
        {
            if (!locked[MIN] && !locked[MAX])
            {
                if (zoomPosition > zoomStart)
                {
    	            if (direction == INCREASING)
    	            {
    		            zoomFixed = MAX;
    	            }
    	            else
    	            {
    		            zoomFixed = MIN;
    	            }
                }
                else
                {
    	            if (direction == INCREASING)
    	            {
    		            zoomFixed = MIN;
    	            }
    	            else
    	            {
    		            zoomFixed = MAX;
    	            }
                }
                
                zoomPrimed = false;

                zoomConfigured = true;
            }
            else
            {
                if (locked[MAX] && locked[MIN])
                {
                    zoomPrimed = false;
                    
                    zoomConfigured = false;
                }
                else
                {
                    if (locked[MAX])
                    {
                        zoomFixed = MAX;
                    }
                    else
                    {
                        zoomFixed = MIN;
                    }
                    
                    zoomPrimed = false;
                    
                    zoomConfigured = true;
                }
            }
            
            startZoomRange = Math.abs(zoomStart - (int)value2Pixel(visible[zoomFixed])); 
        }    
    }
	    

	private void zoom(MouseEvent event)
	{
        int pixel = event.y;

	    if (orientation == HORIZONTAL)
	    {
	        pixel = event.x;
	    }    
	    
	    double diff = Math.abs(pixel - value2Pixel(visible[zoomFixed]));
	    
	    if (diff > 0)
	    {
	        double percentage = getPixelRange()/diff;
	        
	        double temp = visible[complement(zoomFixed)];
	        
	        if (zoomFixed == MIN)
	        {
                visible[complement(zoomFixed)] = visible[zoomFixed] + (percentage * Math.abs(visible[zoomFixed] - zoomValue));
	            
	            if (visible[complement(zoomFixed)] > limit[complement(zoomFixed)])
	            {
	                visible[complement(zoomFixed)] = limit[complement(zoomFixed)];
	            }    
	        }
	        else
	        {
                visible[complement(zoomFixed)] = visible[zoomFixed] - (percentage * Math.abs(visible[zoomFixed] - zoomValue));
	            
	            if (visible[complement(zoomFixed)] < limit[complement(zoomFixed)])
	            {
	                visible[complement(zoomFixed)] = limit[complement(zoomFixed)];
	            }    
	        }
	        
	        if (isMaxTickWidthExceeded() && !isContracting(pixel))
	        {
	            visible[complement(zoomFixed)] = temp;
	        } 
	        
	        if (visible[complement(zoomFixed)] != temp)
	        {
	        	zoomed = true;
	        	
	        	recreateScale();
	        	
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.ZOOMED, false));
	            }
	        }
        }    
	}    
	
	private boolean isContracting(int pixel)
	{
		double newZoomRange = Math.abs(pixel - (int)value2Pixel(visible[zoomFixed]));
		
		return newZoomRange <= startZoomRange;
	}
	
    private void primeTranspose(MouseEvent event)
    {
	    if (!(locked[MIN] || locked[MAX]))
	    {
	        if (orientation == VERTICAL)
	        {
	            transposeStart = event.y;
	        }
	        else
	        {
	            transposeStart = event.x;
	        }
    	    
	        transposePrimed = true;
	    }    
    }    
    
    private void configureTranspose(MouseEvent event)
    {
        int transposePosition = 0;
        
   	    if (orientation == VERTICAL)
	    {
	        transposePosition = event.y;
	    }
	    else
	    {
	        transposePosition = event.x;
	    }

        if (Math.abs(transposePosition - transposeStart) > 2)
        {
            transposeValue = pixel2Value(transposePosition);
            
            transposeRange = getVisibleValueRange();

            transposePrimed = false;

            transposeConfigured = true;
        }    
    }
    
   	private void transpose(MouseEvent event)
	{
		double oldVisibleMin = visible[MIN];

        int pixel = event.y;

	    if (orientation == HORIZONTAL)
	    {
	        pixel = event.x;
	    }    
	    
        double diff = Math.abs(pixel - value2Pixel(visible[MIN]));
        
	    visible[MIN] = transposeValue - diff * getSinglePixelRange();
	    
	    if (visible[MIN] < limit[MIN])
	    {
	        visible[MIN] = limit[MIN];
	    }
	    
	    visible[MAX] = visible[MIN] + transposeRange;

        if (visible[MAX] > limit[MAX])
	    {
	        visible[MAX] = limit[MAX];
	        visible[MIN] = visible[MAX] - transposeRange;
	    }
	    
	    if (visible[MIN] != oldVisibleMin)
	    {
	    	transposed = true;
	    	
		    recreateScale();
		    
            for (int i=0; i<listeners.size(); i++)
            {
                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.TRANSPOSED, false));
            }
	    }
	}   

    /**
     * Add a ZoomSliderListener to the list of listeners for this slider
     * @param listener the ZoomSliderListener to get added to the list of ZoomSliderListeners
     */
	public void addZoomSliderListener(ZoomSliderListener listener)
	{
		listeners.add(listener);
	}

    /**
     * Remove a ZoomSliderListener from the list of listeners for this slider
     * @param listener the ZoomSliderListener to remove from the list of ZoomSliderListeners
     */
	public void removeZoomSliderListener(ZoomSliderListener listener)
	{
		listeners.remove(listener);
	}
	
    /**
     * Listen for paint events on this control 
     */
	class SliderPaintListener implements PaintListener
	{
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.PaintListener#paintControl(org.eclipse.swt.events.PaintEvent)
		 */
		public void paintControl(PaintEvent e)
		{
			ZoomSlider.this.paint(e);
		}
	}
	
	/** 
     * Listens for the mouse button being pressed, released and double-clicked 
     */
	class SliderMouseListener implements MouseListener
	{
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseDown(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseDown(MouseEvent e)
		{
			Slider_MouseDown(e);
		}
		
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseUp(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseUp(MouseEvent e)
		{
			Slider_MouseUp(e);
		}

		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseListener#mouseDoubleClick(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseDoubleClick(MouseEvent e)
		{
		}
	}
	
	/** 
     * Listens for the mouse being moved on this slider 
     */
    class SliderMouseMoveListener implements MouseMoveListener
	{
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.MouseMoveListener#mouseMove(org.eclipse.swt.events.MouseEvent)
		 */
		public void mouseMove(MouseEvent event)
		{
		    Slider_MouseMove(event);
		}
	}
	
    /**
     * Listen for slider resizing on this control
     */
	class SliderControlListener extends ControlAdapter 
	{
		/* (non-Javadoc)
		 * @see org.eclipse.swt.events.ControlAdapter#controlResized(org.eclipse.swt.events.ControlEvent)
		 */
		public void controlResized(ControlEvent e)
		{
			ZoomSlider.this.resizeScale();
		}
    } 

    private void Slider_MouseDown(MouseEvent event)
	{
	    if (!barsCollection.mouseDown(event))
	    {
	        if (event.button > 1)
	        {
				setCursor(cursor_transpose);

				if (transposable) primeTranspose(event);
	        }    
	        else
	        {
				if (orientation == VERTICAL) {
					setCursor(cursor_zoom_vertical);
				} else {
					setCursor(cursor_zoom_horizontal);
				}

				if (zoomable) primeZoom(event);
	        } 
	    }
	}
	
    private void Slider_MouseUp(MouseEvent event)
	{
        if (!barsCollection.mouseUp(event))
        {
			setCursor(cursor_normal);

			if (zoomPrimed) zoomPrimed = false;
    	    
	        if (zoomConfigured)
	        {
	            zoomConfigured = false;
	        }    
    	    
	        if (transposePrimed)
	        {
	            transposePrimed = false;
	        }
	        
	        if (transposeConfigured)
	        {
	            transposeConfigured = false;
	        }
	        
			if (zoomed)
			{
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.ZOOMED, true));
	            }
	            
	            zoomed = false;
			}
			
			if (transposed)
			{
	            for (int i=0; i<listeners.size(); i++)
	            {
	                ((ZoomSliderListener)(listeners.elementAt(i))).zoomSliderChanged(new ZoomSliderEvent(event.getSource(), ZoomSliderEvent.TRANSPOSED, true));
	            }
	            
	            transposed = false;
			}
	    }
    }

    private void Slider_MouseMove(MouseEvent event)
    {
    	String ntooltip;
    	
   	    ZoomControlBar bar = barsCollection.getZoomControlBarContaining(event.x, event.y);
   	    if (bar != null) {
   	    	ntooltip = bar.getToolTipText();
   	    	if (getToolTipText() == null || !getToolTipText().equals(ntooltip)) {
   	    		setToolTipText(ntooltip);
   	    	}
   	    } else {
  	    	ntooltip = null;	
   	    	if (getToolTipText() != null) {
   	    		setToolTipText(ntooltip);
   	    	}
   	    }

		if (!barsCollection.mouseMove(event))
		{
		    if (zoomPrimed) configureZoom(event);
    		
		    if (zoomConfigured) zoom(event);
    		
		    if (transposePrimed) configureTranspose(event);
		    
		    if (transposeConfigured) transpose(event);
		}    
    }
    
    class SliderDisposeListener implements DisposeListener
    {
    	/* (non-Javadoc)
    	 * @see org.eclipse.swt.events.DisposeListener#widgetDisposed(org.eclipse.swt.events.DisposeEvent)
    	 */
    	public void widgetDisposed(DisposeEvent e)
    	{
			if (image != null) 
                image.dispose();
			if (gImage != null) 
                gImage.dispose();
			
			if (plainScale != null) 
                plainScale.dispose();
			if (gPlain != null) 
                gPlain.dispose();
			
			if (titleImage != null) 
                titleImage.dispose();
			if (gTitle != null) 
                gTitle.dispose();
			
			if (titleFont != null) 
                titleFont.dispose();
			
			barsCollection.dispose();
			indicatorsCollection.dispose();

			cursor_zoom_vertical.dispose();
			cursor_zoom_horizontal.dispose();
			cursor_transpose.dispose();
			cursor_normal.dispose();
    	}
    }
    
    boolean minimized = false;
    
    public boolean isMinimized()
    {
    	return minimized;
    }
    
    Rectangle oldBounds = null;
    
    public void setMinimized(boolean minimized)
    {
    	this.minimized = minimized;    	
    }
}
