/**********************************************************************
 * Copyright (c) 2005, 2007 Scapa Technologies Limited and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: 
 * Scapa Technologies Limited - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.ui.widgets.zoomslider;

import org.eclipse.swt.graphics.GC;

import com.ibm.icu.text.DecimalFormatSymbols;

/**
 * A class representing a tick on a zoom slider
 * 
 * @author gchristelis
 * @since 4.0.0
 */
public class ZoomSliderTick extends java.lang.Object
{
	protected static DecimalFormatSymbols dfs = new DecimalFormatSymbols();
    /**
     * The slider this tick appears on
     */
    protected ZoomSlider slider;
    
    /**
     * Boolean as to whether text is displayed on the slider 
     */
    protected boolean textVisible;
    
    /**
     * The slider tick location 
     */
    protected double sliderValue;
    
    /** 
     * The slider tick pixel location
     */
    protected int sliderPixel;

    /**
     * The current string representation
     */
    protected String representation;        
    
    /**
     * Default ZoomSliderTick constructor
     */
    public ZoomSliderTick()
    {
    }
    
    /**
     * Construct a new ZoomSliderTick instance associated with a given slider at the specified value
     * @param slider the slider this tick will appear on
     * @param sliderValue the double location value of where the tick should appear on the slider
     */
    public ZoomSliderTick(ZoomSlider slider, double sliderValue)
    {
        this.slider = slider;        
        this.sliderValue = sliderValue;        
        sliderPixel = (int)slider.value2Pixel(sliderValue);        
        representation = ZoomSliderUtilities.double2String(sliderValue);
        textVisible = isTextVisible();
    }                   
           
    /**
     * Get the current pixel location of this tick on the slider canvas
     * @return the integer pixel location
     */
    public int getPixel()
    {
        return sliderPixel;
    }
    
    /**
     * Get the current location value of this tick
     * @return the double location value
     */
    public double getValue()
    {
        return sliderValue;
    }
    
    /**
     * Get the state of the cached textVisible boolean
     * @return the cached textvisible boolean
     */
    public boolean getTextVisibleBoolean()
    {
        return textVisible;
    }
        
    /**
     * Draw the slider tick line on the given GC
     * @param g the GC to draw the line on
     */
    public void paintLine(GC g)
    {
        g.setForeground(slider.getForeground());
        
        if (slider.getOrientation() == ZoomSlider.VERTICAL)
        {
            if (textVisible)
            {
                g.drawLine(0, sliderPixel, slider.getBounds().width, sliderPixel);
            }
            else
            {
                g.drawLine(0, sliderPixel, 5, sliderPixel);
                g.drawLine(slider.getBounds().width - 5, sliderPixel, slider.getBounds().width - 1, sliderPixel);
            }    
        }
        else
        {
            if (textVisible)
            {
                g.drawLine(sliderPixel, 0, sliderPixel, slider.getBounds().height);
            }
            else
            {
                g.drawLine(sliderPixel, 0, sliderPixel, 5);
                g.drawLine(sliderPixel, slider.getBounds().height - 5, sliderPixel, slider.getBounds().height - 1);
            }    
        }
    }
    
    /**
     * Draw the slider text on the given GC
     * @param g the GC to draw the line on
     */    
    public void paintText(GC g)
    {
        if (textVisible)
        {
            int ascent = g.getFontMetrics().getAscent();
            
            int width = g.stringExtent(representation).x;
            
            int head = width;            
                         
            int dotIndex = representation.indexOf(dfs.getDecimalSeparator());

            if (dotIndex != -1)
            {
                head = g.stringExtent(representation.substring(0, dotIndex)).x;
            }
            
            int xPos = 0;
            int yPos = 0;
            
			g.setBackground(slider.getBackground());
			
            if (slider.getOrientation() == ZoomSlider.VERTICAL)
            {
                xPos = slider.getDecimalPointPosition() - head;
                
                yPos = sliderPixel - g.stringExtent(representation).y/2;
                
                g.fillRectangle(xPos - 2, yPos, width + 4, g.stringExtent(representation).y);
            }
            else
            {
                xPos = sliderPixel - width/2;
                
                if (this.overlaps(slider.getLastVisible(), g))
                {
                    return;
                }    
                slider.setLastVisible(this);
                yPos = slider.getBounds().height / 5;

                g.fillRectangle(xPos, yPos, width, ascent + 2);
            }                
            
			g.setForeground(slider.getForeground());
            g.drawString(representation, xPos, yPos-1, true);
        }    
    }
    
    /**
     * Return whether the text on the slider is currently visible or not
     * @return a boolean representing the visibility of the text
     */
    protected boolean isTextVisible()
    {         
        int oom = slider.getIncrementOOM();
        oom++;
        double tempValue;
        if (oom > 0)
            tempValue = sliderValue / (Math.pow(10,oom));
        else
            tempValue = sliderValue * (Math.pow(10,(-1)*oom));
        long digit;
        if (Math.abs(tempValue) >= 1)
        {
            long maj = (long)(tempValue*10);
            long min = ((long) tempValue)*10;
            digit = maj - min;
        }
        else 
            digit = (long)(tempValue*10);
        
        digit = Math.abs(digit);
        
        if (digit == 0)
            return true;
        else if (digit == 5)
        {
            if (slider.getIncrement() != 5 * Math.pow(10, slider.getIncrementOOM()))
            {
                return true;
            }
            else
            {
                return false;
            }    
        }
        return false;       
    }
    
    /**
     * Does this slider tick instance overlap the given ZoomSliderTick on the given GC.
     * @param tick the tick to check overlap with
     * @param g the GC
     * @return a boolean of true if the overlap is true, and false otherwise. Returns false if tick is <code>null</code>.
     */
	public boolean overlaps(ZoomSliderTick tick, GC g)
	{
        if (tick == null)
        {
            return false;
        }
        
        if (tick.getPixel() > sliderPixel)
        {
            if (tick.getPixel() < sliderPixel + g.stringExtent(representation).x + 5)
            {
                return true;
            }    
        }
        else
        {
            if (tick.getPixel() > sliderPixel - g.stringExtent(representation).x - 5)
            {
                return true;
            }
        }    

        return false;
	}
}
