/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AllImportsDetails.java,v 1.5 2008/12/08 13:32:07 jkubasta Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.probekit.editor.internal.presentation;
import java.util.List;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.util.EObjectContainmentEList;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.hyades.models.internal.probekit.Import;
import org.eclipse.hyades.models.internal.probekit.Probe;
import org.eclipse.hyades.models.internal.probekit.ProbekitFactory;
import org.eclipse.hyades.models.internal.probekit.ProbekitPackage;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.hyades.probekit.editor.internal.provider.AllImportsItemProvider;
import org.eclipse.hyades.probekit.editor.internal.provider.ImportItemProvider;
import org.eclipse.hyades.probekit.editor.internal.provider.ProbekitItemProviderAdapterFactory;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.FocusEvent;
import org.eclipse.swt.events.FocusListener;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Event;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.ScrollBar;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Widget;


public class AllImportsDetails extends Composite implements IDetails, ModifyListener {
	protected final EAttribute TEXT = ProbekitPackage.eINSTANCE.getImport_Text();
	private ProbekitWidgetFactory _factory;
	protected ImportItemProvider _importsProvider;
	protected AllImportsItemProvider _allImportsProvider;
	private Composite _widgetComposite;
	Button _addButton;
	Button _removeButton;
	private static final int NO_INDEX = -1;
	private Probe _probe;
	private WidgetFocusListener _widgetFocusListener = new WidgetFocusListener();
	protected ScrolledComposite sc1;
	protected Composite page;

	public AllImportsDetails(ProbekitWidgetFactory factory, ProbekitItemProviderAdapterFactory itemFactory, Composite composite, int style) {
		super(composite, style);
		
		_factory = factory;
		_importsProvider = (ImportItemProvider) itemFactory.createImportAdapter();
		_allImportsProvider = (AllImportsItemProvider) itemFactory.createAllImportsProviderAdapter();

		Composite parent = this;
		this.setLayoutData(GridUtil.createFill());
		this.setLayout(new GridLayout());
		createControl(parent);
	}
	
	public Control getFocusTo() {
		return _widgetFocusListener.getCurrentControl();
	}
	
	private void createControl(Composite parent) {
		sc1 = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
		sc1.setLayoutData(GridUtil.createFill());
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
		page = new Composite(sc1, SWT.NONE);
		sc1.setContent(page);
		GridLayout detailLayout = new GridLayout();
		detailLayout.numColumns = 2;
		page.setLayout(detailLayout);
		page.setLayoutData(GridUtil.createFill());

		Label label = _factory.createLabel(
				page, 
				ProbekitMessages._20, 
				SWT.NONE);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		label.setLayoutData(data);
		
		createWidgetComposite(page);
		createButtons(page);
		
		refreshPageSize();
	}
	
	private void createWidgetComposite(Composite parent) {
		_widgetComposite = _factory.createComposite(parent, SWT.NONE);
		GridLayout widgetLayout = new GridLayout();
		_widgetComposite.setLayout(widgetLayout);
		_widgetComposite.setLayoutData(GridUtil.createFill());
	}
	
	private void createButtons(Composite parent) {
		ButtonSelectionListener listener = new ButtonSelectionListener();

		Composite buttonGroup = _factory.createComposite(parent, SWT.NONE);
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.makeColumnsEqualWidth = true;
		buttonGroup.setLayout(buttonLayout);
		buttonGroup.setLayoutData(GridUtil.createVerticalFill());
		buttonGroup.setFont(parent.getFont());

		_addButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._125,
				SWT.PUSH);
		GridData addData = GridUtil.createHorizontalFill();
		_addButton.setLayoutData(addData);
		_addButton.addSelectionListener(listener);
		_addButton.addFocusListener(_widgetFocusListener);
		
		_removeButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._119,
				SWT.PUSH);
		GridData removeData = GridUtil.createHorizontalFill();
		_removeButton.setLayoutData(removeData);
		_removeButton.addSelectionListener(listener);
		_removeButton.addFocusListener(_widgetFocusListener);
		
		refreshButtons();
	}
	
	private Text createWidget() {
		Text importWidget = _factory.createText(_widgetComposite, ResourceUtil.NO_TEXT, SWT.NONE);
		importWidget.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		// add listeners - don't change the order in which the listeners are
		// added
		// because of team support issues modify listeners should always be the
		// last one added
		importWidget.addModifyListener(this);
		importWidget.addFocusListener(_widgetFocusListener);

		refreshPane();
		refreshPageSize();
		return importWidget;
	}
	
	protected Import createImport() {
		Import anImport = ProbekitFactory.eINSTANCE.createImport();
		_allImportsProvider.addAndNotify(getProbe(), anImport);
		return anImport;
	}
	
	protected void removeImport(int index) {
		Import selected = getImport(index);
		if(selected != null) {
			_allImportsProvider.removeAndNotify(getProbe(), selected);
			refreshImports(index);
		}
	}
	
	protected int removeImportAndWidget() {
		// Whenever an import is removed, move all of the other imports
		// underneath it up by one so that the column of entry fields
		// doesn't have blank lines.
		Text widget = getSelectedWidget();
		int index = getIndex(widget);
		removeImport(index);
		
		// Shift the cursor if the last item in the list was deleted.
		if(index >= getImportList().size()) {
			index -= 1;
		}
		
		refreshWidgets();
		refreshPane();
		refreshPageSize();
		return index;
	}
	
	protected Text addWidgetIfNecessary(int index) {
		Text widget = null;
		Control[] children = _widgetComposite.getChildren();
		if(children.length <= index) { 
			widget = createWidget();
		}
		else {
			widget = (Text)children[index];
			widget.setVisible(true);
			refreshPane();
		}
		return widget;
	}
	
	protected Import addImportIfNecessary(int index) {
		Import anImport = null;
		int size = getImportList().size();
		if(index >= size) {
			anImport = createImport();
		}
		return anImport;
	}
	
	private void refreshWidgets() {
		Control[] children = _widgetComposite.getChildren();
		int importSize = getImportList().size();
		for(int i=children.length-1; i>=importSize; i--) {
			Text text = (Text)children[i];
			text.setVisible(false); // Can't remove the widget once it's created?
		}
	}
	
	private void refreshPane() {
		_widgetComposite.layout(true);
		_widgetComposite.redraw();
		_widgetComposite.update();
	}
	
	protected void refreshPageSize() {
		sc1.setMinSize(page.computeSize(SWT.DEFAULT, SWT.DEFAULT));
		page.layout();
	}
	
	private void refreshImports(int index) {
		int importSize = getImportList().size();
		Control[] children = _widgetComposite.getChildren();
		for(int i=index; i<importSize; i++) {
			Import anImport = getImport(i);
			Text text = (Text)children[i];
			if(anImport.getText() != null) {
				text.setText(anImport.getText());
			}
		}
	}
	
	void refreshButtons() {
		if((getProbe() == null) || (getImportList().size() == 0)) {
			_removeButton.setEnabled(false);
		}
		else {
			Text text = getSelectedWidget();
			boolean enabled = (text != null);
			_removeButton.setEnabled(enabled);
		}
	}
	
	protected Text getWidget(int index) {
		if(index <= NO_INDEX) {
			return null;
		}

		addWidgetIfNecessary(index);
		addImportIfNecessary(index);
		Control[] children = _widgetComposite.getChildren();
		Text importWidget = (Text)children[index];
		return importWidget;
	}
	
	Text getSelectedWidget() {
		Control control = getFocusTo();
		if(control instanceof Text) {
			return (Text)control;
		}
		return null;
	}
	
	protected EList getImportList() {
		return getProbe().getImport();
	}
	
	protected Probe getProbe() {
		return _probe;
	}
	
	int getIndex(Widget text) {
		Control[] children = _widgetComposite.getChildren();
		for(int i=0; i<children.length; i++) {
			if(children[i].equals(text)) {
				return i;
			}
		}
		return NO_INDEX;
	}
	
	Import getImport(int index) {
		if(index <= NO_INDEX) {
			return null;
		}
		
		if(index < getImportList().size()) {
			Object value = getImportList().get(index);
			if(value != null) {
				return (Import)value;
			}
		}
		return null;
	}
	
	void select(int index) {
		if((index > NO_INDEX) && (index <= _widgetComposite.getChildren().length)) {
			Text widget = getWidget(index);
			widget.setFocus();
			String text = widget.getText();
			int length = text.length();
			widget.setSelection(0, length);
		}
	}
	
	public boolean updateCurrentSelection(Object currentSelection) {
		return false;
	}
	
	private void select(Import importObject) {
		int index = getImportList().indexOf(importObject);
		select(index);
	}
	
	public void display(Object object) {
		Import importObject = null;
		if(object instanceof EObjectContainmentEList) {
			_probe = (Probe)((EObjectContainmentEList) object).getEObject();
		}
		else {
			importObject = (Import)object;
			_probe = (Probe)importObject.eContainer();
		}
		
		if(!isThisFocusControl()) {
			// Initialize the page's contents
			List importList = getImportList();
			int size = importList.size();
			for(int i=0; i<size; i++) {
				Import anImport = (Import)importList.get(i);
				displayImport(anImport, getWidget(i));
			}
		
			refreshWidgets();
			refreshButtons();
			refreshPane();
		}

		// And then if we're supposed to display a specific import,
		// then focus on it.
		if(importObject != null) {
			select(importObject);
		}
	}
	
	/**
	 * Return true if any of the widgets on this page are the current focus.
	 */
	private boolean isThisFocusControl() {
		if(this.isFocusControl()) {
			return true;
		}
		
		if(_widgetComposite.isFocusControl()) {
			return true;
		}
		
		if(_addButton.isFocusControl()) {
			return true;
		}
		
		if(_removeButton.isFocusControl()) {
			return true;
		}
		
		
		Control[] children = _widgetComposite.getChildren();
		int size = children.length;
		for(int i=0; i<size; i++) {
			Control control = children[i];
			if(control.isFocusControl()) {
				return true;
			}
		}
		
		return false;
	}
	
	private void displayImport(Import importObj, Text importWidget) {
		if (importWidget.isFocusControl()) {
			return;
		}
		importWidget.setText(ResourceUtil.getString(importObj.getText()));
	}
	
	public void modifyText(ModifyEvent e) {
		Text importWidget = (Text)e.getSource();
		int index = getIndex(importWidget);
		Import anImport = getImport(index);
		if(anImport == null) {
			// In the process of creating an import
			return;
		}
		String text = importWidget.getText().trim();
		if (!text.equals(anImport.getText())) {
			if((text.length() == 0) && (anImport.getText() == null)) {
				return;
			}
			IItemPropertyDescriptor descriptor = _importsProvider.getPropertyDescriptor(anImport, TEXT);
			if (descriptor != null)
				descriptor.setPropertyValue(anImport, text);
		}
	}
	
	public void dispose() {
		_factory = null;
		_importsProvider = null;
		_allImportsProvider = null;
		_widgetComposite = null;
		_addButton = null;
		_removeButton = null;
		_widgetFocusListener = null;
		_probe = null;
	}
	
	public boolean isDetailsFor(Object object) {
		if(object instanceof EObjectContainmentEList) {
			EObjectContainmentEList list = (EObjectContainmentEList)object;
			if(list.getFeatureID() == ProbekitPackage.PROBE__IMPORT) {
				return true;
			}
		}
		else if(object instanceof Import) {
			return true;
		}
		return false;
	}
	
	public void setFocusTo() {
		if(getImportList().size() > 0) {
			select(0);
		}
	}
	
	private class WidgetFocusListener implements FocusListener {
		private Text _currentControl = null;
		
		public Text getCurrentControl() {
			return _currentControl;
		}
		
		public void focusLost(FocusEvent e) {
		}

		public void focusGained(FocusEvent e) {
			if (e.getSource() instanceof Text) {
				_currentControl = (Text) e.getSource();
			}
			if(e.getSource() instanceof Control) {
				Control control = (Control)e.getSource();
				setVisible(control.getLocation());
			}
			refreshButtons();
		}
	}
	
	/**
	 * If the scroll bars exist, then even though the Text widget
	 * may have focus, it may not be visible. This method assumes
	 * that the location of the widget with focus must be scrolled
	 * to. That is, given a location that this composite should show,
	 * it moves the scroll bars to that location.
	 */
	private void setVisible(Point location) {
		ScrollBar horizontalBar = sc1.getHorizontalBar();
		if(horizontalBar != null) {
			horizontalBar.setSelection(location.x);
			Event event = new Event();
			event.widget = horizontalBar;
			horizontalBar.notifyListeners(SWT.Selection, event);
		}
		ScrollBar verticalBar = sc1.getVerticalBar();
		if(verticalBar != null) {
			verticalBar.setSelection(location.y);
			Event event = new Event();
			event.widget = verticalBar;
			verticalBar.notifyListeners(SWT.Selection, event);
		}
	}

	private class ButtonSelectionListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing to do when the default is selected.
		}

		public void widgetSelected(SelectionEvent event) {
			if(event.widget.equals(_addButton)) {
				add();
			}
			else if(event.widget.equals(_removeButton)) {
				remove();
			}
			
			refreshButtons();
		}
		
		private void add() {
			int size = getImportList().size();
			Import anImport = addImportIfNecessary(size);
			Text widget = addWidgetIfNecessary(size);
			String text = (anImport.getText() == null) ? ResourceUtil.NO_TEXT : anImport.getText();
			widget.setText(text);
			select(size);
			refresh();
		}
		
		private void remove() {
			int selected = removeImportAndWidget();
			select(selected);
		}
	}

	public void refresh() {
	}
	
	public Control getFocusTo(Object object) {
		if (!(object instanceof EObjectContainmentEList)) {
			int index = getImportList().indexOf(object);
			if ((index > NO_INDEX)
					&& (index <= _widgetComposite.getChildren().length)) {
				return getWidget(index);
			}
		}
		return null;
	}
}