  /**********************************************************************
 * Copyright (c) 2005, 2010 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: AllTargetsDetails.java,v 1.5 2010/01/05 19:33:02 kchan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.hyades.probekit.editor.internal.presentation;

import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.ecore.util.EObjectContainmentEList;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.hyades.models.internal.probekit.Probe;
import org.eclipse.hyades.models.internal.probekit.ProbekitFactory;
import org.eclipse.hyades.models.internal.probekit.ProbekitPackage;
import org.eclipse.hyades.models.internal.probekit.Target;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.hyades.probekit.editor.internal.provider.AllTargetsItemProvider;
import org.eclipse.hyades.probekit.editor.internal.provider.ProbekitItemProviderAdapterFactory;
import org.eclipse.hyades.probekit.editor.internal.provider.TargetItemProvider;
import org.eclipse.hyades.probekit.ui.internal.ProbekitUIPlugin;
import org.eclipse.jface.dialogs.Dialog;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ControlEvent;
import org.eclipse.swt.events.ControlAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.graphics.Point;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;




public class AllTargetsDetails extends BaseDetails {
	private static final String DOT = "."; //$NON-NLS-1$
	private static final String STAR = "*"; //$NON-NLS-1$
	private ProbekitWidgetFactory _factory;
	private ProbekitItemProviderAdapterFactory _itemFactory;
	protected AllTargetsItemProvider _allTargetsProvider;
	protected TargetItemProvider _itemProvider;
	private Probe _probe;
	
	private Button _upButton;
	private Button _downButton;
	private Button _addButton;
	private Button _editButton;
	private Button _removeButton;
	
	private TableViewer _tableViewer;

	static final String[] COLUMNS = new String[] { ProbekitMessages._106,
			ProbekitMessages._107, ProbekitMessages._108,
			ProbekitMessages._109, ProbekitMessages._110 };
	static final EAttribute[] FEATURES = {
		ProbekitPackage.eINSTANCE.getTarget_Type(),
		ProbekitPackage.eINSTANCE.getTarget_Package(),
		ProbekitPackage.eINSTANCE.getTarget_ClassName(),
		ProbekitPackage.eINSTANCE.getTarget_Method(),
		ProbekitPackage.eINSTANCE.getTarget_Signature()
	};
	ComboBoxCellEditor _comboBoxCellEditor;

	public AllTargetsDetails(ProbekitWidgetFactory factory, ProbekitItemProviderAdapterFactory itemFactory, Composite composite, int style) {
		super(composite, style);
		
		_factory = factory;
		_itemFactory = itemFactory;
		_allTargetsProvider = (AllTargetsItemProvider) itemFactory.createAllTargetsProviderAdapter();
		_itemProvider = (TargetItemProvider) itemFactory.createTargetAdapter();
		
		Composite parent = this;
		this.setLayoutData(GridUtil.createFill());
		this.setLayout(new GridLayout());
		createControl(parent);
	}
	
	private void createControl(Composite parent) {
		
		ScrolledComposite sc1 = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
		sc1.setLayoutData(GridUtil.createFill());
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
		Composite page = new Composite(sc1, SWT.NONE);
		sc1.setContent(page);
		  
		GridLayout detailLayout = new GridLayout();
		detailLayout.numColumns = COLUMNS.length;
		detailLayout.makeColumnsEqualWidth = false;
		page.setLayout(detailLayout);
		page.setLayoutData(GridUtil.createFill());

		Label label = _factory.createLabel(
				page, 
				ProbekitMessages._23, 
				SWT.NONE);
		GridData data = new GridData();
		data.horizontalSpan = COLUMNS.length;
		data.widthHint = 400;		
		label.setLayoutData(data);
		
		createTable(page);
		
		sc1.setMinSize(page.computeSize(SWT.DEFAULT, SWT.DEFAULT));
	}
	
	private void createTable(Composite parent) {
		TargetModifyListener listener = new TargetModifyListener();
		createTableViewer(parent, listener);
		createTableButtons(parent, listener);
	}
	
	private void createTableViewer(Composite parent, TargetModifyListener listener) {
		_tableViewer = new TableViewer(parent, SWT.BORDER | SWT.H_SCROLL | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION );
		Table table = _tableViewer.getTable();
		
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];
		
		TableColumn comboCell = new TableColumn(table, SWT.LEFT);
		comboCell.setText(COLUMNS[0]);
		String[] comboboxItems = new String[]{
				TargetTypeEnum.INCLUDE.getTranslatedType(),
				TargetTypeEnum.EXCLUDE.getTranslatedType()
		};
		_comboBoxCellEditor = 
			new ComboBoxCellEditor(
					table, 
					comboboxItems, 
					 SWT.READ_ONLY
			);
		cellEditors[0] = _comboBoxCellEditor;
		
		for( int i = 1; i < COLUMNS.length; i++) {
			TableColumn textCell = new TableColumn(table, SWT.LEFT);
			textCell.setText(COLUMNS[i]);
			cellEditors[i] = new TextCellEditor(table);
		}

		_tableViewer.setCellEditors(cellEditors);
		_tableViewer.setCellModifier(new TargetCellModifier());
		_tableViewer.setContentProvider(new TargetContentProvider());
		_tableViewer.setLabelProvider(new TargetColumnLabelProvider());
		_tableViewer.setColumnProperties(COLUMNS);
		
		GridData tableData = GridUtil.createFill();
		table.setLayoutData(tableData);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setFont(parent.getFont());
		table.addSelectionListener(listener);
		// Why use the resize listener when TableLayout is used?
		// The layout seems to be broken. See bug 13467.
		table.addControlListener(new TargetControlListener(table));
	}

	private void createTableButtons(Composite parent, TargetModifyListener listener) {
		Composite buttonGroup = _factory.createComposite(parent, SWT.NONE);
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.makeColumnsEqualWidth = true;
		buttonGroup.setLayout(buttonLayout);
		GridData buttonData = GridUtil.createVerticalFill();
		buttonData.grabExcessHorizontalSpace = false;
		buttonGroup.setLayoutData(buttonData);
		buttonGroup.setFont(parent.getFont());

		_addButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._125,
				SWT.PUSH);
		GridData addData = GridUtil.createHorizontalFill();
		_addButton.setLayoutData(addData);
		_addButton.addSelectionListener(listener);
		
		_editButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._118,
				SWT.PUSH);
		GridData editData = GridUtil.createHorizontalFill();
		_editButton.setLayoutData(editData);
		_editButton.addSelectionListener(listener);
		
		_removeButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._119,
				SWT.PUSH);
		GridData removeData = GridUtil.createHorizontalFill();
		_removeButton.setLayoutData(removeData);
		_removeButton.addSelectionListener(listener);
		
		_upButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._120,
				SWT.PUSH);
		GridData upData = GridUtil.createHorizontalFill();
		_upButton.setLayoutData(upData);
		_upButton.addSelectionListener(listener);
		
		_downButton = _factory.createButton(
				buttonGroup, 
				ProbekitMessages._121,
				SWT.PUSH);
		GridData downData = GridUtil.createHorizontalFill();
		_downButton.setLayoutData(downData);
		_downButton.addSelectionListener(listener);
		
		refreshButtons();
	}
	
	protected Target addTarget() {
		Target aTarget = ProbekitFactory.eINSTANCE.createTarget();
		aTarget.setType(TargetTypeEnum.INCLUDE.getType()); // Default new types to "include".
		aTarget.setPackage("*"); // Default to "*".
		aTarget.setClassName("*"); // Default to "*".
		aTarget.setMethod("*"); // Default new to "*".
		aTarget.setSignature("*"); // Default to "*".
		_allTargetsProvider.addAndNotify(getProbe(), aTarget);
		return aTarget;
	}
	
	protected void removeTarget(Target selected) {
		if(selected != null) {
			_allTargetsProvider.removeAndNotify(getProbe(), selected);
		}
	}
	
	protected EList getTargetItems() {
		return getProbe().getTarget();
	}
	
	protected Probe getProbe() {
		return _probe;
	}
	
	protected TableViewer getTableViewer() {
		return _tableViewer;
	}
	
	Target getItem(int index) {
		if (index == -1) {
			return null;
		}
		else {
			return (Target) getTableViewer().getElementAt(index);
		}
	}
	
	int getSelectedIndex() {
		return getTableViewer().getTable().getSelectionIndex();
	}
	
	void refreshTable() {
		getTableViewer().refresh();
	}
	
	public boolean updateCurrentSelection(Object currentSelection) {
		setCurrentSelection(currentSelection);
		return false;
	}
	
	public void setCurrentSelection(Object currentSelection) {
		if(currentSelection instanceof EObjectContainmentEList) {
			_probe = (Probe)((EObjectContainmentEList) currentSelection).getEObject();
		}
		else {
			Target targetObject = (Target)currentSelection;
			_probe = (Probe)targetObject.eContainer();
			select(targetObject);
		}
		getTableViewer().setInput(getTargetItems());
	}
	
	private void select(Target targetObject) {
		TableItem[] items = _tableViewer.getTable().getItems();
		for(int i=0; i<items.length; i++) {
			TableItem item = items[i];
			if(item.getData().equals(targetObject)) {
				select(i);
				return;
			}
		}
	}

	private void select(int index) {
		getTableViewer().getTable().select(index);
	}

	public void display(Object o) {
		setCurrentSelection(o);
	}
	
	public void setFocusTo() {
	}
	
	public boolean isDetailsFor(Object object) {
		if(object instanceof EObjectContainmentEList) {
			EObjectContainmentEList list = (EObjectContainmentEList)object;
			if(list.getFeatureID() == ProbekitPackage.PROBE__TARGET) {
				return true;
			}
		}
		else if(object instanceof Target) {
			return true;
		}
		return false;
	}
	
	void refreshButtons() {
		if((getProbe() == null) || (getTargetItems().size() == 0)) {
			_removeButton.setEnabled(false);
			_editButton.setEnabled(false);
			_upButton.setEnabled(false);
			_downButton.setEnabled(false);
		}
		else {
			int index = getSelectedIndex();
			Target item = getItem(index);
			boolean enabled = (item != null);
			_removeButton.setEnabled(enabled);
			_editButton.setEnabled(enabled);
			_upButton.setEnabled(enabled && (index > 0));
			_downButton.setEnabled(enabled && (index < (getTargetItems().size()-1)));
		}
	}
	
	String getFilterName(Target target) {
		StringBuffer buffer = new StringBuffer();
		String pkg = target.getPackage();
		if((pkg != null) && (!ResourceUtil.NO_TEXT.equals(pkg))) {
			buffer.append(pkg);
		}
		else {
			buffer.append(STAR);
		}
		String clazz = target.getClassName();
		buffer.append(DOT);
		if((clazz != null) && (!ResourceUtil.NO_TEXT.equals(clazz))) {
			buffer.append(clazz);
		}
		else {
			buffer.append(STAR);
		}
		String method = target.getMethod();
		buffer.append(DOT);
		if((method != null) && (!ResourceUtil.NO_TEXT.equals(method))) {
			buffer.append(method);
		}
		else {
			buffer.append(STAR);
		}
		String signature = target.getSignature();
		buffer.append(DOT);
		if((signature != null) && (!ResourceUtil.NO_TEXT.equals(signature))) {
			buffer.append(signature);
		}
		else {
			buffer.append(STAR);
		}
		
		return buffer.toString();
	}
	
	protected Button getAddButton() {
		return _addButton;
	}
	
	protected Button getEditButton() {
		return _editButton;
	}
	
	protected Button getRemoveButton() {
		return _removeButton;
	}
	
	protected Button getUpButton() {
		return _upButton;
	}
	
	protected Button getDownButton() {
		return _downButton;
	}
	
	private class TargetModifyListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing to do when the default is selected.
		}

		public void widgetSelected(SelectionEvent event) {
			if(event.widget.equals(getAddButton())) {
				add();
			}
			else if(event.widget.equals(getEditButton())) {
				edit();
			}
			else if(event.widget.equals(getRemoveButton())) {
				remove();
			}
			else if(event.widget.equals(getUpButton())) {
				moveUp();
			}
			else if(event.widget.equals(getDownButton())) {
				moveDown();
			}
			
			refreshButtons();
		}
		
		private void add() {
			addTarget();
			
			// Refresh before selecting so that the new item is added to the table,
			// and refresh after selecting so that the "Edit" and "Remove" buttons
			// are enabled.
			refreshTable();
			select(getTableViewer().getTable().getItemCount()-1);
		}
		
		private void remove() {
			int index = getSelectedIndex();
			Target item = getItem(index);
			if(item != null) {
				removeTarget(item);
				index -= 1;
			}
			
			refreshTable();
			select(index);
		}
		
		private void edit() {
			int index = getSelectedIndex();
			Target item = getItem(index);
			if(item != null) {
				// Can't select a single cell in a row.
				// Launch an editor dialog instead.
				TargetDialog dialog = new TargetDialog(
							getShell(),
							item,
							_factory,
							_itemFactory);
				dialog.setBlockOnOpen(true);
				dialog.setTitle(ProbekitMessages._111);
				dialog.open();
				
				// Don't need to get the result from the dialog
				// because the dialog fires Commands if necessary,
				// and doesn't fire anything if not necessary.
				refreshTable();
				select(index);
			}
		}
		
		private void moveUp() {
			int selectedIndex = getSelectedIndex();
			Target item = getItem(selectedIndex);
			if((item != null) && (selectedIndex > 0)) {
				getTargetItems().remove(item);
				getTargetItems().add(selectedIndex - 1, item);
				refreshTable();
				select(selectedIndex - 1);
			}
		}
		
		private void moveDown() {
			int selectedIndex = getSelectedIndex();
			Target item = getItem(selectedIndex);
			if((item != null) && (selectedIndex < getTargetItems().size()-1)) {
				getTargetItems().remove(item);
				getTargetItems().add(selectedIndex + 1, item);
				refreshTable();
				select(selectedIndex + 1);
			}
		}
	}

	private class TargetColumnLabelProvider extends LabelProvider implements ITableLabelProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		
		public String getColumnText(Object element, int columnIndex) {
			String result = null;
			if(element instanceof Target) {
				Target item = (Target)element;
				if(columnIndex == 0) {
					result = item.getType();
				} else if (columnIndex == 1) {
					result = item.getPackage();
				} else if (columnIndex == 2) {
					result = item.getClassName();
				} else if (columnIndex == 3) {
					result = item.getMethod();
				} else if (columnIndex == 4) {
					result = item.getSignature();
				}
			}
			return (result == null) ? ResourceUtil.NO_TEXT : result;
		}
		
	}

	private class TargetContentProvider implements IStructuredContentProvider {
		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object[] getElements(Object inputElement) {
			return getTargetItems().toArray();
		}
	}
	
	private class TargetCellModifier implements ICellModifier {
		public Object getValue(Object element, String property) {
			Object value = null;
			if (element instanceof Target) {
				Target item = (Target)element;
				if (property.equals(COLUMNS[0])) {
					String typeName = (item.getType() == null) ? ResourceUtil.NO_TEXT : item.getType();
					value = getCellIndex(typeName);				
				} else if (property.equals(COLUMNS[1])) {
					value = item.getPackage();
				} else if (property.equals(COLUMNS[2])) {
					value = item.getClassName();
				} else if (property.equals(COLUMNS[3])) {
					value = item.getMethod();
				} else if (property.equals(COLUMNS[4])) {
					value = item.getSignature();
				}
			}
			return (value == null) ? ResourceUtil.NO_TEXT : value;
		}
		
		private Integer getCellIndex(String typeName) {
			Integer index = null;
			String[] items = _comboBoxCellEditor.getItems();
			for(int i=0; i<items.length; i++) {
				String itemText = items[i];
				if(itemText.equals(typeName)) {
					index = new Integer(i);
					break;
				}
			}
			
			if(index == null) {
				index = new Integer(0);
			}
			
			return index;
		}

		public boolean canModify(Object element, String property) {
			return true;
		}

		public void modify(Object element, String property, Object value) {
			TableItem item = (TableItem) element;
			Target target = (Target)item.getData();
			
			String newval = null, oldval = null;
			EAttribute propertyId = null;
			
			if(property.equals(COLUMNS[0])) {
				Integer cellIndex = (Integer)value;
				String[] items = _comboBoxCellEditor.getItems();
				String typeName = items[cellIndex.intValue()];
				TargetTypeEnum type = TargetTypeEnum.getTargetType(typeName);
				oldval = target.getType();
				newval = type.getType();
				propertyId = FEATURES[0];
			} else if (property.equals(COLUMNS[1])) {
				oldval = target.getPackage();
				newval = (String)value;	
				propertyId = FEATURES[1];
			} else if (property.equals(COLUMNS[2])) {
				oldval = target.getClassName();
				newval = (String)value;				 
				propertyId = FEATURES[2];
			} else if (property.equals(COLUMNS[3])) {
				oldval = target.getMethod();
				newval = (String)value;	
				propertyId = FEATURES[3];
			} else if (property.equals(COLUMNS[4])) {
				oldval = target.getSignature();
				newval = (String)value;	
				propertyId = FEATURES[4];
			}
			
			if (!newval.equals(oldval)) {
				IItemPropertyDescriptor descriptor = 
					_itemProvider.getPropertyDescriptor(target, propertyId);
				if (descriptor != null)
					descriptor.setPropertyValue(target, newval);

				_tableViewer.refresh();
			}
		}
	};
	
	private static class TargetDialog extends Dialog {
		private Target _item;
		private ProbekitWidgetFactory _factory;
		private ProbekitItemProviderAdapterFactory _itemFactory;
		private String _title;
		
		public TargetDialog(Shell shell, Target item, ProbekitWidgetFactory factory, ProbekitItemProviderAdapterFactory itemFactory) {
			super(shell);
			int shellStyle = getShellStyle();
			setShellStyle(shellStyle | SWT.RESIZE);
			_item = item;
			_factory = factory;
			_itemFactory = itemFactory;
		}
		
		public Target getResult() {
			return _item;
		}
		
		protected Control createDialogArea(Composite parent) {
			TargetDetails details = new TargetDetails(_factory, _itemFactory, parent, SWT.RESIZE);
			details.display(_item);
			return details;
		}
		
		protected void configureShell(Shell shell) {
			super.configureShell(shell);
			if (getTitle() != null) {
				shell.setText(getTitle());
			}
		}
		
		private String getTitle() {
			return _title;
		}
		
		public void setTitle(String title) {
			_title = title;
		}
	}
	
	class TargetControlListener extends ControlAdapter {

		Table table;

		public TargetControlListener(Table table) {
			this.table = table;
		}

		protected void resizeColumns(int width) {
			TableColumn[] cols = table.getColumns();
			for (int i = 0; i < cols.length; i++) {
				cols[i].setWidth(width);
			}
		}

		public void controlResized(ControlEvent e) {

			Rectangle area = table.getClientArea();
			Point preferredSize = table.computeSize(SWT.DEFAULT, SWT.DEFAULT);
			int width = area.width - 2 * table.getBorderWidth();
			if (preferredSize.y > area.height) {
				// Subtract the scrollbar width from the total column width
				// if a vertical scrollbar will be required
				Point vBarSize = table.getVerticalBar().getSize();
				width -= vBarSize.x;
			}
			resizeColumns(width / COLUMNS.length);
		}
	}

	public void refresh() {
	}
}
