/**********************************************************************
 * Copyright (c) 2005, 2008 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: FragmentDetails.java,v 1.7 2008/12/08 13:32:07 jkubasta Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.probekit.editor.internal.presentation;

import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;

import org.eclipse.emf.ecore.EAttribute;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.AdapterFactoryEditingDomain;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.emf.edit.provider.IItemPropertyDescriptor;
import org.eclipse.hyades.models.internal.probekit.DataItem;
import org.eclipse.hyades.models.internal.probekit.DataType;
import org.eclipse.hyades.models.internal.probekit.Fragment;
import org.eclipse.hyades.models.internal.probekit.FragmentType;
import org.eclipse.hyades.models.internal.probekit.ProbekitPackage;
import org.eclipse.hyades.models.internal.probekit.impl.DataItemImpl;
import org.eclipse.hyades.models.internal.probekit.impl.FragmentImpl;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.ProbekitTypesSingleton;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitUtil;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.hyades.probekit.editor.internal.provider.FragmentItemProvider;
import org.eclipse.hyades.probekit.editor.internal.provider.ProbekitItemProviderAdapterFactory;
import org.eclipse.hyades.probekit.editor.internal.ui.DataDialog;
import org.eclipse.hyades.probekit.editor.internal.ui.DialogUtil;
import org.eclipse.jdt.ui.PreferenceConstants;
import org.eclipse.jdt.ui.text.JavaSourceViewerConfiguration;
import org.eclipse.jdt.ui.text.JavaTextTools;
import org.eclipse.jface.preference.IPreferenceStore;
import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.IDocumentPartitioner;
import org.eclipse.jface.text.source.ISourceViewer;
import org.eclipse.jface.text.source.SourceViewer;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.ScrolledComposite;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.ui.PlatformUI;


public class FragmentDetails extends BaseDetails {
	protected final static EAttribute TYPE = ProbekitPackage.eINSTANCE.getFragment_Type();
	protected CCombo fragmentCombo;
	protected ISourceViewer _methodViewer;
	protected boolean isDisplaying;
	protected Fragment currentSelection;
	protected FragmentItemProvider itemProvider;
	protected ProbekitItemProviderAdapterFactory itemFactory;
	protected ProbekitWidgetFactory widgetFactory;
	protected JavaTextTools _javaTextTools;
	protected JavaSourceViewerConfiguration _methodViewerConfiguration;
	Button _addButton = null;
	Button _editButton = null;
	Button _removeButton = null;
	private Label _errorMessage;
	private Label _codeLabelImage;
	static final String[] COLUMNS = new String[] { ProbekitMessages._40,
			ProbekitMessages._41};
	TableViewer _tableViewer;
	ComboBoxCellEditor _comboBoxCellEditor;
	TextCellEditor _textCellEditor;
	
    public FragmentDetails(ProbekitWidgetFactory factory, ProbekitItemProviderAdapterFactory itemFactory, Composite composite, int style) {
		super(composite, style);
		
		this.widgetFactory = factory;
		this.itemFactory = itemFactory;
		
		itemProvider = (FragmentItemProvider) itemFactory.createFragmentAdapter();
		
		Composite parent = this;
		this.setLayoutData(GridUtil.createFill());
		this.setLayout(new GridLayout());
		createControl(parent);
    }
    
    protected void createControl(Composite parent) {
    	
	  	ScrolledComposite sc1 = new ScrolledComposite(parent, SWT.V_SCROLL | SWT.H_SCROLL);
		sc1.setLayoutData(GridUtil.createFill());
		sc1.setExpandHorizontal(true);
		sc1.setExpandVertical(true);
		Composite page = new Composite(sc1, SWT.NONE);
		sc1.setContent(page);
		  
		GridLayout detailLayout = new GridLayout();
		detailLayout.numColumns = 1;
		page.setLayout(detailLayout);
		page.setLayoutData(GridUtil.createFill());
		
		createCombo(page);
		createTable(page);
		createCode(page);
		
		sc1.setMinSize(page.computeSize(SWT.DEFAULT, SWT.DEFAULT));
    }
    
	protected void createCombo(Composite parent) {
		widgetFactory.createLabel(parent, ProbekitMessages._19, SWT.NONE);
		fragmentCombo = widgetFactory.createCombo(parent, SWT.READ_ONLY | SWT.V_SCROLL | SWT.H_SCROLL);
		fragmentCombo.setLayoutData(new GridData(GridData.FILL_HORIZONTAL));
		List list = FragmentType.VALUES;
		List fragmentComboLabels = new ArrayList();
		if (list != null) {
			for (int i = 0; i < list.size(); i++) {
				fragmentComboLabels.add(((FragmentType) (list.get(i))).getName());
			}
		}
		Collections.sort(fragmentComboLabels);
		SelectionListener listener = new ComboSelectionListener();
		for (Iterator iter = fragmentComboLabels.iterator(); iter.hasNext();) {
			String element = (String) iter.next();
			fragmentCombo.add(element);
			fragmentCombo.addSelectionListener(listener);
		}
		
		addFocusListenerTo(fragmentCombo);
    }
    
    protected void createCode(Composite parent) {
		Composite page = new Composite(parent, SWT.NONE);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		page.setLayout(pageLayout);
		GridData pageData = GridUtil.createFill();
		page.setLayoutData(pageData);
    	
    	_codeLabelImage = new Label(page, SWT.NONE);
    	_codeLabelImage.setImage(getCodeImage());
    	
		widgetFactory.createLabel(page, ProbekitMessages._11, SWT.NONE); 

		IPreferenceStore store = PreferenceConstants.getPreferenceStore();
		_javaTextTools = new JavaTextTools(store);
		_methodViewer = new SourceViewer(page, null, null, false, ProbekitWidgetFactory.BORDER_STYLE | SWT.V_SCROLL | SWT.H_SCROLL | SWT.MULTI);
		_methodViewerConfiguration = new JavaSourceViewerConfiguration(_javaTextTools.getColorManager(), store, null, null);
		_methodViewer.configure(_methodViewerConfiguration);
		IDocument document= new Document();
		
		//	setup Java partitioner
		IDocumentPartitioner partitioner = _javaTextTools.createDocumentPartitioner();
		document.setDocumentPartitioner(partitioner);
		partitioner.connect(document);
		
		_methodViewer.setDocument(document);
		_methodViewer.setEditable(true);
		GridData descLayout = new GridData(GridData.FILL_BOTH);
		descLayout.horizontalSpan = 2;
		_methodViewer.getTextWidget().setLayoutData(descLayout);
		
		// add listeners - don't change the order in which the listeners are
		// added
		// because of team support issues modify listeners should always be the
		// last one added
		addFocusListenerTo(_methodViewer.getTextWidget());
		
		_methodViewer.getTextWidget().addModifyListener(new CodeModifyListener());
	}
    
	private void createTable(Composite parent) {
		Composite page = new Composite(parent, SWT.BORDER);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		page.setLayout(pageLayout);
		GridData pageData = GridUtil.createHorizontalFill();
		page.setLayoutData(pageData);

		createTableTitle(page);
		createErrorMessage(page);
    	createTableViewer(page);
		createTableButtons(page);
	}
	
	private void createTableTitle(Composite page) {
    	Label label = widgetFactory.createLabel(page, ProbekitMessages._24, SWT.NONE);
    	GridData data = new GridData();
    	data.horizontalSpan = 2;
    	label.setLayoutData(data);
	}
	
	private void createErrorMessage(Composite page) {
		_errorMessage = widgetFactory.createLabel(page, ResourceUtil.NO_TEXT, SWT.WRAP);
		GridData data = new GridData();
		data.horizontalSpan = 2;
		_errorMessage.setLayoutData(data);
		_errorMessage.setBackground(page.getBackground());
		_errorMessage.setForeground(PlatformUI.getWorkbench().getDisplay().getSystemColor(SWT.COLOR_RED));
		_errorMessage.setFont(page.getFont());
	}

	protected void setErrorMessage(String message) {
		String text = ResourceUtil.getString(message);
		_errorMessage.setForeground(PlatformUI.getWorkbench().getDisplay().getSystemColor(SWT.COLOR_RED));
		_errorMessage.setText(text);
		_errorMessage.pack(true);
	}
	
	void validate() {
		String errorMessage = null;
		DataItem[] items = (DataItem[])currentSelection.getData().toArray(new DataItem[0]);
		if(!ProbekitTypesSingleton.singleton().isValidDataItemNames(items)) {
			errorMessage = ProbekitMessages._42;
		}
		
		if(!ProbekitTypesSingleton.singleton().isValidDataItemTypes(items)) {
			errorMessage = ProbekitMessages._43;
		}

		setErrorMessage(errorMessage);
	}
	
	private void createTableViewer(Composite page) {
		_tableViewer = new TableViewer(page, SWT.BORDER | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION );
		TableLayout tableLayout = new TableLayout();
		Table table = _tableViewer.getTable();
		
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];
		
		TableColumn comboCell = new TableColumn(table, SWT.LEFT);
		comboCell.setText(COLUMNS[0]);
		tableLayout.addColumnData(new ColumnWeightData(50));
		String[] comboboxItems = getAvailableDataTypeNames();
		_comboBoxCellEditor = 
			new ComboBoxCellEditor(
					table, 
					comboboxItems
			);
		cellEditors[0] = _comboBoxCellEditor;
		
		TableColumn textCell = new TableColumn(table, SWT.LEFT);
		textCell.setText(COLUMNS[1]);
		tableLayout.addColumnData(new ColumnWeightData(50));
		_textCellEditor = new TextCellEditor(table);
		cellEditors[1] = _textCellEditor;

		_tableViewer.setCellEditors(cellEditors);
		_tableViewer.setCellModifier(new DataItemCellModifier());
		_tableViewer.setContentProvider(new DataItemContentProvider());
		_tableViewer.setLabelProvider(new DataItemColumnLabelProvider());
		_tableViewer.setColumnProperties(COLUMNS);
		
		GridData tableData = new GridData(GridData.FILL_BOTH);
		table.setLayoutData(tableData);
		table.setLayout(tableLayout);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setFont(page.getFont());
		// Add the button listener to the table so that when the 
		// user edits the table, the listener will notify the 
		// viewer that a change has been made, and the viewer will
		// be marked dirty.
		table.addSelectionListener(new ButtonListener());
	}
	
	private void createTableButtons(Composite page) {
		ButtonListener listener = new ButtonListener();
		
		Composite buttonGroup = new Composite(page, SWT.NONE);
		buttonGroup.setFont(page.getFont());
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.makeColumnsEqualWidth = true;
		buttonGroup.setLayout(buttonLayout);
		GridData buttonData = new GridData();
		buttonGroup.setLayoutData(buttonData);
		
		_addButton = new Button(buttonGroup, SWT.PUSH);
		_addButton.setText(ProbekitMessages._125);
		GridData addData = GridUtil.createHorizontalFill();
		addData.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
		_addButton.setLayoutData(addData);
		_addButton.addSelectionListener(listener);
		
		_editButton = new Button(buttonGroup, SWT.PUSH);
		_editButton.setText(ProbekitMessages._118);
		GridData editData = GridUtil.createHorizontalFill();
		editData.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
		_editButton.setLayoutData(editData);
		_editButton.addSelectionListener(listener);
		
		_removeButton = new Button(buttonGroup, SWT.PUSH);
		_removeButton.setText(ProbekitMessages._119);
		GridData removeData = GridUtil.createHorizontalFill();
		removeData.verticalAlignment = GridData.VERTICAL_ALIGN_BEGINNING;
		_removeButton.setLayoutData(removeData);
		_removeButton.addSelectionListener(listener);
		
		refreshButtons();
	}
	
	private void refreshButtons() {
		DataType[] availableTypes = getAvailableDataTypes();
		_addButton.setEnabled(availableTypes.length > 0);
		DataItem dataItem = getSelectedItem();
		boolean enabled = (dataItem != null);
		_editButton.setEnabled(enabled);
		_removeButton.setEnabled(enabled);
	}
	
	void refreshCodeLabel() {
		_codeLabelImage.setImage(getCodeImage());
	}
	
	public void refresh() {
		refreshCodeLabel();
		refreshTable();
		refreshButtons();
		validate();
	}
	
	void refreshTable() {
		_tableViewer.refresh();
	}
	
	public void display(Object object) {
		if (fragmentCombo.isFocusControl() || _methodViewer.getTextWidget().isFocusControl()) {
			refreshCodeLabel();
			return;
		}
		
		Fragment fragment = null;
		if(object instanceof Fragment)
			fragment = (Fragment)object;
		else if(object instanceof DataItem)
		{
			Object container = ((DataItem)object).eContainer();
			if(container instanceof Fragment)
				fragment = (Fragment)container;									
		}
		if(fragment == null)
		{
			setCurrentSelection(null);
			return;
		}
		setCurrentSelection(fragment);
		
		isDisplaying = true;
		FragmentType frag = currentSelection.getType();
		if (frag != null) {
			fragmentCombo.setText(frag.getName());
		}
		
		_methodViewer.getTextWidget().setText(currentSelection.getCode() == null ? "" : currentSelection.getCode());
//		setFocusTo();
		_methodViewer.getTextWidget().setFocus();
		isDisplaying = false;
	}
	
	private Image getCodeImage() {
		return DialogUtil.getImage((URL)itemProvider.getImage(currentSelection));
	}
	
	public void setCurrentSelection(Fragment fragment) {
		currentSelection = fragment;
		_tableViewer.setInput(currentSelection.getData());
		refresh();
	}

	public boolean updateCurrentSelection(Object currentSelection) {
		setCurrentSelection((Fragment) currentSelection);
		this.currentSelection.setType(FragmentType.get(fragmentCombo.getText().trim()));
		return false;
	}
	
	public void setFocusToText() {
		fragmentCombo.setFocus();
	}
	
	public void dispose() {
		itemFactory = null;
		itemProvider = null;
		fragmentCombo = null;
		_methodViewer = null;
		_javaTextTools = null;
		_methodViewerConfiguration = null;
	}
	
	public boolean isDetailsFor(Object object) {
		if (object instanceof Fragment || object instanceof FragmentImpl) {
			return true;
		}
		if(object instanceof DataItem || object instanceof DataItemImpl) {
			return true;
		}
		return false;
	}

	public void setFocusTo() {
		refreshCodeLabel();
		setFocusToText();
	}

	DataItem getSelectedItem() {
		int idx = _tableViewer.getTable().getSelectionIndex();
		if (idx == -1) {
			return null;
		}
		else {
			return (DataItem) _tableViewer.getElementAt(idx);
		}
	}
	
	public void createDataItem() {
		DataType[] availableTypes = getAvailableDataTypes();
		DataItem item = ProbekitTypesSingleton.singleton().createDataItem(availableTypes, currentSelection.getData().size());

		itemProvider.addAndNotify(currentSelection, item);
	}
		
	public void updateDataName(DataItem dataItem, String newName) {		
		EditingDomain domain = AdapterFactoryEditingDomain.getEditingDomainFor(dataItem);
		if(domain != null) {
			SetCommand command = new SetCommand(domain, dataItem, ProbekitPackage.eINSTANCE.getDataItem_Name(), newName);
			domain.getCommandStack().execute(command);
		}
	}
	
	public void updateDataType(DataItem dataItem, String newType) {
		EditingDomain domain = AdapterFactoryEditingDomain.getEditingDomainFor(dataItem);
		if(domain != null) {
			DataType type = DataType.get(newType);
			SetCommand command = new SetCommand(domain, dataItem, ProbekitPackage.eINSTANCE.getDataItem_Type(), type);
			domain.getCommandStack().execute(command);
		}
	}
	
	public void removeDataItem(DataItem item) {
		itemProvider.removeAndNotify(currentSelection, item);
	}
	
	public DataType[] getAvailableDataTypes() {
		if(currentSelection == null) {
			return new DataType[0];
		}
		else {
			return ProbekitTypesSingleton.singleton().getDataTypes(currentSelection.getType());
		}
	}
	
	public String[] getAvailableDataTypeNames() {
		DataType[] allTypes = getAvailableDataTypes();
		String[] allTypeNames = new String[allTypes.length];
		for(int i=0; i<allTypes.length; i++) {
			allTypeNames[i] = allTypes[i].getName();
		}
		return allTypeNames;
	}
	
	private class CodeModifyListener implements ModifyListener {
		public void modifyText(ModifyEvent e) {
			if (currentSelection == null)
				return;
			if (e.getSource() == _methodViewer.getTextWidget()) {
				String text = _methodViewer.getTextWidget().getText();
				if (!text.equals(currentSelection.getCode())) {
					if (text.length() == 0 && currentSelection.getCode() == null)
						return;
					IItemPropertyDescriptor descriptor = itemProvider.getPropertyDescriptor(currentSelection, ProbekitPackage.eINSTANCE.getFragment_Code());
					if (descriptor != null)
						descriptor.setPropertyValue(currentSelection, text);
				}
			}
		}
	}
	
	private class ComboSelectionListener implements SelectionListener {
		public void widgetSelected(SelectionEvent e) {
			if (isDisplaying || currentSelection == null)
				return;
			if (e.getSource() == fragmentCombo) {
				String text = fragmentCombo.getText().trim();
				if (!text.equals(currentSelection.getType().getName())) {
					if (text.length() == 0 && currentSelection.getType() == null)
						return;
					IItemPropertyDescriptor descriptor = itemProvider.getPropertyDescriptor(currentSelection, TYPE);
					if (descriptor != null)
						descriptor.setPropertyValue(currentSelection, FragmentType.get(text));
				}
			}
		}
	
		public void widgetDefaultSelected(SelectionEvent arg0) {
		}
	}

	private class DataItemColumnLabelProvider extends LabelProvider implements ITableLabelProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		
		public String getColumnText(Object element, int columnIndex) {
			if(element instanceof DataItem) {
				DataItem item = (DataItem)element;
				if(columnIndex == 0) {
					return ProbekitUtil.getDataTypeName(item);
				}
				else {
					return ProbekitUtil.getDataItemName(item);
				}
			}
			return ResourceUtil.NO_TEXT;
		}
	}

	private class DataItemContentProvider implements IStructuredContentProvider {
		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object[] getElements(Object inputElement) {
			return currentSelection.getData().toArray();
		}
	}
	
	private class DataItemCellModifier implements ICellModifier {
		public Object getValue(Object element, String property) {
			Object value = null;
			if (element instanceof DataItem) {
				DataItem item = (DataItem)element;
				if (property.equals(COLUMNS[0])) {
					refreshComboBeforeEdit();
					String typeName = ProbekitUtil.getDataTypeName(item);
					value = getCellIndex(typeName);
				}
				else if (property.equals(COLUMNS[1])) {
					value = ProbekitUtil.getDataItemName(item);
				}
			}
			return value;
		}
		
		private Integer getCellIndex(String dataTypeName) {
			Integer index = null;
			String[] items = _comboBoxCellEditor.getItems();
			for(int i=0; i<items.length; i++) {
				String itemText = items[i];
				if(itemText.equals(dataTypeName)) {
					index = new Integer(i);
					break;
				}
			}
			
			if(index == null) {
				index = new Integer(0);
			}
			
			return index;
		}

		public boolean canModify(Object element, String property) {
			return true;
		}

		public void modify(Object element, String property, Object value) {
			TableItem item = (TableItem) element;
			DataItem dataItem = (DataItem) item.getData();
			if(property.equals(COLUMNS[0])) {
				Integer cellIndex = (Integer)value;
				String[] items = _comboBoxCellEditor.getItems();
				String typeName = items[cellIndex.intValue()];
				updateDataType(dataItem,typeName);
			}
			else if(property.equals(COLUMNS[1])) {
				String itemName = (String)value;
				updateDataName(dataItem,itemName);
			}
			refresh();
		}

		private void refreshComboBeforeEdit() {
			_comboBoxCellEditor.setItems(
				getAvailableDataTypeNames()
			);
		}
	
	};
	
	private class ButtonListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing to do when the default is selected.
		}

		public void widgetSelected(SelectionEvent event) {
			if(event.widget.equals(_addButton)) {
				add();
			}
			else if (event.widget.equals(_editButton)) {
				edit();
			}
			else if(event.widget.equals(_removeButton)) {
				remove();
			}
			else if(event.widget.equals(_tableViewer.getTable())) {
				refresh();
				// let cell editors work
				return;
			}
			
			refresh();
		}
		
		private void add() {
			createDataItem(); 
			
			// Refresh before selecting so that the new item is added to the table,
			// and refresh after selecting so that the "Edit" and "Remove" buttons
			// are enabled.
			refreshTable();		
			select(_tableViewer.getTable().getItemCount()-1);
		}
		
		private void edit() {
			DataItem item = getSelectedItem();
			if(item != null) {
				String[] allTypeNames = getAvailableDataTypeNames();
				
				DataItem updatedItem = ProbekitPackage.eINSTANCE.getProbekitFactory().createDataItem();
				updatedItem.setName(item.getName());
				updatedItem.setType(item.getType());
				
				DataDialog dialog = new DataDialog(
						getShell(),
						allTypeNames,
						updatedItem);
				dialog.setBlockOnOpen(true);
				dialog.setTitle(ProbekitMessages._44);
				
				if(dialog.open() != DataDialog.OK) {
					return;
				}
				
				updatedItem = dialog.getResult();
				
				if(!item.getName().equals(updatedItem.getName())){
					updateDataName(item,updatedItem.getName());
				}
				if(!item.getType().equals(updatedItem.getType())){
					updateDataType(item,updatedItem.getType().getName());
				}
			}
		}
		
		private void remove() {
			DataItem item = getSelectedItem();
			int index = _tableViewer.getTable().getSelectionIndex();
			if(item != null) {
				removeDataItem(item);
				index -= 1;
			}
			
			refreshTable();
			select(index);
		}

		private void select(int index) {
			_tableViewer.getTable().select(index);
		}
	}
}