/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: GeneralTab.java,v 1.4 2005/12/30 02:58:09 popescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.hyades.probekit.editor.internal.presentation;

import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.edit.command.CreateChildCommand;
import org.eclipse.emf.common.command.Command; 
import org.eclipse.emf.edit.command.AddCommand;
import org.eclipse.emf.edit.command.RemoveCommand;
import org.eclipse.emf.edit.command.SetCommand;
import org.eclipse.emf.edit.domain.EditingDomain;
import org.eclipse.hyades.models.internal.probekit.Label;
import org.eclipse.hyades.models.internal.probekit.Probekit;
import org.eclipse.hyades.models.internal.probekit.ProbekitFactory;
import org.eclipse.hyades.models.internal.probekit.ProbekitPackage;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;



public class GeneralTab extends Composite {
	private static final String LABEL_DEFAULT_NAME_PREFIX = "a"; //$NON-NLS-1$
	protected final String[] COLUMNS = new String[]{
		ProbekitMessages._112, 
		ProbekitMessages._113, 
		ProbekitMessages._114};
	private static final String LANGUAGE_DEFAULT = ResourceUtil.getLanguageDefault();
	private static final String DESCRIPTION_DEFAULT = ResourceUtil.NO_TEXT;

	private Probekit _probekit;
	private TableViewer _tableViewer;
	private Button _addButton;
	private Button _removeButton;
	private Button _editButton;
	private Text _errorMessage;
	private TextCellEditor _nameCellEditor;
	private ComboBoxCellEditor _languageBoxCellEditor;
	private TextCellEditor _descriptionCellEditor;
	private EditingDomain _editingDomain;
	private Set _languagesInUse;
	private Text _idField;
	private Text _versionField;
	public final GeneralTabSelectionListener GENERAL_TAB_SELECTION_LISTENER = new GeneralTabSelectionListener();
	
	public GeneralTab(EditingDomain domain, Composite parent, int style, Probekit probekit) {
		super(parent, style);
		_probekit = probekit;
		_editingDomain = domain;
		_languagesInUse = new HashSet();
		
		createControl(this);
	}
	
	protected Probekit getProbekit() {
		return _probekit;
	}
	
	public void dispose() {
		_editingDomain = null;
		_languagesInUse = null;
		_addButton = null;
		_removeButton = null;
		_editButton = null;
		_errorMessage = null;
		_nameCellEditor = null;
		_languageBoxCellEditor = null;
		_descriptionCellEditor = null;
		_idField = null;
		_versionField = null;
		_probekit = null;
		_tableViewer = null;
	}
	
	public boolean isPageFor(Object object) {
		if((object instanceof Probekit) || 
			(object instanceof Label)) {
			return true;
		}
		return false;
	}
	
	private void createControl(Composite parent) {
		GridData pageData = GridUtil.createFill();
		parent.setLayoutData(pageData);
		GridLayout pageLayout = new GridLayout();
		parent.setLayout(pageLayout);
		
		createErrorMessage(parent);
		createLabels(parent);
		createTable(parent, getProbekit());
		validate();
	}
	
	private void createLabels(Composite parent) {
		ProbekitListener listener = new ProbekitListener();
		Composite page = new Composite(parent, SWT.NONE);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		page.setLayout(layout);
		page.setLayoutData(GridUtil.createHorizontalFill());
		
		org.eclipse.swt.widgets.Label idLabel = new org.eclipse.swt.widgets.Label(page, SWT.NONE);
		idLabel.setText(ProbekitMessages._116);
		GridData idLabelData = new GridData(GridData.BEGINNING);
		idLabel.setLayoutData(idLabelData);
		
		_idField = new Text(page, SWT.LEFT | SWT.BORDER);
		GridData gd = GridUtil.createHorizontalFill();
		_idField.setLayoutData(gd);
		_idField.setText(ResourceUtil.getString(getProbekit().getId()));
		_idField.addModifyListener(listener);
		_idField.setFocus();
		
		org.eclipse.swt.widgets.Label versionLabel = new org.eclipse.swt.widgets.Label(page, SWT.NONE);
		versionLabel.setText(ProbekitMessages._117);
		GridData versionLabelData = new GridData(GridData.BEGINNING);
		versionLabel.setLayoutData(versionLabelData);
		
		_versionField = new Text(page, SWT.LEFT | SWT.BORDER);
		GridData gd2 = GridUtil.createHorizontalFill();
		_versionField.setLayoutData(gd2);
		_versionField.setText(ResourceUtil.getString(getProbekit().getVersion()));
		_versionField.addModifyListener(listener);
	}
	
	private void createTable(Composite parent, Probekit probekit) {
		Composite page = new Composite(parent, SWT.NONE);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		page.setLayout(pageLayout);
		GridData pageData = GridUtil.createFill();
		page.setLayoutData(pageData);

		createTableViewer(page);
		createTableButtons(page);
	}
	
	private void createErrorMessage(Composite page) {
		_errorMessage = new Text(page, SWT.READ_ONLY | SWT.SINGLE | SWT.WRAP);
		_errorMessage.setLayoutData(GridUtil.createHorizontalFill());
		_errorMessage.setBackground(page.getBackground());
		_errorMessage.setForeground(PlatformUI.getWorkbench().getDisplay().getSystemColor(SWT.COLOR_RED));
		_errorMessage.setFont(page.getFont());
	}
	
	private void createTableViewer(Composite page) {
		_tableViewer = new TableViewer(page, SWT.BORDER | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION );
		TableLayout tableLayout = new TableLayout();
		Table table = _tableViewer.getTable();
		
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];
		
		TableColumn nameCell = new TableColumn(table, SWT.LEFT);
		nameCell.setText(COLUMNS[0]);
		tableLayout.addColumnData(new ColumnWeightData(40));
		_nameCellEditor = new TextCellEditor(table);
		cellEditors[0] = _nameCellEditor;

		TableColumn comboCell = new TableColumn(table, SWT.LEFT);
		comboCell.setText(COLUMNS[1]);
		tableLayout.addColumnData(new ColumnWeightData(20));
		String[] comboboxItems = ResourceUtil.getLanguages();
		_languageBoxCellEditor = 
			new ComboBoxCellEditor(
					table, 
					comboboxItems
			);
		cellEditors[1] = _languageBoxCellEditor;
		
		TableColumn textCell = new TableColumn(table, SWT.LEFT);
		textCell.setText(COLUMNS[2]);
		tableLayout.addColumnData(new ColumnWeightData(40));
		_descriptionCellEditor = new TextCellEditor(table);
		cellEditors[2] = _descriptionCellEditor;

		_tableViewer.setCellEditors(cellEditors);
		_tableViewer.setCellModifier(new LanguageCellModifier());
		_tableViewer.setContentProvider(new LanguageContentProvider());
		_tableViewer.setLabelProvider(new LanguageColumnLabelProvider());
		_tableViewer.setInput(getLabels());
		_tableViewer.setColumnProperties(COLUMNS);
		
		GridData tableData = new GridData(GridData.FILL_BOTH);
		table.setLayoutData(tableData);
		table.setLayout(tableLayout);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setFont(page.getFont());
		table.addSelectionListener(new LanguageSelectionListener());
	}
	
	private void createTableButtons(Composite page) {
		LanguageSelectionListener listener = new LanguageSelectionListener();
		
		Composite buttonGroup = new Composite(page, SWT.NONE);
		buttonGroup.setFont(page.getFont());
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.makeColumnsEqualWidth = true;
		buttonGroup.setLayout(buttonLayout);
		GridData buttonData = GridUtil.createVerticalFill();
		buttonGroup.setLayoutData(buttonData);
		
		_addButton = new Button(buttonGroup, SWT.PUSH);
		_addButton.setText(ProbekitMessages._125);
		GridData addData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_addButton.setLayoutData(addData);
		_addButton.addSelectionListener(listener);
		
		_editButton = new Button(buttonGroup, SWT.PUSH);
		_editButton.setText(ProbekitMessages._118);
		GridData editData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_editButton.setLayoutData(editData);
		_editButton.addSelectionListener(listener);
		
		_removeButton = new Button(buttonGroup, SWT.PUSH);
		_removeButton.setText(ProbekitMessages._119);
		GridData removeData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_removeButton.setLayoutData(removeData);
		_removeButton.addSelectionListener(listener);
		
		refreshButtons();
	}
	
	private void refreshButtons() {
		_addButton.setEnabled(true);
		Label item = getSelectedItem();
		boolean enabled = (item != null);
		_editButton.setEnabled(enabled);
		_removeButton.setEnabled(enabled);
	}
	
	Label getSelectedItem() {
		int idx = _tableViewer.getTable().getSelectionIndex();
		if (idx == -1) {
			return null;
		}
		else {
			return (Label) _tableViewer.getElementAt(idx);
		}
	}
	
	List getLabels() {
		return getProbekit().getLabel();
	}
	
	void refresh() {
		refreshTable();
		refreshButtons();
	}
	
	void refreshTable() {
		_tableViewer.refresh();
	}
	
	void validate() {
		String errorMessage = null;
		
		List labels = getLabels();
		Iterator iterator = labels.iterator();
		while(iterator.hasNext()) {
			Label label = (Label)iterator.next();
			_languagesInUse.add(ResourceUtil.getString(label.getLang()));
		}
		if(_languagesInUse.size() != labels.size()) {
			// Duplicate somewhere...
			errorMessage = ProbekitMessages._115;
		}

		setErrorMessage(errorMessage);
		_languagesInUse.clear();
	}

	public void createLabel() {
		Label label = ProbekitFactory.eINSTANCE.createLabel();
		label.setLang(LANGUAGE_DEFAULT);
		label.setName(getUniqueLabelName(LANGUAGE_DEFAULT));
		label.setDescription(DESCRIPTION_DEFAULT);

		CreateChildCommand command = new CreateChildCommand(_editingDomain, _probekit, ProbekitPackage.eINSTANCE.getProbekit_Label(), label, Collections.EMPTY_LIST);
		_editingDomain.getCommandStack().execute(command);
	}
	
	// For performance reasons, don't iterate over the list of used names
	// to guarantee that the name is unique. 
	private String getUniqueLabelName(String language) {
		int count = _probekit.getLabel().size();
		StringBuffer buffer = new StringBuffer(LABEL_DEFAULT_NAME_PREFIX);
		buffer.append(language);
		buffer.append(count);
		return buffer.toString();
	}
	
	public void updateLabel(int index, Label label, Label oldLabel) {
		// label's unique attribute is true, and it's already in the probekit;
		// so can't just call SetCommand
		Command command = new RemoveCommand(_editingDomain, _probekit, ProbekitPackage.eINSTANCE.getProbekit_Label(), oldLabel);
		_editingDomain.getCommandStack().execute(command);
		command = new AddCommand(_editingDomain, _probekit, ProbekitPackage.eINSTANCE.getProbekit_Label(), label, index);
		_editingDomain.getCommandStack().execute(command);
	}
	
	public void updateProbekit() {
		EObject documentRoot = _probekit.eContainer();
		SetCommand command = new SetCommand(_editingDomain, documentRoot, ProbekitPackage.eINSTANCE.getDocumentRoot_Probekit(), _probekit);
		_editingDomain.getCommandStack().execute(command);
	}
	
	public void removeLabel(Label label) {
		RemoveCommand command = new RemoveCommand(_editingDomain, _probekit, ProbekitPackage.eINSTANCE.getProbekit_Label(), label);
		_editingDomain.getCommandStack().execute(command);
	}
	
	protected void setErrorMessage(String message) {
		String text = ResourceUtil.getString(message);
		_errorMessage.setText(text);
		_errorMessage.pack(true);
	}
	
	public boolean setFocus() {
		// Set the focus to the default widget on this page
		super.setFocus();
		return _idField.setFocus();
	}
	
	protected boolean isEqual(Label label1, Label label2) {
		if(label1 == label2) {
			return true;
		}
		
		if(!isEqual(label1.getName(), label2.getName())) {
			return false;
		}
		
		if(!isEqual(label1.getLang(), label2.getLang())) {
			return false;
		}
		
		if(!isEqual(label1.getDescription(), label2.getDescription())) {
			return false;
		}
		
		return true;
	}
	
	protected boolean isEqual(String string1, String string2) {
		if(string1 == null) {
			if(string2 == null) {
				return true;
			}
			else {
				return false;
			}
		}
		else {
			return string1.equals(string2);
		}	
	}

	private class LanguageSelectionListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing to do when the default is selected.
		}

		public void widgetSelected(SelectionEvent event) {
			if(event.widget.equals(_addButton)) {
				add();
			}
			else if (event.widget.equals(_editButton)) {
				edit();
			}
			else if(event.widget.equals(_removeButton)) {
				remove();
			} else {
				// if not return now for other cases, cell editors won't work
				refreshButtons();
				return;
			}
			
			refresh();
			validate();
		}
		
		private void add() {
			createLabel();
			
			// Refresh before selecting so that the new item is added to the table,
			// and refresh after selecting so that the "Edit" and "Remove" buttons
			// are enabled.
			refreshTable();		
			select(_tableViewer.getTable().getItemCount()-1);
		}
		
		private void edit() {
			Label item = getSelectedItem();
			if(item != null) {
				// Can't select a single cell in a row.
				// Launch an editor dialog instead.
				LabelDialog dialog = new LabelDialog(
							getParent().getShell(),
							ResourceUtil.getLanguages(),
							item);
				dialog.setBlockOnOpen(true);
				dialog.setTitle(ProbekitMessages._137);
				dialog.open();
				
				int index = _tableViewer.getTable().getSelectionIndex();
				Label updatedItem = dialog.getResult();
				if(!isEqual(item, updatedItem)) {
					updateLabel(index, updatedItem, item);
					refreshTable();
					select(index);
				}
			}
		}
		
		private void remove() {
			Label item = getSelectedItem();
			int index = _tableViewer.getTable().getSelectionIndex();
			if(item != null) {
				removeLabel(item);
				index -= 1;
			}
			
			refreshTable();
			select(index);
		}

		private void select(int index) {
			_tableViewer.getTable().select(index);
		}
	}
	
	private class LanguageCellModifier implements ICellModifier {
		public Object getValue(Object element, String property) {
			Object value = null;
			if (element instanceof Label) {
				Label item = (Label)element;
				if (property.equals(COLUMNS[0])) {
					value = ResourceUtil.getString(item.getName());
				}
				else if (property.equals(COLUMNS[1])) {
					// For performance reasons, don't constrain the contents of the combo
					// box to only the valid choices. (Given the number of valid language
					// locales -- over 50 -- the string comparisons would take too long.)
					// Instead, validate the user's choice afterwards and display an error
					// message if there is a language used more than once.
					String typeName = ResourceUtil.getString(item.getLang());
					value = getCellIndex(typeName);
				}
				else if (property.equals(COLUMNS[2])) {
					value = ResourceUtil.getString(item.getDescription());
				}
			}
			return value;
		}
		
		private Integer getCellIndex(String name) {
			Integer index = null;
			String[] items = _languageBoxCellEditor.getItems();
			for(int i=0; i<items.length; i++) {
				String itemText = items[i];
				if(itemText.equals(name)) {
					index = new Integer(i);
					break;
				}
			}
			
			if(index == null) {
				index = new Integer(-1);
			}
			
			return index;
		}

		public boolean canModify(Object element, String property) {
			return true;
		}

		public void modify(Object element, String property, Object value) {
			TableItem item = (TableItem) element;
			Label label = (Label) item.getData();
			int index = _tableViewer.getTable().getSelectionIndex();
			boolean isDirty = false;
			if(property.equals(COLUMNS[0])) {
				String itemName = (String)value;
				if(!itemName.equals(label.getName())) {
					isDirty = true;
					label.setName(itemName);
				}
			}
			else if(property.equals(COLUMNS[1])) {
				Integer cellIndex = (Integer)value;
				String itemName = "";
				if(cellIndex.intValue() < 0) {
					// User typed in language instead of selecting it from the list
					itemName = ((CCombo)_languageBoxCellEditor.getControl()).getText();
					Integer languageIndex = getCellIndex(itemName);
					if(languageIndex.intValue() < 0) {
						// Add the new language to the list.
						String[] items = _languageBoxCellEditor.getItems();
						String[] newItems = new String[items.length + 1];
						System.arraycopy(items, 0, newItems, 0, items.length);
						newItems[items.length] = itemName;
						_languageBoxCellEditor.setItems(newItems);
					}
				}
				else {
					String[] items = _languageBoxCellEditor.getItems();
					itemName = items[cellIndex.intValue()];
				}
				if(!itemName.equals(label.getLang())) {
					isDirty = true;
					label.setLang(itemName);
				}
			}
			else if(property.equals(COLUMNS[2])) {
				String itemName = (String)value;
				if(!itemName.equals(label.getDescription())) {
					isDirty = true;
					label.setDescription(itemName);
				}
			}
			
			if(isDirty) {
				updateLabel(index, label, label);
				refresh();
				validate();
			}
		}
	}
	
	private class LanguageContentProvider implements IStructuredContentProvider {
		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object[] getElements(Object inputElement) {
			return getLabels().toArray();
		}
	}
	
	private static class LanguageColumnLabelProvider extends LabelProvider implements ITableLabelProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		
		public String getColumnText(Object element, int columnIndex) {
			if(element instanceof Label) {
				Label item = (Label)element;
				if(columnIndex == 0) {
					return ResourceUtil.getString(item.getName());
				}
				else if(columnIndex == 1) {
					return ResourceUtil.getString(item.getLang());
				}
				else {
					return ResourceUtil.getString(item.getDescription());
				}
			}
			return ResourceUtil.NO_TEXT;
		}
	}
	
	private class ProbekitListener implements ModifyListener {
		public void modifyText(ModifyEvent e) {
			_probekit.setId(_idField.getText());
			_probekit.setVersion(_versionField.getText());
			updateProbekit();
			validate();
		}
	}
	
	private class GeneralTabSelectionListener implements ISelectionChangedListener {
		public void selectionChanged(SelectionChangedEvent event) {
			if ((event.getSource() != null)	&& (event.getSource() instanceof ProbekitEditor)) {
				ProbekitEditor editor = (ProbekitEditor)event.getSource();
	            ISelection selection = event.getSelection();
	            Object currentSelection = null;
	            if((selection != null) && (!selection.isEmpty()) && (selection instanceof IStructuredSelection)) {
					currentSelection = ((IStructuredSelection)selection).getFirstElement();	
	            }
	              
				if(currentSelection instanceof Probekit) {
					editor.setActivePage(GeneralTab.this); // Only change the page if the EObject is displayed on this page.
					_idField.setFocus();
				}
				else if(currentSelection instanceof Label) {
					editor.setActivePage(GeneralTab.this);
					Label selected = (Label)currentSelection;
					TableItem[] items = _tableViewer.getTable().getItems();
					for(int i=0; i<items.length; i++) {
						TableItem item = items[i];
						Label tableLabel = (Label)item.getData();
						if(isEqual(tableLabel, selected)) {
							_tableViewer.getTable().setSelection(i);
							refreshButtons();
							break;
						}
					}
				}
			}
		}
	}
}
