/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: ProbekitWidgetFactory.java,v 1.1 2005/06/07 04:04:15 popescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/

package org.eclipse.hyades.probekit.editor.internal.presentation;

import java.util.Enumeration;
import java.util.Hashtable;

import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.custom.StyledText;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.GC;
import org.eclipse.swt.graphics.Rectangle;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Canvas;
import org.eclipse.swt.widgets.Combo;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.Text;
import org.eclipse.swt.widgets.Tree;

public class ProbekitWidgetFactory {

	public static final String DEFAULT_HEADER_COLOR = "__default__header__";
	public static final String COLOR_BACKGROUND = "__bg";
	public static final String COLOR_BORDER = "__border";
	public static final String COLOR_COMPOSITE_SEPARATOR = "__compSep";

	private Hashtable colorRegistry = new Hashtable();
	private Color backgroundColor;
	private Color clientAreaColor;
	private Color foregroundColor;
	private Color disabledColor;
	private Color darkGrayColor;
	private Display display;
	public static final int BORDER_STYLE = SWT.NONE;
	private BorderPainter borderPainter;
	private Color borderColor;

	class BorderPainter implements PaintListener {
		public void paintControl(PaintEvent event) {
			Composite composite = (Composite) event.widget;
			Control[] children = composite.getChildren();

			for (int i = 0; i < children.length; i++) {
				Control c = children[i];

				if (c.isVisible() == false)
					continue;

				if (c instanceof Text
					|| c instanceof StyledText
					|| c instanceof Canvas
					|| c instanceof Combo
					|| c instanceof CCombo) {
					Rectangle b = c.getBounds();
					GC gc = event.gc;
					if (c.isEnabled() == true)
						c.setBackground(backgroundColor);
					else
						c.setBackground(disabledColor);
					c.setForeground(foregroundColor);
					gc.drawRectangle(
						b.x - 2,
						b.y - 2,
						b.width + 3,
						b.height + 3);
				} else if (c instanceof Tree || c instanceof Table) {
					Rectangle b = c.getBounds();
					GC gc = event.gc;
					gc.setForeground(borderColor);
					if (c.isEnabled() == true)
						c.setBackground(backgroundColor);
					else
						c.setBackground(disabledColor);
					gc.drawRectangle(
						b.x - 1,
						b.y - 1,
						b.width + 2,
						b.height + 2);
				}
			}
		}
	}

	public ProbekitWidgetFactory() {
		this(Display.getCurrent());
	}
	public ProbekitWidgetFactory(Display display) {
		this.display = display;
		initialize();
	}

	public Composite createComposite(Composite parent) {
		return createComposite(parent, SWT.NULL);
	}
	public Composite createComposite(Composite parent, int style) {
		Composite composite = new Composite(parent, style);
		composite.setBackground(backgroundColor);
		return composite;
	}

	public Canvas createCanvas(Composite parent) {
		return createCanvas(parent, SWT.NULL);
	}

	public Canvas createCanvas(Composite parent, int style) {
		Canvas canvas = new Canvas(parent, style);
		canvas.setBackground(backgroundColor);
		canvas.setForeground(foregroundColor);
		return canvas;
	}

	public Button createButton(Composite parent, String text, int style) {
		int flatStyle = BORDER_STYLE == SWT.BORDER ? SWT.NULL : SWT.FLAT;

		Button button = new Button(parent, style | flatStyle);
		button.setBackground(backgroundColor);
		button.setForeground(foregroundColor);
		if (text != null)
			button.setText(text);
		return button;
	}

	public Composite createCompositeSeparator(Composite parent) {
		Composite composite = new Composite(parent, SWT.NONE);
		composite.setBackground(getColor(COLOR_COMPOSITE_SEPARATOR));
		GridData sepGrid = new GridData(GridData.FILL_HORIZONTAL);
		sepGrid.heightHint = 3;
		composite.setLayoutData(sepGrid);

		return composite;
	}

	public Label createLabel(Composite parent, String text) {
		return createLabel(parent, text, SWT.NONE);
	}

	public Group createGroup(Composite parent, String text, int style) {
		Group groupWidget = new Group(parent, SWT.FLAT);
		groupWidget.setText(text);
		groupWidget.setBackground(backgroundColor);
		return groupWidget;
	}

	public Table createTable(Composite parent, int style) {
		int flatStyle = BORDER_STYLE == SWT.BORDER ? SWT.NULL : SWT.FLAT;
		return new Table(parent, style | flatStyle);
	}
	public Label createLabel(Composite parent, String text, int style) {
		Label label = new Label(parent, style);
		if (text != null)
			label.setText(text);
		label.setBackground(backgroundColor);
		label.setForeground(foregroundColor);
		return label;
	}

	public Label createSeparator(Composite parent, int style) {
		Label label = new Label(parent, SWT.SEPARATOR | style);
		label.setBackground(backgroundColor);
		label.setForeground(borderColor);
		return label;
	}

	public Text createText(Composite parent, String value) {
		return createText(parent, value, BORDER_STYLE | SWT.SINGLE);
	}

	public StyledText createStyledText(Composite parent, String value) {
		return createStyledText(parent, value, BORDER_STYLE | SWT.SINGLE);
	}

	public StyledText createStyledText(
		Composite parent,
		String value,
		int style) {
		StyledText text = new StyledText(parent, style);
		if (value != null)
			text.setText(value);
		text.setBackground(clientAreaColor);
		text.setForeground(foregroundColor);
		return text;
	}

	public Text createText(Composite parent, String value, int style) {
		Text text = new Text(parent, style);
		if (value != null)
			text.setText(value);
		text.setBackground(clientAreaColor);
		text.setForeground(foregroundColor);
		return text;
	}

	public CCombo createCombo(Composite parent, int style) {
		CCombo combo = new CCombo(parent, SWT.FLAT | style);
		combo.setBackground(clientAreaColor);
		combo.setForeground(foregroundColor);
		return combo;
	}

	public Tree createTree(Composite parent, int style) {
		Tree tree = new Tree(parent, BORDER_STYLE | style);
		tree.setBackground(backgroundColor);
		tree.setForeground(foregroundColor);
		return tree;
	}

	public void dispose() {
		Enumeration colors = colorRegistry.elements();
		while (colors.hasMoreElements()) {
			Color c = (Color) colors.nextElement();
			c.dispose();
		}
		colorRegistry = null;
	}
	public Color getBackgroundColor() {
		return backgroundColor;
	}
	public Color getBorderColor() {
		return borderColor;
	}

	public Color getClientAreaColor() {
		return clientAreaColor;
	}
	public Color getColor(String key) {
		return (Color) colorRegistry.get(key);
	}
	public Color getForegroundColor() {
		return foregroundColor;
	}

	public Color getDisabledColor() {
		return disabledColor;
	}
	
	public Color getDarkGrayColor() {
		return darkGrayColor;
	}

	private void initialize() {
		clientAreaColor = display.getSystemColor(SWT.COLOR_LIST_BACKGROUND);
		registerColor(COLOR_BACKGROUND, 0xff, 0xfe, 0xf9);
		registerColor(COLOR_BORDER, 195, 191, 179);
		registerColor(COLOR_COMPOSITE_SEPARATOR, 152, 170, 203);
		registerColor(DEFAULT_HEADER_COLOR, 0x48, 0x70, 0x98);
		backgroundColor = clientAreaColor;
		borderColor = getColor(COLOR_BORDER);
		foregroundColor = display.getSystemColor(SWT.COLOR_LIST_FOREGROUND);
		disabledColor = display.getSystemColor(SWT.COLOR_GRAY);
		darkGrayColor = display.getSystemColor(SWT.COLOR_DARK_GRAY);
	}

	public void paintBordersFor(Composite parent) {
		if (BORDER_STYLE == SWT.BORDER)
			return;
		if (borderPainter == null)
			borderPainter = new BorderPainter();
		parent.addPaintListener(borderPainter);
	}

	public void paintChildControlsFor(Composite parent) {

		paintBordersFor(parent);

		Control[] children = parent.getChildren();

		if (!parent
			.getBackground()
			.equals(getColor(COLOR_COMPOSITE_SEPARATOR))) {

			if (parent.isEnabled()) {
				parent.setBackground(getBackgroundColor());
			} else {
				parent.setBackground(getDisabledColor());
			}
		}

		for (int i = 0; i < children.length; i++) {
			Control c = children[i];
			if (c instanceof Text
				|| c instanceof CCombo
				|| c instanceof Label
				|| c instanceof Button
				|| c instanceof Tree) {
				c.setBackground(backgroundColor);
				c.setForeground(foregroundColor);
			} else if (c instanceof Composite) {
				paintChildControlsFor((Composite) c);
			}
		}

	}

	public Color registerColor(String key, int r, int g, int b) {
		Color c = new Color(display, r, g, b);
		colorRegistry.put(key, c);
		return c;
	}

}
