/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: JavaSnippet.java,v 1.3 2005/12/30 02:58:11 popescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.hyades.probekit.editor.internal.presentation.codeassist;

import java.net.URL;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Set;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IProject;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.emf.common.notify.Adapter;
import org.eclipse.emf.common.notify.Notifier;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.util.EObjectContainmentEList;
import org.eclipse.hyades.models.internal.probekit.Probekit;
import org.eclipse.hyades.models.internal.probekit.StaticField;
import org.eclipse.hyades.probekit.CompilerFactory;
import org.eclipse.hyades.probekit.IProbeCompiler;
import org.eclipse.hyades.probekit.LocationAdapter;
import org.eclipse.hyades.probekit.ProbekitException;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.IPackageFragment;
import org.eclipse.jdt.core.IPackageFragmentRoot;
import org.eclipse.jdt.core.IProblemRequestor;
import org.eclipse.jdt.core.IType;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jdt.core.WorkingCopyOwner;
import org.eclipse.jdt.core.compiler.IProblem;
import org.osgi.framework.Bundle;
import org.eclipse.core.resources.IMarker;
import org.eclipse.osgi.util.NLS;

/**
 * Instances of this type check for javac problems on a Probekit object,
 * and then associated those IProblem instances with the Probekit EObject
 * (or its child) that the IProblem is reported against.
 */
public final class JavaSnippet {
	private static final int ERROR = 0x1;
	private static final int WARNING = 0x2;
	private static final int NO_PROBLEM = 0;
	
	private IPackageFragment _defaultPkg = null;
	private ICompilationUnit _compilationUnit = null;
	private IFile _probeFile = null;
	private IProbeCompiler _probeCompiler = null;
	private static final String BASE_NAME_SUFFIX = "_checkSyntax.java"; // Don't use _probe.java because if the user has generated the source before, then "Check Syntax" overwrites that source.
	private String _javaFileName = null;
	private Probekit _probekit = null;
	
	private static URL _errorURL = null;
	private static URL _warningURL = null;

	public JavaSnippet(IFile probeFile, Probekit probekit) {
		_probeFile = probeFile;
		_probekit = probekit;
		_javaFileName = ResourceUtil.getBaseName(probeFile) + BASE_NAME_SUFFIX;
	}
	
	private String getJavaFileName() {
		return _javaFileName;
	}
	
	private IProbeCompiler getProbeCompiler() throws ProbekitException {
		_probeCompiler = null;
		_probeCompiler = CompilerFactory.INSTANCE.createCompiler();
		String basename = _probeCompiler.makeValidJavaIdentifier(getJavaFileName());
		_probeCompiler.setClassPrefix(basename);
		_probeCompiler.addResource(getResource(), getProbeFile());
		return _probeCompiler;
	}
	
	public IFile getProbeFile() {
		return _probeFile;
	}
	
	private Resource getResource() {
		return _probekit.eResource();
	}
	
	/**
	 * Compile this snippet with the JDT (javac) compiler, associate
	 * the JDT IProblem with their corresponding EObjects, and return
	 * a Set of the IProblem instances that were found on the probekit.
	 */
	public Set compile() throws CoreException, JavaModelException, ProbekitException {
		clearOldProblems();
		// delete error markers
		try {
			_probeFile.deleteMarkers(IMarker.PROBLEM, false, IResource.DEPTH_ZERO);
		} catch( CoreException e ) {
			// some problem
		}
		
		Set problems = internalCompile();
		
		// Before returning the result, add the SyntaxAdapter and line number to
		// each of the affected EObjects.
		if(problems.size() > 0) {
		    Iterator iterator = problems.iterator();
		    while(iterator.hasNext()) {
		        IProblem problem = (IProblem)iterator.next();
		        addProblem(problem);
				
			    // create error marker
			    IMarker marker = _probeFile.createMarker(IMarker.PROBLEM);
			    marker.setAttribute(IMarker.MESSAGE, 
			    		problem.getMessage());
			    marker.setAttribute(IMarker.SEVERITY, 
			    		problem.isError() ? IMarker.SEVERITY_ERROR : IMarker.SEVERITY_WARNING);
			    marker.setAttribute(IMarker.LOCATION, _probeFile.getName());		   
				
		    }
		     
		}
		
		// Next, check the static fields, which aren't emitted into the .java file
		// and thus aren't checked by the above code.
		compileStaticFields();
		
		return problems;
	}
	
	private void addProblem(IProblem problem) {
        int lineNumber = problem.getSourceLineNumber();
        EObject badObject = getEObject(lineNumber);
        if(badObject != null) {
	        SyntaxAdapter adapter = getSyntaxAdapter(badObject);
	        adapter.addProblem(problem);
        }
	}
	
	private EObject getEObject(int lineNumber) {
		Resource resource = getResource();
		Iterator resourceIterator = resource.getAllContents();
		Notifier result = null;
		LocationAdapter location = null;
		while(resourceIterator.hasNext()) {
			EObject eObject = (EObject)resourceIterator.next();
			LocationAdapter lAdapter = getLocationAdapter(eObject);
			if((lAdapter != null) && (lAdapter.contains(lineNumber))) {
				// Check if this is the best match yet (i.e., don't 
				// flag the Probe if we can flag a child of the probe,
				// e.g. the fragment. The closer we can flag the problem
				// the more help the user gets.
				location = getPreciseMatch(lineNumber, lAdapter, location);
				result = location.getTarget();
			}
		}
		return (EObject)result;
	}
	
	/**
	 * Return the EObject instance that this IProblem should be 
	 * reported against, or return null if the corresponding EObject
	 * cannot be found.
	 * 
	 * The "compile" method must be called on this snippet before this
	 * method is used.
	 */
	public EObject getEObject(IProblem problem) {
		if(problem instanceof StaticFieldUnknownProblem) {
			return getStaticField(problem);
		}
		else {
			int lineNumber = problem.getSourceLineNumber();
			return getEObject(lineNumber);
		}
	}
	
	private StaticField getStaticField(IProblem problem) {
		Resource resource = getResource();
		Iterator resourceIterator = resource.getAllContents();
		while(resourceIterator.hasNext()) {
			Object object = resourceIterator.next();
			if(object instanceof StaticField) {
				StaticField staticField = (StaticField)object;
				SyntaxAdapter adapter = getSyntaxAdapter(staticField);
				IProblem[] problems = adapter.getKnownProblems();
				for(int i=0; i<problems.length; i++) {
					IProblem p = problems[i];
					if(problem.equals(p)) {
						return staticField;
					}
				}
			}
		}
		return null;
	}
	
	private LocationAdapter getPreciseMatch(int lineNumber, LocationAdapter a, LocationAdapter b) {
		if(a == null) {
			return b;
		}
		
		if(b == null) {
			return a;
		}
		
		int aStart = a.getStartLineNumber();
		int bStart = b.getStartLineNumber();
		int aStartDiff = lineNumber - aStart;
		int bStartDiff = lineNumber - bStart;
		if(aStartDiff > bStartDiff) {
			return b;
		}
		else if(aStartDiff < bStartDiff) {
			return a;
		}
		
		int aEnd = a.getEndLineNumber();
		int bEnd = b.getEndLineNumber();
		int aEndDiff = aEnd - lineNumber;
		int bEndDiff = bEnd - lineNumber;
		if(aEndDiff > bEndDiff) {
			return b;
		}
		
		return a;
	}
	
	private void clearOldProblems() {
		Resource resource = getResource();
		Iterator resourceIterator = resource.getAllContents();
		while(resourceIterator.hasNext()) {
			EObject eObject = (EObject)resourceIterator.next();
			SyntaxAdapter adapter = getSyntaxAdapter(eObject);
			adapter.clearProblems();
		}
	}
	
	private void compileStaticFields() throws JavaModelException {
		Resource resource = getResource();
		Iterator resourceIterator = resource.getAllContents();
		while(resourceIterator.hasNext()) {
			Object object = resourceIterator.next();
			if(object instanceof StaticField) {
				StaticField field = (StaticField)object;
				String type = field.getType();
				if(!isKnownType(type)) {
					SyntaxAdapter adapter = getSyntaxAdapter(field);
					StaticFieldUnknownProblem problem = 
						new StaticFieldUnknownProblem(
							 NLS.bind(ProbekitMessages._134, new String[]{type})
						);
					adapter.addProblem(problem);
				}
			}
		}
	}
	
	private boolean isKnownType(String typeName) throws JavaModelException {
		IProject project = getProbeFile().getProject();
		IJavaProject jp = JavaCore.create(project);
		IType type = jp.findType(typeName);
		return (type != null);
	}
	
	public static SyntaxAdapter getSyntaxAdapter(EObject object) {
		SyntaxAdapter adapter = null;
	
		Collection c = object.eAdapters();
		if (c != null) {
			Iterator iterator = c.iterator();
			while (iterator.hasNext()) {
				Adapter a = (Adapter)iterator.next();
				if ((a != null) && (a instanceof SyntaxAdapter)) {
					adapter = (SyntaxAdapter)a;
					break;
				}
			}
		}
		
		if(adapter == null) {
			adapter = new SyntaxAdapter();
			adapter.setTarget(object);
			object.eAdapters().add(adapter);
		}
		
		return adapter;
	}
	
	public static URL getImageURL(Object object) {
		int problemState = JavaSnippet.getProblemState(object);
		URL url = null;
		if((problemState & JavaSnippet.ERROR) == JavaSnippet.ERROR) {
			url = getErrorURL();
		}
		else if((problemState & JavaSnippet.WARNING) == JavaSnippet.WARNING) {
			url = getWarningURL();
		}
		return url;
	}
	
	public static URL getIconURL(IProblem problem) {
		if(problem.isError()) {
			return getErrorURL();
		}
		else if(problem.isWarning()) {
			return getWarningURL();
		}
		return null;
	}
	
	private static URL getErrorURL() {
		if(_errorURL == null) {
			Bundle plugin = Platform.getBundle("org.eclipse.ui");
			if(plugin != null) {
				_errorURL = plugin.getEntry("icons/full/obj16/error_tsk.gif");
			}
		}
		return _errorURL;
	}
	
	private static URL getWarningURL() {
		if(_warningURL == null) {
			Bundle plugin = Platform.getBundle("org.eclipse.ui");
			if(plugin != null) {
				_warningURL = plugin.getEntry("icons/full/obj16/warn_tsk.gif");
			}
		}
		return _warningURL;
	}
	
	private static int getProblemState(Object object) {
		if(object instanceof EObject) {
			EObject eObject = (EObject)object;
			return internalGetProblemState(eObject);
		}
		else if(object instanceof EObjectContainmentEList) {
			EObjectContainmentEList list = (EObjectContainmentEList)object;
			Iterator iterator = list.iterator();
			int result = NO_PROBLEM;
			while(iterator.hasNext()) {
				EObject eObject = (EObject)iterator.next();
				result = (result | internalGetProblemState(eObject));
			}
			return result;
		}
		return NO_PROBLEM;
	}
	
	private static int internalGetProblemState(EObject eObject) {
		SyntaxAdapter adapter = getSyntaxAdapter(eObject);
		int result = NO_PROBLEM;
		if(adapter.containsErrors()) {
			result = result | ERROR;
		}
		else if(adapter.containsWarnings()) {
			result = result | WARNING;
		}
		return result;
	}
	
	// Will return null if a location adapter doesn't exist.
	public static LocationAdapter getLocationAdapter(EObject object) {
		LocationAdapter adapter = null;
	
		Collection c = object.eAdapters();
		if (c != null) {
			Iterator iterator = c.iterator();
			while (iterator.hasNext()) {
				Adapter a = (Adapter)iterator.next();
				if ((a != null) && (a instanceof LocationAdapter)) {
					adapter = (LocationAdapter)a;
					break;
				}
			}
		}
	
		return adapter;
	}

	private Set internalCompile() throws CoreException, JavaModelException, ProbekitException {
		String javaCode = generateSource();
		// create requestor for accumulating discovered problems
		JavaProblemRequestor problemRequestor = new JavaProblemRequestor();
		    
		// use working copy to hold source with error
		ICompilationUnit workingCopy = getCompilationUnit().getWorkingCopy(new WorkingCopyOwner() {}, problemRequestor, null);
		workingCopy.getBuffer().setContents(javaCode);
		
		// trigger reconciliation			
		workingCopy.reconcile(ICompilationUnit.NO_AST, true, null, null);
		workingCopy.discardWorkingCopy();
		
		return problemRequestor.getProblems();
	}
	
	private ICompilationUnit getCompilationUnit() throws CoreException, JavaModelException {
		if(_compilationUnit == null) {
			IPackageFragment defaultPackage = getDefaultPackage();
			if(defaultPackage == null) {
				// internal error...is the probe file in a source folder?
				String error = ProbekitMessages._138;
				IStatus status = ResourceUtil.createInitialStatus(IStatus.ERROR, error);
				throw new CoreException(status);
			}
			else {
				// create the CompilationUnit handle in memory only; 
				// no need to create a resource and update the workspace
				_compilationUnit = defaultPackage.getCompilationUnit(getJavaFileName()); 
			}
		}
		return _compilationUnit;
	}
	
	/**
	 * Release any resources created by this snippet.
	 */
	public void release() throws JavaModelException {
		// if the CompilationUnit is actually present in the workspace, 
		// delete it
		if(_compilationUnit != null && _compilationUnit.exists()) {
		    _compilationUnit.delete(true, new NullProgressMonitor());
		}
	}
	
	private IPackageFragment getDefaultPackage() throws JavaModelException {
	    if(_defaultPkg == null) {
		    IJavaElement pkgRoot = JavaCore.create(getProbeFile().getParent());
		    if(pkgRoot instanceof IPackageFragmentRoot) {
			    _defaultPkg = ((IPackageFragmentRoot)pkgRoot).getPackageFragment("");
		    }
		    else if(pkgRoot instanceof IJavaProject) {
		    	IJavaProject jp = (IJavaProject)pkgRoot;
				IPath path = jp.getPath();
				IPath defaultPath = path.append(""); // default package 'name' is the empty string
		    	_defaultPkg = jp.findPackageFragment(defaultPath);
		    }
	    }
	    return _defaultPkg;
	}
	
	private String generateSource() throws ProbekitException {
		return getProbeCompiler().getGeneratedSource();
	}
	
	private static class JavaProblemRequestor implements IProblemRequestor {
		public void acceptProblem(IProblem problem) {
			getProblems().add(problem);
		}
		
		public void beginReporting() {}
		
		public void endReporting() {}
		
		public boolean isActive() {	
		    // will detect problems if active
		    return true; 
		}
		
		public Set getProblems() {
		    if(_problems == null) {
		        _problems = new HashSet();
		    }
		    return _problems;
		}
		
		private Set _problems = null;
	}
	
	private static class StaticFieldUnknownProblem implements IProblem {
		private String _message;
		
		private StaticFieldUnknownProblem(String translatedMessage) {
			_message = translatedMessage;
		}
		
		public String[] getArguments() {
			return new String[0];
		}

		public int getID() {
			return IProblem.UndefinedType;
		}
		
		public String getMessage() {
			return _message;
		}
		
		public char[] getOriginatingFileName() {
			return new char[0];
		}
		
		public int getSourceEnd() {
			return 0;
		}
		
		public int getSourceLineNumber() {
			return 0;
		}
		
		public int getSourceStart() {
			return 0;
		}
		
		public boolean isError() {
			return true;
		}
		
		public boolean isWarning() {
			return false;
		}
		
		public void setSourceEnd(int sourceEnd) {
		}
		
		public void setSourceLineNumber(int lineNumber) {
		}
		
		public void setSourceStart(int sourceStart) {
		}
	}
}
