/**********************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NewProbekitDataPage.java,v 1.4 2005/12/30 02:58:10 popescu Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.hyades.probekit.editor.internal.ui.newFile;

import org.eclipse.hyades.models.internal.probekit.DataItem;
import org.eclipse.hyades.models.internal.probekit.DataType;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.ProbeFileModel;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitUtil;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.hyades.probekit.editor.internal.ui.DataDialog;
import org.eclipse.hyades.probekit.ui.internal.ContextIds;
import org.eclipse.jface.viewers.CellEditor;
import org.eclipse.jface.viewers.ColumnWeightData;
import org.eclipse.jface.viewers.ComboBoxCellEditor;
import org.eclipse.jface.viewers.ICellModifier;
import org.eclipse.jface.viewers.IStructuredContentProvider;
import org.eclipse.jface.viewers.ITableLabelProvider;
import org.eclipse.jface.viewers.LabelProvider;
import org.eclipse.jface.viewers.TableLayout;
import org.eclipse.jface.viewers.TableViewer;
import org.eclipse.jface.viewers.TextCellEditor;
import org.eclipse.jface.viewers.Viewer;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Image;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Table;
import org.eclipse.swt.widgets.TableColumn;
import org.eclipse.swt.widgets.TableItem;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;



public final class NewProbekitDataPage extends WizardPage {
	public static final String PAGE_ID = "New Probekit File (Data)"; //$NON-NLS-1$
	
	ProbeFileModel _model = null;
	Button _addButton = null;
	Button _editButton = null;
	Button _removeButton = null;
	static final String[] COLUMNS = new String[] { ProbekitMessages._40,
			ProbekitMessages._41 };
	TableViewer _tableViewer;
	private Label _explanationTitle;
	private Text _explanationText;
	ComboBoxCellEditor _comboBoxCellEditor;
	TextCellEditor _textCellEditor;
	private Label _generalDataExplanation;
	
	public NewProbekitDataPage(ProbeFileModel model) {
		super(PAGE_ID);
		_model = model;
		
		setTitle(ProbekitMessages._38);
		setMessage(ProbekitMessages._39);
	}

	ProbeFileModel getModel() {
		return _model;
	}
	
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if(visible) {
			refresh();
		}
	}
	
	public void createControl(Composite parent) {
		Composite page = new Composite(parent, SWT.NONE);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		page.setLayout(pageLayout);
		GridData pageData = new GridData(GridData.FILL_BOTH);
		page.setLayoutData(pageData);

		createTableDescription(page);
		createTable(page);
		createExplanationText(page);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(page, ContextIds.PROBEKIT_NEWWIZ_DATA);
		
		setControl(page);
	}
	
	private void createTableDescription(Composite page) {
		_generalDataExplanation = new Label(page, SWT.WRAP);
		GridData generalData = new GridData(GridData.FILL_BOTH);
		_generalDataExplanation.setLayoutData(generalData);
	}
	
	private void createTable(Composite parent) {
		Composite page = new Composite(parent, SWT.NONE);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		pageLayout.numColumns = 2;
		page.setLayout(pageLayout);
		GridData pageData = new GridData(GridData.FILL_BOTH);
		page.setLayoutData(pageData);

		createTableViewer(page);
		createTableButtons(page);
	}
	
	private void createTableViewer(Composite page) {
		_tableViewer = new TableViewer(page, SWT.BORDER | SWT.V_SCROLL | SWT.SINGLE | SWT.FULL_SELECTION );
		TableLayout tableLayout = new TableLayout();
		Table table = _tableViewer.getTable();
		
		CellEditor[] cellEditors = new CellEditor[COLUMNS.length];
		
		TableColumn comboCell = new TableColumn(table, SWT.LEFT);
		comboCell.setText(COLUMNS[0]);
		tableLayout.addColumnData(new ColumnWeightData(50));
		String[] comboboxItems = getModel().getProbeMetaData().getDataTypeNames(null);
		_comboBoxCellEditor = 
			new ComboBoxCellEditor(
					table, 
					comboboxItems
			);
		cellEditors[0] = _comboBoxCellEditor;
		
		TableColumn textCell = new TableColumn(table, SWT.LEFT);
		textCell.setText(COLUMNS[1]);
		tableLayout.addColumnData(new ColumnWeightData(50));
		_textCellEditor = new TextCellEditor(table);
		cellEditors[1] = _textCellEditor;

		_tableViewer.setCellEditors(cellEditors);
		_tableViewer.setCellModifier(new DataItemCellModifier());
		_tableViewer.setContentProvider(new DataItemContentProvider());
		_tableViewer.setLabelProvider(new DataItemColumnLabelProvider());
		_tableViewer.setInput(getModel().getProbeMetaData().getDataItems());
		_tableViewer.setColumnProperties(COLUMNS);
		
		GridData tableData = new GridData(GridData.FILL_BOTH);
		table.setLayoutData(tableData);
		table.setLayout(tableLayout);
		table.setHeaderVisible(true);
		table.setLinesVisible(true);
		table.setFont(page.getFont());
		table.addSelectionListener(new PageWidgetListener());
	}
	
	private void createTableButtons(Composite page) {
		PageWidgetListener listener = new PageWidgetListener();
		
		Composite buttonGroup = new Composite(page, SWT.NONE);
		buttonGroup.setFont(page.getFont());
		GridLayout buttonLayout = new GridLayout();
		buttonLayout.makeColumnsEqualWidth = true;
		buttonGroup.setLayout(buttonLayout);
		GridData buttonData = new GridData();
		buttonGroup.setLayoutData(buttonData);
		
		_addButton = new Button(buttonGroup, SWT.PUSH);
		_addButton.setText(ProbekitMessages._125);
		GridData addData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_addButton.setLayoutData(addData);
		_addButton.addSelectionListener(listener);
		
		_editButton = new Button(buttonGroup, SWT.PUSH);
		_editButton.setText(ProbekitMessages._118);
		GridData editData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_editButton.setLayoutData(editData);
		_editButton.addSelectionListener(listener);
		
		_removeButton = new Button(buttonGroup, SWT.PUSH);
		_removeButton.setText(ProbekitMessages._119);
		GridData removeData = new GridData(GridData.FILL_HORIZONTAL | GridData.VERTICAL_ALIGN_BEGINNING);
		_removeButton.setLayoutData(removeData);
		_removeButton.addSelectionListener(listener);
		
		refreshButtons();
	}
	
	private void createExplanationText(Composite page) {
		_explanationTitle = new Label(page, SWT.BOLD | SWT.WRAP);
		GridData explanationTitleData = new GridData(GridData.FILL_HORIZONTAL);
		_explanationTitle.setLayoutData(explanationTitleData);
		_explanationTitle.setFont(page.getFont());
		_explanationTitle.setBackground(page.getBackground());
		
		_explanationText = new Text(page, SWT.READ_ONLY | SWT.MULTI | SWT.WRAP | SWT.V_SCROLL | SWT.H_SCROLL);
		GridData explanationData = new GridData(GridData.FILL_BOTH);
		_explanationText.setLayoutData(explanationData);
		_explanationText.setFont(page.getFont());
		_explanationText.setBackground(page.getBackground());
	}
	
	void refresh() {
		_generalDataExplanation.setText(getModel().getDataExplanation());
		refreshTable();
		refreshExplanation();
		refreshButtons();
	}
	
	private void refreshExplanation() {
		DataItem item = getSelectedItem();
		_explanationTitle.setText(ProbekitUtil.getDataTypeName(item));
		_explanationText.setText(getModel().getProbeMetaData().getDataTypeExplanation(item));
	}
	
	void refreshTable() {
		_tableViewer.refresh();
	}
	
	private void refreshButtons() {
		DataType[] availableTypes = getModel().getProbeMetaData().getAvailableDataTypes();
		_addButton.setEnabled(availableTypes.length > 0);
		DataItem dataItem = getSelectedItem();
		boolean enabled = (dataItem != null);
		_editButton.setEnabled(enabled);
		_removeButton.setEnabled(enabled);
	}
	
	DataItem getSelectedItem() {
		int idx = _tableViewer.getTable().getSelectionIndex();
		if (idx == -1) {
			return null;
		}
		else {
			return (DataItem) _tableViewer.getElementAt(idx);
		}
	}
	
	private DataItem[] getDataItems() {
		TableItem[] tableItems = _tableViewer.getTable().getItems();
		DataItem[] dataItems = new DataItem[tableItems.length];
		for(int i=0; i<tableItems.length; i++) {
			TableItem tableItem = tableItems[i];
			dataItems[i] = (DataItem)tableItem.getData();
		}
		return dataItems;
	}

	void checkPageComplete() {
		getModel().getProbeMetaData().setDataItems(getDataItems());
		
		String errorMessage = null;
		if(!getModel().isValidDataItems()) {
			errorMessage = ProbekitMessages._42;
		}
		setErrorMessage(errorMessage);
		boolean isComplete = (errorMessage == null);
		setPageComplete(isComplete);
	}
	
	private class DataItemColumnLabelProvider extends LabelProvider implements ITableLabelProvider {
		public Image getColumnImage(Object element, int columnIndex) {
			return null;
		}
		
		public String getColumnText(Object element, int columnIndex) {
			if(element instanceof DataItem) {
				DataItem item = (DataItem)element;
				if(columnIndex == 0) {
					return ProbekitUtil.getDataTypeName(item);
				}
				else {
					return ProbekitUtil.getDataItemName(item);
				}
			}
			return ResourceUtil.NO_TEXT;
		}
	}

	private class DataItemContentProvider implements IStructuredContentProvider {
		public void dispose() {
		}

		public void inputChanged(Viewer viewer, Object oldInput, Object newInput) {
		}

		public Object[] getElements(Object inputElement) {
			return getModel().getProbeMetaData().getDataItems();
		}
	}
	
	private class DataItemCellModifier implements ICellModifier {
		public Object getValue(Object element, String property) {
			Object value = null;
			if (element instanceof DataItem) {
				DataItem item = (DataItem)element;
				if (property.equals(COLUMNS[0])) {
					refreshComboBeforeEdit();
					String typeName = ProbekitUtil.getDataTypeName(item);
					value = getCellIndex(typeName);
				}
				else if (property.equals(COLUMNS[1])) {
					value = ProbekitUtil.getDataItemName(item);
				}
			}
			return value;
		}
		
		private Integer getCellIndex(String dataTypeName) {
			Integer index = null;
			String[] items = _comboBoxCellEditor.getItems();
			for(int i=0; i<items.length; i++) {
				String itemText = items[i];
				if(itemText.equals(dataTypeName)) {
					index = new Integer(i);
					break;
				}
			}
			
			if(index == null) {
				index = new Integer(0);
			}
			
			return index;
		}

		public boolean canModify(Object element, String property) {
			return true;
		}

		public void modify(Object element, String property, Object value) {
			TableItem item = (TableItem) element;
			DataItem dataItem = (DataItem) item.getData();
			if(property.equals(COLUMNS[0])) {
				Integer cellIndex = (Integer)value;
				String[] items = _comboBoxCellEditor.getItems();
				String typeName = items[cellIndex.intValue()];
				DataType type = DataType.get(typeName);
				dataItem.setType(type);
			}
			else if(property.equals(COLUMNS[1])) {
				String itemName = (String)value;
				dataItem.setName(itemName);
			}
			refresh();
			checkPageComplete();
		}

		private void refreshComboBeforeEdit() {
			_comboBoxCellEditor.setItems(
				getModel().getProbeMetaData().getDataTypeNames(getSelectedItem())
			);
		}
	
	};
	
	private class PageWidgetListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing to do when the default is selected.
		}

		public void widgetSelected(SelectionEvent event) {
			if(event.widget.equals(_addButton)) {
				add();
			}
			else if (event.widget.equals(_editButton)) {
				edit();
			}
			else if(event.widget.equals(_removeButton)) {
				remove();
			}
			
			refresh();
			checkPageComplete();
		}
		
		private void add() {
			getModel().createDataItem(); 
			
			// Refresh before selecting so that the new item is added to the table,
			// and refresh after selecting so that the "Edit" and "Remove" buttons
			// are enabled.
			refreshTable();		
			select(_tableViewer.getTable().getItemCount()-1);
		}
		
		private void edit() {
			DataItem item = getSelectedItem();
			if(item != null) {
				// Can't select a single cell in a row.
				// Launch an editor dialog instead.
				DataDialog dialog = new DataDialog(
							getContainer().getShell(),
							getModel().getProbeMetaData().getDataTypeNames(item),
							item);
				dialog.setBlockOnOpen(true);
				dialog.setTitle(ProbekitMessages._44);
				dialog.open();
				
				int index = _tableViewer.getTable().getSelectionIndex();
				DataItem updatedItem = dialog.getResult();
				getModel().getProbeMetaData().setDataItem(index, updatedItem);
				
				refreshTable();
				select(index);
			}
		}
		
		private void remove() {
			DataItem item = getSelectedItem();
			int index = _tableViewer.getTable().getSelectionIndex();
			if(item != null) {
				getModel().getProbeMetaData().removeDataItem(item);
				index -= 1;
			}
			
			refreshTable();
			select(index);
		}

		private void select(int index) {
			_tableViewer.getTable().select(index);
		}
	}
}
