/**********************************************************************
 * Copyright (c) 2005, 2006 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * $Id: NewProbekitProbePage.java,v 1.7 2006/01/27 17:35:20 ewchan Exp $
 *
 * Contributors:
 * IBM - Initial API and implementation
 **********************************************************************/


package org.eclipse.hyades.probekit.editor.internal.ui.newFile;

import java.util.List;

import org.eclipse.core.resources.IContainer;
import org.eclipse.hyades.models.internal.probekit.DataItem;
import org.eclipse.hyades.models.internal.probekit.FragmentType;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.CallsiteProbeMetaData;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.IProbeMetaData;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.MethodProbeMetaData;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.NoProbeMetaData;
import org.eclipse.hyades.probekit.editor.internal.core.newFile.ProbeFileModel;
import org.eclipse.hyades.probekit.editor.internal.core.util.JavaUtil;
import org.eclipse.hyades.probekit.editor.internal.core.util.ProbekitMessages;
import org.eclipse.hyades.probekit.editor.internal.core.util.ResourceUtil;
import org.eclipse.hyades.probekit.editor.internal.ui.DataItemLabelProvider;
import org.eclipse.hyades.probekit.editor.internal.ui.DataItemTreeProvider;
import org.eclipse.hyades.probekit.editor.internal.ui.DialogUtil;
import org.eclipse.hyades.probekit.editor.internal.ui.SelectOneIncludedValidator;
import org.eclipse.hyades.probekit.ui.internal.ContextIds;
import org.eclipse.hyades.probekit.ui.internal.ProbekitUIPlugin;
import org.eclipse.jdt.core.IJavaProject;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.jface.viewers.ViewerSorter;
import org.eclipse.jface.window.Window;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.custom.CCombo;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.events.SelectionListener;
import org.eclipse.swt.graphics.Font;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.swt.widgets.Group;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.dialogs.CheckedTreeSelectionDialog;



public final class NewProbekitProbePage extends WizardPage {
	public static final String PAGE_ID = "New Probekit File (Probe)"; //$NON-NLS-1$
	private ProbeFileModel _model = null;
	private Button _methodProbeButton = null;
	private Button _callsiteProbeButton = null;
	private Button _noProbeButton = null;
	private Group _fragmentTypes = null;
	private Text _explanationText = null;
	private Text _fragmentExplanationText = null;
	private Text _fileNameField = null;
	Text _sourceContainerField = null;
	private CCombo _encodingField = null;	
	
	public NewProbekitProbePage(ProbeFileModel model) {
		super(PAGE_ID);
		_model = model;
		
		setTitle(ProbekitMessages._35);
		setMessage(ProbekitMessages._36);
	}
	
	public void setVisible(boolean visible) {
		super.setVisible(visible);
		if(visible) {
			refreshProbeExplanation();
			refreshFragmentTypeExplanation();
		}
	}
	
	public void createControl(Composite parent) {
		Composite page = new Composite(parent, SWT.NONE);
		page.setFont(parent.getFont());
		GridLayout pageLayout = new GridLayout();
		page.setLayout(pageLayout);
		
		createFile(page);
		createFolder(page);
		createXMLEncoding(page);

		Label line = new Label(page, SWT.SEPARATOR | SWT.HORIZONTAL);
		GridData lineData = new GridData(GridData.FILL_HORIZONTAL);
		line.setLayoutData(lineData);

		createProbeSection(page);

		Label line2 = new Label(page, SWT.SEPARATOR | SWT.HORIZONTAL);
		GridData line2Data = new GridData(GridData.FILL_HORIZONTAL);
		line2.setLayoutData(line2Data);
		
		createFragmentSection(page);

		setControl(page);
		
		PlatformUI.getWorkbench().getHelpSystem().setHelp(page, ContextIds.PROBEKIT_NEWWIZ_PROBE);
		
		checkPageComplete();
	}
	
	private void createFile(Composite page) {
		Font font = page.getFont();
		FileOrContainerModifiedListener listener = new FileOrContainerModifiedListener();

		Composite group = new Composite(page, SWT.NONE);
		GridData groupData = new GridData(GridData.FILL_BOTH);
		group.setLayoutData(groupData);
		GridLayout groupLayout = new GridLayout();
		groupLayout.numColumns = 2;
		groupLayout.makeColumnsEqualWidth = false;
		group.setLayout(groupLayout);
		
		/*Text fileNameLabel = new Text(group, SWT.READ_ONLY);
		GridData fileNameLabelData = new GridData();
		fileNameLabel.setLayoutData(fileNameLabelData);
		fileNameLabel.setFont(font);
		fileNameLabel.setText(ProbekitMessages._45);*/
		Label fileNameLabel = new Label(group, SWT.LEFT); {
			fileNameLabel.setText(ProbekitMessages._45);
			GridData data = new GridData();
			data.horizontalAlignment = GridData.FILL;
			fileNameLabel.setLayoutData(data);
		}
		
		_fileNameField = new Text(group, SWT.BORDER);
		GridData fileNameFieldData = new GridData(GridData.FILL_HORIZONTAL);
		_fileNameField.setFont(font);
		_fileNameField.setLayoutData(fileNameFieldData);
		_fileNameField.setText(_model.getFileName());

		_fileNameField.addModifyListener(listener);
	}
	
	private void createFolder(Composite page) {
		Font font = page.getFont();
		FileOrContainerModifiedListener listener = new FileOrContainerModifiedListener();

		Composite group = new Composite(page, SWT.NONE);
		GridData groupData = new GridData(GridData.FILL_BOTH);
		group.setLayoutData(groupData);
		GridLayout groupLayout = new GridLayout();
		groupLayout.numColumns = 3;
		groupLayout.makeColumnsEqualWidth = false;
		group.setLayout(groupLayout);

		/*Text sourceFolderLabel = new Text(group, SWT.READ_ONLY);
		GridData sourceFolderLabelData = new GridData();
		sourceFolderLabel.setLayoutData(sourceFolderLabelData);
		sourceFolderLabel.setFont(font);
		sourceFolderLabel.setText(ProbekitMessages._46);*/
		Label sourceFolderLabel = new Label(group, SWT.LEFT); {
			sourceFolderLabel.setText(ProbekitMessages._46);
			GridData data = new GridData();
			data.horizontalAlignment = GridData.FILL;
			sourceFolderLabel.setLayoutData(data);
		}
		
		_sourceContainerField = new Text(group, SWT.BORDER);
		GridData sourceFolderFieldData = new GridData(GridData.FILL_HORIZONTAL);
		sourceFolderFieldData.grabExcessHorizontalSpace = true;
		_sourceContainerField.setFont(font);
		_sourceContainerField.setLayoutData(sourceFolderFieldData);
		_sourceContainerField.setText(_model.getSourceContainerPath());
		
		Button button = new Button(group, SWT.PUSH);
		GridData buttonData = new GridData();
		button.setLayoutData(buttonData);
		button.setFont(font);
		button.setText(ProbekitMessages._47);

		_sourceContainerField.addModifyListener(listener);
		button.addSelectionListener(new BrowseListener());
	}
	
	private void createXMLEncoding(Composite page) {
		
		Composite encoding = new Composite(page,SWT.NULL);
		GridLayout layout = new GridLayout();
		layout.numColumns = 2;
		encoding.setLayout(layout);
		GridData gd= new GridData();
		gd.horizontalAlignment= GridData.FILL;
		gd.grabExcessHorizontalSpace= true;
		encoding.setLayoutData(gd);			
		
		Label encodingLabel = new Label(encoding, SWT.LEFT); {
			encodingLabel.setText(ProbekitMessages._8);
	
			GridData data = new GridData();
			data.horizontalAlignment = GridData.FILL;
			encodingLabel.setLayoutData(data);
		}
		_encodingField = new CCombo(encoding, SWT.BORDER); {
			GridData data = new GridData();
			data.horizontalAlignment = GridData.FILL;
			data.grabExcessHorizontalSpace = true;
			_encodingField.setLayoutData(data);
		}
	
		String[] xmlEncodingChoices = _model.getXMLEncodingChoices();
		int selectionIndex = 0;
		for (int i = 0; i < xmlEncodingChoices.length; i++) {
			_encodingField.add(xmlEncodingChoices[i]);
			
			if (_model.getDefaultEncoding().equals(xmlEncodingChoices[i]))
				selectionIndex = i;
		}
		
		_encodingField.select(selectionIndex);
		
		_encodingField.addSelectionListener(new XMLEncodingSelectionListener());
	}	
	
	private void createProbeSection(Composite page) {
		Composite buttonComposite = new Composite(page, SWT.NONE);
		GridData buttonCompositeData = new GridData(GridData.FILL_BOTH);
		buttonComposite.setLayoutData(buttonCompositeData);
		GridLayout buttonCompositeLayout = new GridLayout();
		buttonCompositeLayout.numColumns = 2;
		buttonComposite.setLayout(buttonCompositeLayout);
		buttonComposite.setFont(page.getFont());
		
		createProbeButtons(buttonComposite);
		createProbeExplanation(buttonComposite);
	}
	
	private void createProbeButtons(Composite buttonComposite) {	
		Font font = buttonComposite.getFont();
		ProbeButtonSelectionListener listener = new ProbeButtonSelectionListener();
		
		Group buttonGroup = new Group(buttonComposite, SWT.SHADOW_NONE | SWT.NONE);
		GridData buttonData = new GridData();
		buttonGroup.setLayoutData(buttonData);
		buttonGroup.setFont(font);
		buttonGroup.setText(ProbekitMessages._48);
		GridLayout buttonLayout = new GridLayout();
		buttonGroup.setLayout(buttonLayout);
		
		_methodProbeButton = new Button(buttonGroup, SWT.RADIO);
		_methodProbeButton.setText(ProbekitMessages._49);
		GridData mData = new GridData();
		_methodProbeButton.setLayoutData(mData);
		_methodProbeButton.setSelection(_model.getProbeMetaData().getProbeType() == IProbeMetaData.METHOD_PROBE);
		_methodProbeButton.setFont(font);
		
		_callsiteProbeButton = new Button(buttonGroup, SWT.RADIO);
		_callsiteProbeButton.setText(ProbekitMessages._50);
		GridData cData = new GridData();
		_callsiteProbeButton.setLayoutData(cData);
		_callsiteProbeButton.setSelection(_model.getProbeMetaData().getProbeType() == IProbeMetaData.CALLSITE_PROBE);
		_callsiteProbeButton.setFont(font);
		
		_noProbeButton = new Button(buttonGroup, SWT.RADIO);
		_noProbeButton.setText(ProbekitMessages._51);
		GridData bData = new GridData();
		_noProbeButton.setLayoutData(bData);
		_noProbeButton.setSelection(_model.getProbeMetaData().getProbeType() == IProbeMetaData.NO_PROBE);
		_noProbeButton.setFont(font);

		_methodProbeButton.addSelectionListener(listener);
		_callsiteProbeButton.addSelectionListener(listener);
		_noProbeButton.addSelectionListener(listener);
	}

	private void createProbeExplanation(Composite buttonComposite) {
		_explanationText = new Text(buttonComposite, SWT.READ_ONLY | SWT.MULTI | SWT.WRAP | SWT.V_SCROLL | SWT.H_SCROLL);
		GridData explanationData = new GridData(GridData.FILL_BOTH);
		_explanationText.setLayoutData(explanationData);
		_explanationText.setFont(buttonComposite.getFont());
		_explanationText.setBackground(buttonComposite.getBackground());
	}
	
	private void createFragmentSection(Composite page) {
		Composite buttonComposite = new Composite(page, SWT.NONE);
		GridData buttonCompositeData = new GridData(GridData.FILL_BOTH);
		buttonComposite.setLayoutData(buttonCompositeData);
		GridLayout buttonCompositeLayout = new GridLayout();
		buttonCompositeLayout.numColumns = 2;
		buttonComposite.setLayout(buttonCompositeLayout);
		buttonComposite.setBackground(page.getBackground());
		
		_fragmentTypes = new Group(buttonComposite, SWT.NONE);
		GridData fragmentData = new GridData();
		_fragmentTypes.setLayoutData(fragmentData);
		_fragmentTypes.setText(ProbekitMessages._52);
		GridLayout fragmentLayout = new GridLayout();
		_fragmentTypes.setLayout(fragmentLayout);
		
		createFragmentExplanation(buttonComposite);
	}
	
	private void createFragmentExplanation(Composite buttonComposite) {
		_fragmentExplanationText = new Text(buttonComposite, SWT.READ_ONLY | SWT.MULTI | SWT.WRAP | SWT.V_SCROLL | SWT.H_SCROLL);
		GridData explanationData = new GridData(GridData.FILL_BOTH);
		_fragmentExplanationText.setLayoutData(explanationData);
		_fragmentExplanationText.setBackground(buttonComposite.getBackground());

		refreshFragmentTypes();
	}
	
	private DiscardDataReadOnlyDialog createDialog(IProbeMetaData probe, FragmentType type) {
		DataItem[] items = _model.getInvalidDataItems(probe, type);
		DiscardDataReadOnlyDialog dialog = 
			new DiscardDataReadOnlyDialog(
					getContainer().getShell(),
					new DataItemLabelProvider(),
					new DataItemTreeProvider()
			);
		dialog.setInput(items);
		dialog.setSorter(new ViewerSorter());
		dialog.setBlockOnOpen(true);
		dialog.setTitle(ProbekitMessages._56);
		dialog.setMessage(ProbekitMessages._57);
		dialog.setInitialSelections(items);
		return dialog;
	}
	
	void updateProbe(SelectionEvent e) {
		Object source = e.getSource();
		int probeType = _model.getProbeMetaData().getProbeType();
		IProbeMetaData oldProbe = _model.getProbeMetaData();
		IProbeMetaData newProbe = null;
		if((source.equals(_methodProbeButton)) && (probeType != IProbeMetaData.METHOD_PROBE)) {
			newProbe = new MethodProbeMetaData(oldProbe);
		}
		else if((source.equals( _callsiteProbeButton)) && (probeType != IProbeMetaData.CALLSITE_PROBE)) {
			newProbe = new CallsiteProbeMetaData(oldProbe);
		}
		else if((source.equals(_noProbeButton)) && (probeType != IProbeMetaData.NO_PROBE)) {
			newProbe = new NoProbeMetaData(oldProbe);
		}
		
		if(newProbe != null) {
			boolean doChange = confirmChange(newProbe, newProbe.getFragmentType());
			if(doChange) {
				oldProbe.release();
				newProbe.forceValid();
				_model.setProbeMetaData(newProbe);
		
				refreshProbeExplanation();
				refreshFragmentTypes();
				refreshFragmentTypeExplanation();
			} else {
				// Put the selection back the way that it was
				_methodProbeButton.setSelection(probeType == IProbeMetaData.METHOD_PROBE);
				_callsiteProbeButton.setSelection(probeType == IProbeMetaData.CALLSITE_PROBE);
				_noProbeButton.setSelection(probeType == IProbeMetaData.NO_PROBE);
			}
		}
		
		checkPageComplete();
	}
	
	void updateFragmentType(SelectionEvent e) {
		FragmentType[] types = _model.getProbeMetaData().getAvailableFragmentTypes();
		Button selectionEventButton = (Button)e.getSource();
		if(!selectionEventButton.getSelection()) {
			// Only want to process a fragment type if it's selected.
			return;
		}
		
		Control[] allChildren = _fragmentTypes.getChildren();
		String buttonText = null;
		for(int j=0; j<allChildren.length; j++) {
			Control control = allChildren[j];
			Button button = (Button)control;
			if(button.getSelection()) {
				buttonText = button.getText();
			}
		}
		
		FragmentType newType = null;
		if(buttonText == null) {
			if(types.length > 0) {
				newType = types[0];
			}
		}
		else {
			for(int i=0; i<types.length; i++) {
				FragmentType type = types[i];
				if(buttonText.equals(type.getName())) {
					newType = type;
					break;
				}
			}
		}
		
			
		boolean doChange = confirmChange(_model.getProbeMetaData(), newType);
		if(doChange) {
			_model.getProbeMetaData().setFragmentType(newType);
			_model.getProbeMetaData().forceValid();
			refreshFragmentTypeExplanation();
		}
		else {
			for(int j=0; j<allChildren.length; j++) {
				Control control = allChildren[j];
				Button button = (Button)control;
				button.setSelection(button.getText().equals(_model.getProbeMetaData().getFragmentType().getName()));
			}
		}
		
		checkPageComplete();
	}
	
	private void refreshProbeExplanation() {
		_explanationText.setText(_model.getProbeMetaData().getProbeTypeExplanation());
	}
	
	private void refreshFragmentTypeExplanation() {
		_fragmentExplanationText.setText(_model.getProbeMetaData().getFragmentTypeExplanation());
	}
	
	private void refreshFragmentTypes() {
		FragmentType[] types = _model.getProbeMetaData().getAvailableFragmentTypes();
		Control[] allChildren = _fragmentTypes.getChildren();
		if(types.length > allChildren.length) {
			FragmentButtonSelectionListener listener = new FragmentButtonSelectionListener();
			int missing = types.length - allChildren.length;
			for(int i=0; i<missing; i++) {
				Button button = new Button(_fragmentTypes, SWT.RADIO);
				button.setFont(_fragmentTypes.getFont());
				button.addSelectionListener(listener);
			}
			allChildren = _fragmentTypes.getChildren();
		}
		
		for(int j=0; j<allChildren.length; j++) {
			Control control = allChildren[j];
			control.setVisible(j < types.length);
			if(control.isVisible()) {
				FragmentType type = types[j];
				Button button = (Button)control;
				button.setText(type.getName());
				button.pack(true);
				button.setSelection(type.equals(_model.getProbeMetaData().getFragmentType()));
			}
		}
	}
	
	void checkPageComplete() {
		_model.setFileName(_fileNameField.getText());
		_model.setSourceContainerPath(_sourceContainerField.getText());
		
		String errorMessage = null;
		if(!_model.isValidFileName()) {
			errorMessage = ProbekitMessages._53;
		}
		
		if(!_model.isValidFile()) {
			errorMessage = ProbekitMessages._54;
		}

		if(!_model.isValidSourceContainer()) {
			errorMessage = ProbekitMessages._55;
		}
	
		setErrorMessage(errorMessage);
		boolean isComplete = (errorMessage == null);
		setPageComplete(isComplete);
	}
	
	private boolean confirmChange(IProbeMetaData probe, FragmentType type) {
		boolean arePagesConsistent = _model.isValidCombination(probe, type);
		if(arePagesConsistent) {
			return true;
		}
		
		DiscardDataReadOnlyDialog dialog = createDialog(probe, type);
		dialog.open();
		int returnCode = dialog.getReturnCode();
		if(returnCode == Window.CANCEL) {
			return false;
		}
		else if(returnCode == Window.OK) {
			return true;
		}
		else {
			// Shouldn't happen, but just in case, don't prevent the change unless
			// you know why you shouldn't.
			return true;
		}
	}
	
	private class ProbeButtonSelectionListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing special needs to be done when the default button
			// is selected.
		}
		
		public void widgetSelected(SelectionEvent e) {
			updateProbe(e);
		}
	}
	
	private class FragmentButtonSelectionListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing special needs to be done when the default button
			// is selected.
		}
		
		public void widgetSelected(SelectionEvent e) {
			updateFragmentType(e);
		}
	}
	
	private class FileOrContainerModifiedListener implements ModifyListener {
		public void modifyText(ModifyEvent e) {
			checkPageComplete();
		}
	}
	
	private class XMLEncodingSelectionListener implements SelectionListener
	{
		public void widgetSelected(SelectionEvent e) {
			_model.setEncoding(_encodingField.getText());
		}

		public void widgetDefaultSelected(SelectionEvent e) {
		}
	}
	
	private class BrowseListener implements SelectionListener {
		public void widgetDefaultSelected(SelectionEvent e) {
			// Nothing special needs to be done when the default button
			// is selected.
		}
		
		public void widgetSelected(SelectionEvent e) {
			try {
				IJavaProject[] javaProjects = JavaUtil.getSourceJavaProjects();
				List sourceContainers = JavaUtil.getSourceContainers(javaProjects);

				CheckedTreeSelectionDialog dialog = DialogUtil.createResourceDialog(
						getContainer().getShell(), 
						JavaUtil.getProjects(javaProjects), 
						sourceContainers
					);
				dialog.setTitle(ProbekitMessages._58);
				dialog.setValidator(new SelectOneIncludedValidator(
						sourceContainers, ProbekitMessages._59,
						ProbekitMessages._60));
				
				IContainer selectedContainer = JavaUtil.getSourceContainer(_sourceContainerField.getText());
				if((selectedContainer != null) && (selectedContainer.isAccessible())) {
					dialog.setInitialSelection(selectedContainer);
				}
				
				dialog.open();
				
				Object[] selected = dialog.getResult();
				if(selected != null) {
					Object[] res = ResourceUtil.filter(selected, sourceContainers);
					_sourceContainerField.setText(((IContainer)res[0]).getFullPath().toString());
				}
			}
			catch(JavaModelException exc) {
	    		ProbekitUIPlugin.getPlugin().log(exc);
			}
		}
	}

	public boolean canFlipToNextPage() {
		if(_model.getProbeMetaData().getProbeType() == IProbeMetaData.NO_PROBE) {
			return false;
		}
		
		return super.canFlipToNextPage();
	}
}
