/************************************************************************
 * Copyright (c) 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id: Profiler.java,v 1.3 2008/02/07 14:47:31 ialelekov Exp $  
 ************************************************************************/
package org.eclipse.tptp.profiler;

/**
 * Exposes an API for controlling the profiler. 
 * This class is a singleton and cannot be instantiated by the user. 
 * In order to access it an application must use {@link #getProfiler()}.
 * 
 * @since TPTP 4.5
 */

public class Profiler {
	private final static String AGENT_NAME = "JPIAgent";
	private static Profiler instance = null;
	private static boolean nativesAvailable = false;

	private boolean isProfiling = false;
	private boolean isActive = false;

	static {
		try {
			System.loadLibrary(AGENT_NAME);
			nativesAvailable = true;
		}
		catch(Throwable e) {}
	}
	
	/**
	 * Native delegate to startup the profiler.  Checks if profiling is possible.
	 */
	 private native int initialize0();

	 private Profiler() {
		isActive = initialize0() == 0;
	}
	
	/**
	 * Returns the singleton Profiler's instance.
	 * 	 
	 * @return Profiler
	 * @throws ProfilerAPINotAvailableException - If API is disabled
	 * @throws ProfilerNotAvailableException - If native library <code>JPIAgent</code> not found
	 */
	public static Profiler getProfiler() throws ProfilerNotAvailableException {
		if (!nativesAvailable) throw new ProfilerNotAvailableException();
		
		if (instance == null) instance = new Profiler();
		
		if (!instance.isActive) throw new ProfilerAPINotAvailableException();
		
		return instance;
	}

	/**
	 * Native delegate of startProfiling().
	 * @returns  0 - the profile session was created successfuly.
	 *
	 */
	private native int startProfiling0();

	/**
	 * Starts data collection.
	 * 
	 * @return <code>true</code> if data collection started; <code>false</code> otherwise
	 * @throws ProfilerAPINotAvailableException - If API is disabled
	 */
	public boolean startProfiling() throws ProfilerAPINotAvailableException {
		if (!isActive) throw new ProfilerAPINotAvailableException();

		synchronized(this) {
			if (!isProfiling) {
				isProfiling = startProfiling0() == 0;
			}
		}
		
		return isProfiling;
	}
	
	/**
	 * Native delgate to stopProfiling.
	 */
	private native int stopProfiling0();
	
	/**
	 * Stops data collection.
	 * 
	 * @return <code>true</code> if data collection stopped; <code>false</code> otherwise
	 * @throws ProfilerAPINotAvailableException - If API is disabled
	 */
	public boolean stopProfiling() throws ProfilerAPINotAvailableException {
		if (!isActive) throw new ProfilerAPINotAvailableException();

		synchronized(this) {
			if (isProfiling) {
				isProfiling = !(stopProfiling0() == 0);
			}
		}
		
		return !isProfiling;
	}
	
	/**
	 * Native delgate to runGC.
	 */
	private native int runGC0();
	
	/**
	 * Runs the garbage collector.
	 * 
	 * @throws ProfilerAPINotAvailableException - If API is disabled
	 */
	public void runGC() throws ProfilerAPINotAvailableException {
		if (!isActive) throw new ProfilerAPINotAvailableException();

		runGC0();
	}
	
	/**
	 * Native delgate to emitXMLFragment.
	 */
	private native void emitXMLFragment0(String xmlFragment);
	
	/**
	 * Saves the XML fragment to the output destination.
	 * 
	 * @param xmlFragment - The XML fragment to save
	 * @throws ProfilerAPINotAvailableException - If API is disabled
	 */
	public void emitXMLFragment(String xmlFragment) throws ProfilerAPINotAvailableException {
		if (!isActive) throw new ProfilerAPINotAvailableException();

		emitXMLFragment0(xmlFragment);
	}
}
