/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_RTUTIL_MITERATOR
#define MRTE_RTUTIL_MITERATOR

namespace Martini { namespace RTUtil {

/**
 * @brief <b> IIterator Interface Class </b> An interface for an iterator
 * 
 */
template <class T>
class IIterator
{
public:
    /**
     * @brief <b> HasNext </b> Returns true if the iteration has more elements
     *
     * @returns - true if the iterator has more elements
     * @returns - false if the iterator points to the last element in the collection
     */
    virtual bool HasNext() = 0;

    /**
     * @brief <b> GetNext </b> Returns the next element in the iteration
     *
     * Returns the next element in the iteration, and advances the iterator.
     * The method returns NULL if the iterator has already returned all elements
     * of the collection (i.e., when a call to HasNext would return false).
     *
     * @returns - the next element in the iteration, or NULL if there are no more 
     * elements in the collection.
     */
    virtual T GetNext() = 0;

    /**
     * @brief <b> Free </b> Deallocates the iterator and all its resources.
     *
     * Deallocates the iterator object and all its resources. Call this method when
     * the iterator object is no longer needed. Do not use the delete operator.
     *
     * NOTE: if you don't call Free(), the iterator will be freed by the container
     *       from which it was allocated when the container is freed.
     */
    virtual void Free() = 0;
    
};

/**
 * @brief <b> IListIterator Interface Class </b> An interface for a bidirectional list 
 *            iterator.
 */
template <class T>
class IListIterator: public IIterator<T>
{
public:
    /**
     * @brief <b> GetFirst </b> Returns the first element in the list
     *
     * Returns the first element in the list, and advances the iterator. A successor
     * call to GetNext() will return the second element in the list.
     * The method returns NULL when the list is empty. 
     * Note that calling GetFirst() is equivalent to calling GetNext() immediately
     * after creating the IListIterator object.
     *
     * @returns - the first element in the list, or NULL if the list is empty.
     */
    virtual T GetFirst() = 0;

    /**
     * @brief <b> GetLast </b> Returns the last element in the list
     *
     * Returns the last element in the list, and moves the iterator backwards. A successor
     * call to GetPrev() will return the one-before-last element in the list.
     * The method returns NULL when the list is empty.
     *
     * @returns - the last element in the list, or NULL if the list is empty.
     */
    virtual T GetLast() = 0;

    /**
     * @brief <b> HasPrev </b> Returns true if there is a previous element in the list
     *
     * @returns - true if there is a previous element in the list
     * @returns - false if the iterator points at the first element in the list
     */
    virtual bool HasPrev() = 0;

    /**
     * @brief <b> GetPrev </b> Returns the previous element in the list
     *
     * Returns the previous element in the list, and moves the iterator backwards.
     * The method returns NULL if the iterator has passed the beginning of
     * the list. i.e. when a call to HasPrev() would return false.
     *
     * @returns - the previous element in the list, or NULL if the iterator has passed
     * the beginning of the list.
     */
    virtual T GetPrev() = 0;

     /**
     * @brief <b> Clone </b> Clones an iterator and returns the clone
     *
     */
    virtual IListIterator<T> * Clone() = 0;
};

}}


#endif // #define MRTE_RTUTIL_MITERATOR
