/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

#ifndef __MARTINI_EXTERNAL_CONTROL_H__
#define __MARTINI_EXTERNAL_CONTROL_H__

#include "MRTEResults.h"

/**
 * @file MartiniExternalControl.h
 * @brief Martini External Control Interface
 */

/** 
 * @page page_ec Martini External Control Interface
 *
 * @section page_ec_content Table of Contents
 * -# @ref ec_intro "Introduction"
 * -# @ref ec_init "External Control Module Initialization"
 * -# @ref ec_callbacks "External Control Callback Registration"
 * -# @ref ec_events "Generating External Control Events"
 *
 * @section ec_intro Introduction
 *
 * Most profiling tools are comprised of two layers, usually implemented as two separate
 * processes:
 * -# The Profiler back-end: a module (usually a dyanamically-linked library) loaded into
 *    the virtual machine process executing the profiled application. The back-end is 
 *    responsible for monitoring the application and collecting profiling data. All MPI 
 *    clients can be considered as back-end modules.
 * -# The Profiler front-end: a standalone application which implements the tool's user 
 *    interface and enables the user to interact with the profiler back-end.
 *
 * The Martini External Control Interface is designed to allow Profiler front-ends to
 * interact with their MPI back-end modules by sending "control" messages that will be
 * dispatched to the MPI back-end modules as MPI events.
 * 
 * To interact with MPI clients, a tool implements an "External Control" module which is
 * loaded into the virtual machine process together with the MPI client it needs to control.
 * This External Control module serves as a bridge between the tool's front-end process 
 * and the MPI client back-end and enables the front-end to send certain MPI events 
 * to the MPI client.
 *
 * The following figure outlines the relationship between the tool's front-end, the
 * External Control module, and MPI Client and the profiled application.
 *
 * @image html ECOverview.png "Martini External Control"
 *
 * It is important to note that the Martini External Control Interface does not provide
 * nor define any services for the tool's front-end to interact with its External Control
 * module. It only enables the External Control module to deliver messages to the MPI 
 * client as MPI events.
 *
 * A Martini External Control module is expected to implement and export the following
 * functions defined in the MartiniExternalControl.h header file:
 * -# @ref EC_Init : invoked by the Martini runtime during Virtual Machine initialization to
 *                   initialize the External Control module
 * -# @ref RegisterMartiniCallback : invoked by the Martini runtime when an MPI client 
 *                                   registers for an External Control event
 * -# @ref MartiniProcessShutdown : invoked by the Martini runtime just before the
 *                                  virtual machine shuts down
 * -# @ref MartiniMessage : invoked by the Martini runtime when certain error conditions
 *                          occur.
 * 
 * @section ec_init External Control Module Initialization
 *
 * The Martini External Control module library is loaded by the Martini runtime, 
 * according to its configuration (TBD). Once the External Control module is loaded, its 
 * @ref EC_Init entry-point function is called. The module is expected to initialize
 * itself in the context of the @ref EC_Init function and to establish the communication
 * channel with the tool's front-end. When @ref EC_Init completes, execution returns to the 
 * Martini runtime and the Virtual Machine process continues its initialization.
 *
 * @section ec_callbacks External Control Callback Registration
 *
 * When an MPI client registers for an external control event, the Martini runtime invokes
 * the @ref RegisterMartiniCallback function of the External Control module and hands it
 * a callback function pointer. The External Control module can then use this function
 * pointer to generate the an MPI event to be delivered to the loaded MPI client.
 * 
 * @section ec_events Generating External Control Events
 *
 * When the External Control module needs to send a notification to the MPI Client, it
 * uses the Martini callback function that corresponds to the event it wants to generate.
 *
 * The following External Control events are supported:
 * - @ref Martini::MPI::IEcStartEventObserver "Start"
 * - @ref Martini::MPI::IEcStopEventObserver "Stop"
 * - @ref Martini::MPI::IEcAttachEventObserver "Attach"
 * - @ref Martini::MPI::IEcDetachEventObserver "Detach"
 * - @ref Martini::MPI::IEcSetOutputDirEventObserver "Set Output Directory"
 * - @ref Martini::MPI::IEcCustomCommandEventObserver "Custom Command"
 *
 * @remark Except for the Set Output Directory event, the External Control events are not
 *         interpreted by the Martini runtime in any way, and delivered as-is to any 
 *         MPI Client which registered for these events. Therefore, the true meaning of
 *         these events should be determined and agreed by both the External Control module
 *         and the MPI Client it controls. It is recommended, however, that the 
 *         External Control module will follow the meaning of the External Control events as
 *         described in the documentation.
 *         This will allow the module to correctly interact with any MPI Client which also 
 *         follow these guidelines.
 */

namespace Martini { 

/**
 * @brief Martini External Control Interface namespace
 *
 * Defines the Martini External Control Interface API and related data types
 */ 
namespace ExternalControl
{

    /**
     * @brief Defines the commands supported by the External Control interface
     *
     * Defines the commands supported by the External Control interface
     **/
    enum ECommand 
    {
        CMD_CONFIGURE,      //!< @brief configures the output directory of the Martini runtime
                            //!  and loaded MPI client.
        CMD_START,          //!< notifies an MPI client to start collecting information 
        CMD_STOP,           //!< notifies an MPI client to stop collecting information
        CMD_ATTACH,         //!< notifies an MPI client to enable profiling events
        CMD_DETACH,         //!< notifies an MPI client to disable profiling events
        CMD_RELOAD_OPTIONS, //!< not supported
        CMD_CUSTOM,         //!< sends a custom command to an MPI client
        CMD_LAST
    };

    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcSetOutputDirEventObserver "Set Output Directory" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the 
     * @ref Martini::MPI::IEcSetOutputDirEventObserver "Set Output Directory" event
     */
    typedef void (*TConfigureCallback)(char*);

    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcStartEventObserver "Start" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the @ref Martini::MPI::IEcStartEventObserver "Start" event
     */
    typedef void (*TStartCallback)(void);

    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcStopEventObserver "Stop" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the @ref Martini::MPI::IEcStopEventObserver "Stop" event
     */
    typedef void (*TStopCallback)(void);
    
    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcAttachEventObserver "Attach" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the @ref Martini::MPI::IEcAttachEventObserver "Attach" event
     */
    typedef void (*TAttachCallback)(void);

    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcDetachEventObserver "Detach" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the @ref Martini::MPI::IEcDetachEventObserver "Detach" event
     */
    typedef void (*TDetachCallback)(void);

    /**
     * @brief Function prototype for generating the 
     * @ref Martini::MPI::IEcCustomCommandEventObserver "Custom Command" event
     *
     * Prototype for the callback function the External Control module should
     * invoke to generate the 
     * @ref Martini::MPI::IEcCustomCommandEventObserver "Custom Command" event
     * 
     * @param[in] commandId     a numeric id identifying the command
     * @param[in] pData         command data
     */
    typedef void (*TCustomCommandCallback)(unsigned int, void*);

    /**
     *	@brief Union of all external control command callbacks
     */
    union UCallback 
    {
        TConfigureCallback      pfnConfigure;   //!< @brief callback for generating the
                                                //!  Set Output Directory event
        TStartCallback          pfnStart ;      //!< callback for generating the Start event
        TStopCallback           pfnStop;        //!< callback for generating the Stop event
        TAttachCallback         pfnAttach;      //!< callback for generating the Attach event
        TDetachCallback         pfnDetach;      //!< callback for generating the Detach event
        TCustomCommandCallback  pfnCustomCommand;   //!< @brief callback for generating the
                                                    //!  Custom Command event
    };

    /**
     * @brief Defines message severity types for the @ref MartiniMessage function
     *
     * Defines message severity types for the @ref MartiniMessage function
     **/
    enum ESeverityLevel 
    {
        SL_NOT_ERROR,   //!< information message
        SL_WARNING,     //!< warning
        SL_FATAL,       //!< fatal error
    };

    /**
     * @brief Prototype for the @ref RegisterMartiniCallback function
     *
     * Prototype for the @ref RegisterMartiniCallback function exported from the 
     * External Control module.
     **/
    typedef int (*TRegisterMartiniCallback)(ECommand command, UCallback callback);

    /**
     * @brief Prototype for the @ref MartiniMessage function
     *
     * Prototype for the @ref MartiniMessage function exported from the External Control
     * module.
     **/
    typedef void (*TMartiniMessage)(char* szMsg, ESeverityLevel level);

    /**
     * @brief Prototype for the @ref MartiniProcessShutdown function
     *
     * Prototype for the @ref MartiniProcessShutdown function exported from the 
     * External Control module
     **/
    typedef void (*TMartiniProcessShutdown)();

    /**
     * @brief Prototype for the @ref EC_Init function
     *
     * Prototype for the @ref EC_Init function exported from the External Control module.
     */
    typedef TResult (*TECInit)(char **szWorkingDir, const char *szOptions);

}}

#ifdef linux
	#define EC_EXPORT 
#else // linux
	#ifdef ECWRAPPER_EXPORTS
		#define EC_EXPORT __declspec(dllexport)
	#else // ECWRAPPER_EXPORTS
		#define EC_EXPORT __declspec(dllimport)
	#endif // ECWRAPPER_EXPORTS
#endif // linux

/**
 * @brief Invoked by the Martini runtime when an MPI client 
 * registers for an External Control event
 *
 * Invoked by the Martini runtime when an MPI client registers for an External Control event.
 * The function specifies the External Control command associated with the event,
 * and the callback function to call in order to generate the event.
 *
 * @param[in] command   The command to register
 * @param[in] callback  The callback function to invoke in order to generate the event 
 *                      associated with @c command.
 *
 * @retval 1            If the command was successfully registered
 * @retval 0            If the command was not registered
 **/
extern "C" EC_EXPORT 
int RegisterMartiniCallback(Martini::ExternalControl::ECommand command, 
                            Martini::ExternalControl::UCallback callback);

/**
 * @brief Invoked by the Martini runtime to send a message to the External Control module
 *
 * This function is invoked by the Martini runtime when an MPI Client calls the
 * @ref Martini::MPI::IMpi::MessageLog "MessageLog" function, or when certain error conditions
 * occur within the Martini runtime.
 *
 * @param[in] szMsg     Message text
 * @param[in] level     Severity level. If level is SL_FATAL, the External Control
 *                      module should terminate the running process.
 */
extern "C" EC_EXPORT 
void MartiniMessage(char* szMsg, 
                    Martini::ExternalControl::ESeverityLevel level);

/**
 * @brief Invoked by the Martini runtime when the current process terminates
 *
 * This function is invoked by the Martini runtime to notify the External Control module
 * that the current process is being terminated.
 */
extern "C" EC_EXPORT 
void MartiniProcessShutdown(); 

/**
 * @brief External Control module initialization function
 *
 * This function is invoked by the Martini runtime to initialize the External Control module.
 *
 * @param[out] szWorkingDir     The directory that will be used by the Martini
 *                              runtime for logging. The External Control module may return
 *                              NULL if it does not wish to set the working directory.
 * @param[in] szOptions         The External Control module's command-line arguments passed 
 *                              to the Martini boot loader.
 *
 * @retval MRTE_RESULT_OK       The External Control module successfully initialized
 * @retval MRTE_ERROR_FAIL      The External Control module failed to initialize. 
 *                              The module should return this error when it can't complete its
 *                              initialization (for example, if the @c szWorkingDir argument
 *                              is invalid or is not large enough).
 *                              Returning this error will cause the profiled application
 *                              to terminate.
 */
extern "C" EC_EXPORT
TResult EC_Init(char **szWorkingDir, const char *szOptions);

#endif //__MARTINI_EXTERNAL_CONTROL_H__
