/*****************************************************************************
 * Copyright (c) 1997-2007, Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/
//
#ifndef MRTE_MIE
#define MRTE_MIE

#include <stdio.h>

  
#include "MRTEResults.h"
#include "MRTETypes.h"
#include "MIterator.h"


/**
 * @file
 * @brief Martini Instrumentation Engine
 *
 * Defines the Martini Instrumentation Engine API.
 * For more information, refer to the @ref page_mie "Martini Instrumentation Engine Interface" 
 * page.
 */

/** 
 * @page page_mie Martini Instrumentation Engine Interface
 *
 * @section intro Introduction
 *
 * Martini Instrumentation Engine (MIE) is designed to provide full support for low
 * level Java bytecode and MSIL (Microsoft Intermediate Language) inspection and 
 * instrumentation. 
 * MIE provides the instrumentation APIs that are relevant for both: Java and .NET. However,
 * although the two VMs have many things in common, they also differ in some senses. 
 * Additional specific Java related APIs are defined in JIE.h file, while specific .NET 
 * related APIs are defined in CIE.h.
 *
 * In general, instrumentation for .NET is designed to work in three stages:
 *
 * 1. Init time, in which the client registers to MPI events and acquires a CIE interface.
 *    Both operations are done via Martini MPI interface.
 *
 * 2. Module load time, in which the client may register managed function callbacks
 *    to be called at runtime. When and from where they should be called is specified
 *    by the client in the third stage. At this stage the client can iterate the methods, 
 *    change attributes of methods and find specific methods in the module. Requesting 
 *    IModule interface should be done at this stage
 *
 * 3. Just before method JIT compilation, in which IdotNETMethod interface can be acquired 
 *    for low level inspection or instrumentation of the method. During this stage the client 
 *    may request an IdotNETMethodInfo interface for methods that are called from this method.
 *
 * Instrumentation for Java is designed to work in two stages:
 *
 * 1. Init time, at which the client registers to MPI events and acquires a JIE interface.
 *    Both operations are done via Martini MPI interface.
 *
 * 2. During class-load-hook event, in which the class and its methods can be inspected and
 *    instrumented.
 */

//TODO: //DOC: behavior of "wide" instructions in Java (the "wide" opcode is converted to the instruction it widens)

#define MIE_VERSION_01

namespace Martini { 

namespace JIE
{    
    typedef U16 TConstantPoolIndex;
}

/**
 * @brief Martini Instrumentation Engine namespace
 *
 * Contains common definitions for Java and CLR instrumentation.
 */ 
namespace MIE
{
    /**
     * @brief Local Variable Index type
     */
    typedef U16 TVariableID;

    /**
     * @brief Defines Java or MSIL instructions, plus Martini pseudo instructions.
     **/
    enum EMnemonic
    {
#ifdef MARTINI_JIE

        #include "JavaMnemonics"            // Java opcodes

#elif defined MARTINI_CIE
        
        #include "CLR_ILMnemonics"          // .NET opcodes
#endif
        
#include "PsuedoMnemonics"
#include "MieSpecialMnemonics"              // Special MIE opcodes
        
    };


   /**
    * @brief Defines possible errors in Java or .NET instrumented code
    *
    * When the client is done instrumenting a method it should call a write back API 
    * to encode the instrumented method. At that point the client may call a validation API
    * that checks for trivial errors in the instrumented method. This enum contains the 
    * errors the instrumentation engine can catch. 
    **/
    enum EInstrumentationError
    {
        IE_NO_ERROR = 0,                 //!< no error
        IE_INSTRUCTION_HAS_NO_OPERAND,   //!< @brief operand exists for an instruction that 
                                         //!  accepts no operand
        IE_OPERAND_TYPE_MISMATCH,        //!< operand type does not match the instruction
        IE_LAST,                         //!< no code error is greater than this
    };

    class IInstruction; // forward reference
    
    /**
     * @brief Defines instruction iterator types
     **/
    enum EInstructionIteratorType
    {
        IET_ORIGINAL_INSTRUCTIONS = 0,   //!< @brief an iterator that "sees" only original
                                         //   method instructions and pseudo-instructions
        IET_ALL_INSTRUCTIONS             //!< @brief an iterator that "sees" all instructions 
                                         //   including those that were added or modified
                                         //   during instrumentation. Instructions that were
                                         //   deleted are not seen.
    };

    /**
     * @brief an Iterator over an ordered list of IInstruction objects.
     *
     * This type defines a bi-directional iterator over a method's instructions. It allows
     * navigating the instruction list in both directions and provides instant
     * access to the method's first and last instructions.
     */
    typedef RTUtil::IListIterator<IInstruction*> TInstructionListIterator;

    /**
     * @brief defines a bytecode or MSIL error.
     *
     * This struct describes an error in the code uncovered during code validation. It contains
     * an Instruction pointer associated with the offending instruction and the error
     * code
     * 
     **/
    struct SInstrumentationError
    {
        IInstruction*           pInvalidInstruction;    //!< offending instruction
        EInstrumentationError   errorType;              //!< error description
    };
    
   /**
    * @brief <b> SInstrumentationErrorsArray </b> an error array
    *
    * This struct defines a client allocated buffer for the instrumentation engine to use for
    * reporting bytecode or MSIL errors during code verification. The instrumentation engine 
    * uses up to uiErrorEntries and returns the actual number of used entries in 
    * uiActualNumErrors.
    *
    **/
    struct SInstrumentationErrorsArray
    {
        SInstrumentationError*  pErrors;           //!< Errors array
        unsigned int            uiErrorEntries;    //!< number of entries in the pErrors array
        unsigned int            uiActualNumErrors; //!< actual number of entries used
    };

   /**
    * @brief <b> SBuffer </b> a client allocated buffer 
    *
    * This struct defines a client allocated buffer, to be filled by the API function it is 
    * used in. Up to uiBufferSize bytes can be used, actual used size is updated by the API 
    * function in uiActualSize. If the buffer is not big enough to accommodate the requested 
    * data, the API function returns MRTE_ERROR_BUFFER_TOO_SHORT and uiActualSize contains 
    * the required buffer size.
    *
    **/
    struct SBuffer
    {
        unsigned char*    pBuffer;      //!< buffer pointer
        unsigned int      uiBufferSize; //!< buffer size counted in bytes
        unsigned int      uiActualSize; //!< actual data size when buffer is large enough, or
                                        //   required size when it is too short 
    };
    
   /**
    * @brief <b> SWideString </b> a client allocated wide character string
    *
    * This struct defines a client allocated wide character string, to be filled by the API 
    * function it is used in. Up to uiBufferSize wide characters can be used, actual used size
    * is updated by the API function in uiActualSize. If the buffer is not big enough to 
    * accommodate the requested data, the API function returns MRTE_ERROR_BUFFER_TOO_SHORT and
    * uiActualSize contains the required buffer size.
    *
    **/
    struct SWideString
    {
        unsigned short         *pStr;         //!< wide character string buffer pointer
        unsigned int            uiBufferSize; //!< string buffer size
        unsigned int            uiActualSize; //!< actual string size or required size
    };

   /**
    * @brief <b> EOperandType </b> enumerates all instruction operand types
    **/
    enum EOperandType
    {
        OT_NO_OPERAND = 0,        //!< No operand
        OT_INT8,                 //!< operand is signed 8 bits integer
        OT_UINT8,                //!< operand is unsigned 8 bits integer
        OT_INT16,                //!< operand is signed 16 bits integer
        OT_UINT16,               //!< operand is unsigned 16 bits integer
        OT_INT32,                //!< operand is signed 32 bits integer
        OT_UINT32,               //!< operand is unsigned 32 bits integer
        OT_INT64,                //!< operand is signed 64 bits integer
        OT_UINT64,               //!< operand is unsigned 64 bits integer
        OT_FLOAT32,              //!< operand is 32 bits float
        OT_FLOAT64,              //!< operand is 64 bits float
        OT_TARGET,               //!< operand is a branch target
        OT_TARGETS_ARRAY,        //!< operand is a Switch instruction multi target
        OT_VAR_ID,               //!< operand is a local variable ID
        OT_DOT_NET_TOKEN,        //!< operand is a .NET token
        OT_JAVA_CP_INDEX,        //!< operand is a Java constant pool index
        OT_LAST,                 //!< no operand type is greater than this 
    };

   /**
    * @brief <b> SCaseTarget </b> a pair of case value and target - for describing a switch 
    *                             instruction
    * 
    * This struct is used in the STargetsArray struct, in order to describe a switch 
    * instruction. It contains a pair of case value, and the target to jump to if the value of 
    * the switch instruction equals to the case value.
    *
    **/
    struct SCaseTarget
    {
        S32             caseValue; //!< value of the 'case'. Relevant only for the Java
                                   //   lookupswitch instruction.
        IInstruction*   pTarget;   //!< target instruction if the switch value equals the
                                   //   case value. 
    };

   /**
    * @brief <b> STargetsArray </b> a client allocated storage for the switch instruction 
    *                               operand target list
    *
    * This struct stores the switch instruction target list. The API function uses up to 
    * uiEntries and returns the actual number of targets in uiUsedEntries. If the array is not 
    * large enough the API function returns MRTE_ERROR_BUFFER_TOO_SHORT and uiUsedEntries  
    * contains the required size of the array.
    **/
    struct STargetsArray
    {  
        SCaseTarget     *pTargetsArray;     //!< an array with case-target pairs
        IInstruction    *pDefaultTarget;    //!< default target in case none of the case values 
                                            //   equals to the switch value
        unsigned int    uiEntries;          //!< Number of entries in ppTargetsArray
        unsigned int    uiUsedEntries;      //!< actual entries used or required
    };
    

    /**
    * @brief <b> SOperand </b> A structure that holds the operand type and value of 
    *                          an instruction
    *
    * Some of the fields are used for both Java and .NET. Some are related to Java only 
    * and some to .NET only
    **/
    struct SOperand
    {
        
        EOperandType	             type;         //!< type identifier

        union                                      //!< value union of all possible types
        {
            S8                       s8Op;         //!< specific size numeric value  
            U8                       ui8Op;        //!< specific size numeric value 
            S16                      s16Op;        //!< specific size numeric value 
            U16                      ui16Op;       //!< specific size numeric value 
            S32                      s32Op;        //!< specific size numeric value 
            U32                      ui32Op;       //!< specific size numeric value 
            S64                      s64Op;        //!< specific size numeric value 
            U64                      ui64Op;       //!< specific size numeric value  
            float                    f32Op;        //!< specific size numeric value 
            double                   d64Op;        //!< specific size numeric value 
            IInstruction*            pTarget;      //!< target of a branch
            STargetsArray            targetsArray; //!< array of targets, used with the switch
                                                   //   instruction
            TVariableID              varID;        //!< local variable id

#ifdef MARTINI_CIE
            // .NET specific
            U32                      token;
#endif

#ifdef MARTINI_JIE
            // Java specific
            JIE::TConstantPoolIndex  cpIndex;
#endif  
        } val;
    };
    
   /**
    * @brief <b> IMIE Interface Class </b> Martini Instrumentation interface
    *
    */
    //
    // Interface is named IMIEInterface to distinguish it from IMIE defined in
    // MieAPI.h. This is temporary until CIE is reimplemented to fully support
    // the MIE interface. See note at the beginning of this file for more
    // information.
    //
    //class IMIEInterface
    class IMIE
    {
    public:
       /**
        * @brief <b> GetVersion </b> Get the version of the implemented interface
        *
        */
        virtual unsigned int GetVersion() const = 0;
    };
    
   /**
    * @brief <b> IInstruction Interface Class </b> Instruction interface
    *
    * This interface can be used to retrieve information about a single instruction, change 
    * its mnemonics and its operands, and add new instructions before or after the current one.
    * Pseudo instructions associated with try..finally blocks can't be added and their operands
    * can't be set via this interface. Use the IMethod interface function BindTryFinallyBlock
    * to create such blocks.
    **/  
   
    class IInstruction
    {
    public:

        /**
         * @brief <b> GetMnemonic </b> returns the mnemonics of the instruction
         *
         * This API returns the mnemonics of the instruction
         *
         * @returns - EMnemonic - the mnemonics of the instruction.
         *
         */
        virtual const EMnemonic GetMnemonic() const = 0;

        /**
         * @brief <b> SetMnemonic </b> replaces the mnemonics of the instruction
         *
         * This API sets the mnemonics of the instruction to newMnemonic. The caller is
         * responsible for making sure that the operands are valid or to replace it as well
         * via the SetOperand API.
         * 
         * Note: If the number of operands expected for the new mnemonic is different
         *       from the number of operands of the current instruction, all operands of
         *       the current instruction will be discarded and the caller must set them
         *       via the SetOperand API.
         *
         * @param newMnemonic [in]  - the new mnemonics
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   newMnemonic is not valid
         * @returns - MRTE_ERROR_READ_ONLY          instruction was returned from an
         *                                          "inspection-only" iterator and cannot
         *                                          be modified.
         */
        virtual TResult SetMnemonic(const EMnemonic newMnemonic) = 0;

        /**
         * @brief <b> GetOperand </b> returns the operand(s) of the instruction
         *
         * This API returns the operand of the instruction within an SOperand structure.
         *
         * @param pOperand1 [out] - the caller allocated SOperand structure, including internal
         *                          buffers if required, to be filled by the instrumentation 
         *                          engine with the instruction's first operand
         * @param pOperand2 [out] - the caller allocated SOperand structure to be filled by 
         *                          the instrumentation engine with the instruction's second
         *                          operand. This argument is optional and is relevant only 
         *                          for instructions that accept more than one argument.
         *                          Relevant Java byte code instructions are: wide, iinc, 
         *                          multianewarray and invokeinterface.
         * @param pOperand3 [out] - the caller allocated SOperand structure to be filled by 
         *                          the instrumentation engine with the instruction's third
         *                          operand. This argument is optional and is relevant only 
         *                          for instructions that receive more than two arguments.
         *                          Relevant Java byte code instructions are: wide and
         *                          invokeinterface
         * @param pOperand4 [out] - the caller allocated SOperand structure to be filled by 
         *                          the instrumentation engine with the instruction's fourth
         *                          operand. This argument is optional and is relevant only 
         *                          for pseudo instructions.
         *
         * NOTE: for branch instruction, a pointer to the MARTINI_PSEUDO_TARGET
         *       instruction object which is the branch target of the current instruction
         *       is returned in pOperand1. An alternative approach for obtaining the branch
         *       target of a branch instruction is to call the GetBranchInstruction() method.
         *
         * @returns - MRTE_RESULT_OK                 success
         * @returns - MRTE_ERROR_BUFFER_TOO_SHORT    the STargetsArray member in the 
         *                                           caller's SOperand is not big enough
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT    at least one of the instruction operands 
         *                                           cannot be filled. The reason can be: its 
         *                                           pOperand buffer is NULL, or its internal
         *                                           STargetArray is not allocated (for switch
         *                                           instruction).
         * @returns - MRTE_RESULT_NO_OPERAND         instruction has no operand or not as many
         *                                           operands as the number of arguments
         */
        virtual TResult GetOperand(SOperand* pOperand1, 
                                   SOperand *pOperand2 = NULL, 
                                   SOperand *pOperand3 = NULL,
                                   SOperand *pOperand4 = NULL) const = 0;

        /**
         * @brief <b> SetOperand </b> sets the operand(s) of an instruction
         *
         * This API sets or replaces the operands of the instruction with new ones. The 
         * caller has to make sure the new operand types match the instruction mnemonics
         * or change the mnemonics. SetOperand does not assume that the pointer arguments 
         * are valid once it returns (i.e. SetOperand copies all needed info).
         *
         * @param pOperand1 [in] - the caller allocated new instruction SOperand 
         *                         structure including internal buffers if required. This is 
         *                         the instruction's first operand
         * @param pOperand2 [in] - the caller allocated operand to set as the instruction's
         *                         second operand. This is an optional argument, and should be
         *                         used for those instructions that have more than 1 argument.
         *                         Relevant Java byte code instructions are: wide, iinc, 
         *                         multianewarray and invokeinterface. 
         * @param pOperand3 [in] - the caller allocated operand to set as the instruction's
         *                         third operand. This is an optional argument, and should be
         *                         used for those instructions that have more than 2 arguments.
         *                         Relevant Java byte code instructions are: wide and 
         *                         invokeinterface
         * @param pOperand4 [in] - the caller allocated operand to set as the instruction's
         *                         fourth operand. This is an optional argument, and should be
         *                         used only for pseudu-instructions.
         *
         * NOTE: for branch instruction, pOperand1 must contain a pointer to the IInstruction
         *       object which is the branch target. If the target is not a pseudo 
         *       MARTINI_PSEUDO_TARGET, the pseudo MARTINI_PSEUDO_TARGET is automatically
         *       added just before the targeted instruction.
         *       An alternative approach for setting a branch target is to use the 
         *       SetBranchTarget() method. The SetBranchTarget method also returns the 
         *       pseudo target instruction and is the preferable method for setting
         *       branch targets.
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   Operand type is invalid
         * @returns - MRTE_ERROR_ARGUMENT_MISMATCH  Operand type and operand value mismatch.
         *                                          e.g. operand type is OT_TARGET or 
         *                                          OT_TARGETS_ARRAY and the operand value 
         *                                          contains invalid IInstruction pointer(s), 
         *                                          operand type is OT_VAR_ID and the value
         *                                          contains an invalid variable id
         * @returns - MRTE_ERROR_READ_ONLY          instruction was returned from an
         *                                          "inspection-only" iterator and cannot
         *                                          be modified.
         */
        virtual TResult SetOperand(SOperand *pOperand1, 
                                   SOperand *pOperand2 = NULL, 
                                   SOperand *pOperand3 = NULL,
                                   SOperand *pOperand4 = NULL) = 0;

        /**
         * @brief <b> AddBefore </b> inserts a new instruction before the current one
         *
         * This API adds a new instruction, with mnemonics newInstMnemonic and operands
         * pNewInstOperand1-3, immediately before this one. It fails if invoked on the 
         * pseudo MARTINI_PSEUDO_START instruction. All three operand arguments are optional,
         * and should be used according to the number of operands the inserted instruction 
         * accepts.
         *
         * @param ppInstruction   [out] - a buffer for the new added instruction interface, in 
         *                                case of success
         * @param newInstMnemonic  [in] - the mnemonics of the new added instruction
         * @param pNewInstOperand1 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction first operand
         * @param pNewInstOperand2 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction second operand
         * @param pNewInstOperand3 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction third operand
         *
         * @returns - MRTE_RESULT_OK               - success
         * @returns - MRTE_ERROR_OUT_OF_MEMORY     - out of memory
         * @returns - MRTE_ERROR_ILLEGAL_MNEMONIC  - illegal mnemonic
         * @returns - MRTE_ERROR_OPERAND_MISMATCH  - there is a mismatch between the mnemonic
         *                                           and its operands
         * @returns - MRTE_ERROR_ILLEGAL_OPERATION - trying to add an instruction before
         *                                           MARTINI_PSEUDO_START
         * @returns - MRTE_ERROR_FAIL              - internal error
         * @returns - MRTE_ERROR_READ_ONLY         - instruction was returned from an
         *                                           "inspection-only" iterator and cannot
         *                                           be modified.
         *      
         */
        virtual TResult AddBefore(IInstruction   **ppInstruction,
                                  const EMnemonic newInstMnemonic,
                                  const SOperand *pNewInstOperand1 = NULL,
                                  const SOperand *pNewInstOperand2 = NULL,
                                  const SOperand *pNewInstOperand3 = NULL) = 0;
        
        /**
         * @brief <b> AddAfter </b> inserts a new instruction after the current one
         *
         * This API adds a new instruction, with mnemonics newInstMnemonic and operands
         * pNewInstOperand1-3, immediately after this one. It fails if invoked on the 
         * pseudo MARTINI_PSEUDO_END instruction. All three operand arguments are optional,
         * and should be used according to the number of operands the inserted instruction 
         * accepts.
         *
         * @param ppInstruction   [out] - a buffer for the new added instruction interface, in 
         *                                case of success
         * @param newInstMnemonic  [in] - the mnemonics of the new added instruction
         * @param pNewInstOperand1 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction first operand
         * @param pNewInstOperand2 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction second operand
         * @param pNewInstOperand3 [in] - the caller allocated new instruction SOperand 
         *                                structure, including internal buffers if required. 
         *                                This operand is optional, and if not NULL, will be 
         *                                used as the inserted instruction third operand
         *
         * @returns - MRTE_RESULT_OK               - success
         * @returns - MRTE_ERROR_OUT_OF_MEMORY     - out of memory
         * @returns - MRTE_ERROR_ILLEGAL_MNEMONIC  - illegal mnemonic
         * @returns - MRTE_ERROR_OPERAND_MISMATCH  - there is a mismatch between the mnemonic
         *                                           and its operands
         * @returns - MRTE_ERROR_ILLEGAL_OPERATION - trying to add an instruction before
         *                                           MARTINI_PSEUDO_END
         * @returns - MRTE_ERROR_FAIL              - internal error
         * @returns - MRTE_ERROR_READ_ONLY         - instruction was returned from an
         *                                           "inspection-only" iterator and cannot
         *                                           be modified.
         */
        virtual TResult AddAfter(IInstruction   **ppInstruction,
                                 const EMnemonic newInstMnemonic, 
                                 const SOperand *pNewInstOperand1 = NULL,
                                 const SOperand *pNewInstOperand2 = NULL,
                                 const SOperand *pNewInstOperand3 = NULL) = 0;
        
        /**
         * @brief <b> IsOriginalInst </b> checks if an instruction is part of the original 
         *                                Java byte code or MSIL code of the method
         *
         * This API returns 'true' if invoked from an instruction that was present in the 
         * original method, or 'false' if it was added or modified during instrumentation.
         * For all pseudo instructions false is returned.
         * Instructions for which this flag is 'false' <b>will not</b> change to 'true'
         * as long as the method object containing these instructions is alive (even
         * after ApplyInstrumentation was invoked on the method).
         *
         * @returns - bool - true if original, false if not
         */
        virtual bool IsOriginalInst() const = 0;

        /**
         * @brief <b> GetOriginalInstOffset </b> Return the original byte offset from the 
         *                                       start of the method
         *
         * This API returns the instruction's original offset from the start of the method
         * 
         * @param pOriginalOffset [out]  - offset in the original Java byte code or MSIL code
         *
         * @returns - MRTE_RESULT_OK -                success
         * @returns - MRTE_RESULT_ADDED_INSTRUCTION - invoked on an added instruction
         * @returns - MRTE_ERROR_NOT_SUPPORTED -      this API is not supported
         *
         */
        virtual TResult GetOriginalOffset(unsigned int* puiOriginalOffset) const = 0;

        /**
         * @brief <b> GetBranchTarget </b> returns the target of a branch instruction
         *
         * This API returns the interface of the target instruction of a branch.
         * In most cases this is the pseudo MARTINI_PSEUDO_TARGET instruction that preceeds
         * the actual target instruction.
         * The call to this API will fail if the instruction on which it is invoked is
         * not a branch instruction.
         *
         * @returns - IInstruction* - the interface of the target instruction upon success 
         *                            NULL upon failure
         */
        virtual IInstruction* GetBranchTarget() const = 0;

        /**
         * @brief <b> SetBranchTarget </b> sets the target of a branch instruction
         *
         * This API sets the target of a branch instruction. If the target is not a pseudo 
         * MARTINI_PSEUDO_TARGET, the pseudo MARTINI_PSEUDO_TARGET is automatically added
         * just before the targeted instruction
         * The call to this API will fail if the instruction on which it is invoked is
         * not a branch instruction.
         *
         * @param pTargetInst [in] - the IInstruction interface of the new branch target
         *
         * @returns - IInstruction* - the interface of the MARTINI_PSEUDO_TARGET target
         *                            instruction upon success 
         *                            NULL upon failure
         */
        virtual IInstruction* SetBranchTarget(IInstruction *pTargetInst) = 0;
    };

    /**
    * @brief <b> IMethod Interface Class </b> Method Interface base class
    *
    * The IMethod interface provides a base class for low level inspection and 
    * instrumentation support for the method it is instantiated for. Once the caller is done 
    * with instrumenting the method and is ready to apply the changes, the WriteBack API 
    * should be called. 
    * While instrumenting the method, intermediate changes to the original Java byte code or 
    * MSIL code are either immediately visible or not depending on the type of instruction list
    * iterator interface the client requested. 
    * After completing instrumeting a method, the user should call IMethod::WriteBack() in 
    * order to apply the instrumentation changes. If the WriteBack method is not called, all
    * instrumentation changes for this method are discarded. 
    */
    class IMethod
    {
    public:

        /**
         * @brief <b> GetInstructionListIterator </b> returns an iterator over the 
         * method's instructions.
         *
         * This API initializes and returns a new iterator over the method's instructions.
         * The iteratorType argument specifies which type of iterator to create.
         *
         * The iterator points to objects of type IInstruction*.
         * 
         * * NOTE:when done using the iterator, call its Free method to deallocate the 
         * iterator. Do not use the delete operator.
         *
         * @param iteratorType [in] - the iterator type to create. tructions are visible.
         * 
         * @returns - an iterator over the method's instructions or NULL when no more
         *            memory is available.
         */
        virtual TInstructionListIterator *GetInstructionListIterator(
            const EInstructionIteratorType iteratorType) = 0;

        /**
         * @brief <b> BindTryFinallyBlock </b> binds a try..finally block
         *
         * This API is used to bind a try..finally block, i.e. associate between 
         *   MARTINI_PSEUDO_TRY
         *   MARTINI_PSEUDO_ENDTRY
         *   MARTINI_PSEUDO_FINALLY 
         *   MARTINI_PSEUDO_ENDFINALLY 
         * instructions, to define a try..finally block.
         * This API is currently supported for all types of exceptions, i.e. any type of
         * exception that is thrown between the MARTINI_PSEUDO_TRY and MARTINI_PSEUDO_ENDTRY
         * block is handled in the MARTINI_PSEUDO_FINALLY block.
         * The caller should ensure that the Java bytecode or MSIL code is correct after
         * adding the try..finally block, and is responsible for maintaining the correct
         * code structure. The code correctness can be verified by calling the VerifyCode
         * API.
         * This API will add the required pseudo instructions, which can't be added in any
         * other way to make sure they are bound correctly.
         *
         * [Java specific]
         * This API can be used on instance constructors (<init> methods) only when the
         * entire original code is inside the "try .. end_try" block. iTry must point
         * to the first original instruction and iEndTry must point to the last original
         * instruction. This limitation may be removed in the future.
         * [End Java specific]
         *
         *
         * @param iTry [in]           - the interface of the first instruction in the try 
         *                              block
         * @param iEndTry [in]        - the interface of the last instruction in the try 
         *                              block
         * @param iFinally [in]       - the interface of the first instruction in the 
         *                              finally block
         * @param iEndFinally [in]    - the interface of the last instruction in the 
         *                              finally block
         *
         * @returns - MRTE_RESULT_OK                success
         * @returns - MRTE_ERROR_ILLEGAL_ARGUMENT   at least one argument does not point to
         *                                          a valid instruction interface
         * @returns - MRTE_ERROR_OUT_OF_MEMORY      out of memory
         * @returns - MRTE_ERROR_ILLEGAL_TRY_BLOCK  there is a mismatch between the try-endtry
         *                                          or the finally-endfinally blocks
         * @returns - MRTE_ERROR_NOT_SUPPORTED      API is not yet supported
         *
         */
        virtual TResult BindTryFinallyBlock(IInstruction* iTry, 
                                            IInstruction* iEndTry, 
                                            IInstruction* iFinally, 
                                            IInstruction* iEndFinally) = 0;

        /**
         * @brief <b> VerifyCode </b> performs simple code validation
         *
         * This method tries to uncover basic instrumentation errors. It is advised to call it
         * prior to WriteBack() in order to help locating instrumentation errors.
         * The currently checked error is:
         *  1) mismatch between instruction mnemonics and its operand types
         * In the future the following errors will be flagged:
         *  2) 'ret' inside try blocks (.NET only)
         *  3) try block is not ended with a 'leave' instruction (.NET only)
         *  4) error in try blocks nesting
         *
         * @param pErrorsArray [out] - a caller allocated buffer to hold uncovered code errors
         *                             if NULL then the code will be checked up until the first
         *                             error (if any) and will return success/fail only.
         *
         * @returns - MRTE_RESULT_OK -   if no errors discovered
         *            MRTE_ERROR_FAIL -  otherwise
         */
        virtual TResult VerifyCode(SInstrumentationErrorsArray* pErrorsArray = NULL) const = 0;

        /**
         * @brief <b> ApplyInstrumentation </b> Generates a Java or MSIL method from the 
         *                                      method instruction stream and replaces the 
         *                                      original method with the new generated code.
         *
         * This API is used to write back the new instrumented method. It generates a new
         * Java or MSIL method from the current flow graph of instructions, replacing the
         * original method with the new one (in memory) and fixing the line info.
         * When instrumenting Java byte code the client needs to call the WriteBack() class 
         * interface function to present the new method as part of the whole class
         * to the JVM.
         *
         * * IMPORTANT: This API invalidates the method and associated data
         *              structures. It should be the last API invoked on the method object.
         *              After calling ApplyInstrumentation, the client can no longer 
         *              inspect and modify the method code. Any references to IInstruction
         *              objects will become invalid and using them may cause unexpected
         *              results.
         *
         * @returns - MRTE_RESULT_OK          success
         * @returns - MRTE_ERROR_FAIL         failure
         */
        virtual TResult ApplyInstrumentation() = 0;

        /**
         * @brief <b> DebugPrint </b> prints the Java byte code or the MSIL code disassembly
         *                            of the method to the specified output file
         *
         * This API is provided as a debug service to the client module developer for printing 
         * the disassembly of the instrumented method to verify its correctness
         *
         * @param bResolveNames [in]  - Determines whether to resolve tokens/indexes to their
         *                              real value. (Class::Method, strings, etc)
         *                              If 'false': tokens/indexes are printed in hex
         * @param FileName      [in] -  optional output file name. if NULL or omitted,
         *                              output is written to stdout
         *
         * @returns - MRTE_RESULT_OK          success
         */
        virtual TResult DebugPrint(bool bToResolveNames, char* fileName = NULL) = 0;
    };
}}

#endif // #define MRTE_MIE


