/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

/**
 * @file MpiAPI.h
 * @brief Martini Profiling Interface.
 *
 * An API exposed to clients by all Martini interface modules that attach to 
 * Virtual Machines. Those interface modules are: JPI (Java), CPI (.NET)
 * and JITPI (Any VM + JITProfiling).
 * For more information refer to the @ref page_mpiapi "Martini Profiling Interface" page.
 */

//
// Introductory material for the MPI Reference Manual (Doxygen) is available in MpiBase.h
//

#ifndef _MRTE_MPI_H
#define _MRTE_MPI_H

#include "MRTETypes.h"
#include "MRTEResults.h"
#include "MieAPI.h"
#include "MpiBase.h"

namespace Martini 
{ 

/**
 * @brief Martini Profiling Interface namespace
 *
 * Defines the Martini Profiling Interface API and related data types
 */ 
namespace MPI
{
    //////////////////////////////////////////////////////////////////////////
    // General Data Definitions
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Id Type for most MPI elements: modules, classes, methods, etc...
     **/
	typedef U64 TId;

    /**
     * @brief Time stamp type
     */
    typedef U64 TTimeStamp;

    /**
     * @brief Defines an array of MPI ids
     *
     * Defines an array of MPI ids. 
     * For more information on using MPI arrays, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     */
    struct TIdArray
    {
        TId *entries;               //!< Array of MPI ids
        unsigned int uiSize;        //!< @brief The number of entries allocated, either by MPI
                                    //!  during an event callback, or by the client when
                                    //!  using a Data Request API.
        unsigned int uiActualSize;  //!< @brief the actual number of entries available after a
                                    //!  Data Request operation
    };
    
    /**
     * @brief Defines MPI version information
     */
    struct SVersionInfo
    {
        U32 major;          //!< MPI major version number
        U32 minor;          //!< MPI minor version number
    };

    /**
     * @brief Defines MPI Event Groups.
     * 
     * Defines MPI Event Groups.
     */
    enum EEventGroup
    {
        EG_NONE,                   //!< @brief No group
            
        EG_CALL_GRAPH,             //!< @brief Call Graph events:
                                   //!  EV_METHOD_ENTER,
                                   //!  EV_METHOD_LEAVE

        EG_HEAP,                   //!< @brief Heap events:
                                   //!  EV_OBJECT_ALLOC,
                                   //!  EV_OBJECT_FREE

        EG_MONITOR,                //!< @brief Monitor events:
                                   //!  EV_MONITOR_WAIT,
                                   //!  EV_MONITOR_WAITED,
                                   //!  EV_CONTENDED_MONITOR_ENTER,
                                   //!  EV_CONTENDED_MONITOR_ENTERED

        EG_THREAD_INTERACTION,     //!< @brief Thread Interaction events:
                                   //!  EV_THREAD_INTERACTION

        EG_LAST             //! End sentinel
    };

    /**
     * @brief Defines message types for the IMpi::MessageLog function
     *
     * Defines message types for the IMpi::MessageLog function
     **/
    enum EMessageType
    {
        MT_ERROR,           //!< @brief A critical error.
                            //!  When an External Controller module is loaded, it will
                            //!  terminate the application in response to this message

        MT_INFORMATIVE,     //!< Informative message

        MT_DEBUG            //!< Debug message.
    };
    
    /**
     * @brief Defines possible destinations for messages sent with the IMpi::MessageLog 
     * function
     *
     * Defines possible destinations for messages sent with the IMpi::MessageLog function.
     * Values can be OR'ed together to specify multiple destinations.
     *
     **/
    enum EMessageDest
    {
        MD_LOG_FILE         = 0x01, //!< Message is sent to the log file of the MPI module
        MD_EXTERNAL_CONTROL = 0x02  //!< @brief Message is sent to an External Control module, 
                                    //!  if one is loaded
    };

    /** 
     * Defines configuration data
     *
     * Used with the IMpi::Configure() function to configure the virtual
     * machine and the loaded Martini interface module.
     * This struct currently stands as a place holder for future extensibility of 
     * configuration data, and will grow as needed
     */
    struct SConfigData
    {
    };

    /**
     * Defines configuration options
     * 
     * Used with the IMpi::Configure() function as a bit-mask for the configuration options
     * of the loaded Martini interface module.
     * 
     * @see IMpi::Configure()
     *
     * @remark All configuration options are currently supported only in .NET.
     */
    enum EConfigurationFlags
    {
        //!< None
        CF_NONE                                                 = 0x00000000,

        //! @brief Set this to enable the creation of the StackMapTable method
        //! attribute when instrumenting Java classes with version >= 50.0 
        //! (Java 6 and above)
        CF_JAVA_ENABLE_STACKMAP_CALC                            = 0x00000001,

		//! @brief Set this to receive native VM id's instead of MPI allocated id's. 
        //! Applies for the following MPI elements: thread, module, class and method
        CF_DOTNET_USE_VM_IDS                                    = 0x00010000,  

        //! @brief Set this to disable inlining in the VM
        CF_DOTNET_NO_INLINING                                   = 0x00020000,

        //! @brief Set this to disable the VM from using pre-JITted (NGEN'ed) modules
		CF_DOTNET_DISABLE_PREJIT_USE                            = 0x00040000,

        //! @brief End sentinel
        CF_LAST                                                 = 0x80100000
    };

    /**
     * @brief Defines MPI supported runtime types (Virtual Machines)
     *
     * Defines MPI supported runtime types (Virtual Machines)
     **/
    enum ERuntimeType
    {
        RT_JAVA = 2,            //!< Java
        RT_DOTNET,              //!< .NET
        RT_JIT_PROFILING = 9,   //!< JIT profiling
        RT_UNDEFINED
    };

    /**
     * @brief Defines method access (scope or visibility) types
     *
     * Defines method access (scope or visibility) types. Used with Call Graph filters
     **/
    enum EMethodAccessType
    {
        //! @brief None
        MAT_NONE = 0,

        //! @brief Method can be accessed from within the same class only (private method)
        MAT_CLASS_INTERNAL,

        //! @brief Method can be accessed from within the Java package only (public method of
        //! a package-internal class). 
        MAT_PACKAGE_INTERNAL,

        //! @brief Same as MAT_PACKAGE_INTERNAL, using .NET nomenclature instead of Java
        MAT_MODULE_INTERNAL = MAT_PACKAGE_INTERNAL,

        //! @brief Method can be accessed from anywhere, including other Java packages (public
        //! method of a public class)
        MAT_PACKAGE_API,

        //! @brief Same as MAT_PACKAGE_API, using .NET nomenclature instead of Java
        MAT_MODULE_API = MAT_PACKAGE_API,
    };

    /**
     * @brief Defines a mapping between two types of offsets
     *
     * Defines a line number entry that map between any of the following:
     *
     * 1) JITted code and source line numbers 
     *
     * 2) JITted code and managed code (Java Byte Code or .NET IL)
     *
     * 3) Managed code and source line number
     *
     * Used within the SLineNumberTable struct.
     **/
    struct SLineNumberTableEntry
    {
        unsigned int uiOffset;      //!< offset from beginning of method, 
                                    //   either JIT or managed code
        unsigned int uiLineNumber;  //!< line number from beginning of file, 
                                    //   either managed code or source line number
    };

    /**
     * @brief Defines a line number table
     *
     * Defines a line number table. Each entry in the table maps an offset to
     * a line number. The actual meaning of the data depends on the type of line information
     * retrieved.
     * 
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     * 
     * @see SLineInfo
     * 
     */
    struct SLineNumberTable
    {
        SLineNumberTableEntry *entries; //!< array of line number table entries
        unsigned int uiSize;            //!< @brief the number of entries allocated, either by
                                        //!  MPI during an event callback, or by the client when
                                        //!  using a Data Request API.
        unsigned int uiActualSize;      //!< @brief the actual number of entries available 
                                        //!  after a Data Request operation
    };

    /**
     * @brief Defines .NET module pre-JIT information
     *
     * Defies module pre-JIT information. Used with the SMethodJit struct
     */
    struct SDotNetModulePrejit
    {
        void  *pPrejitModuleBaseAddr;      //!< prejit module base address 
        const char  *szModuleName;         //!< prejit module name
    };

    /**
     * @brief Defines a region of continuous native (JITted) code
     * 
     * Defines a region of continuous native (JITted) code. Used in the SCodeInfo struct.
     * 
     * @see SCodeInfo
     */
    struct SCodeRegion 
    {
        const void  *pCodeAddr;                  //!< JITted code start address
        unsigned int uiCodeSize;                 //!< JITted code size in bytes
    };
    
    /**
     * Defines JITted code information
     * 
     * This struct stores JITted code information. The code of a method may be split into
     * several non-continuous regions. Each such region is represented by an entry in the
     * pCodeRegions array.
     *
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     * 
     * @remark - if the entire function is compiled into one code region, the pCodeRegions
     *           array will include only one entry.
     *
     * @see SCodeRegion
     */
    struct SCodeInfo 
    {
        SCodeRegion* pCodeRegions;      //!< array of code regions
        unsigned int uiSize;            //!< @brief the number of entries allocated, either 
                                        //!  by MPI during an event callback, or by the client
                                        //!  when using a Data Request API.
        unsigned int uiActualSize;      //!< @brief the actual number of entries available
                                        //!  after a Data Request operation
    };
    
    /**
     * @brief Defines stack entry information
     *
     * Defines stack entry information. Used in the SStackTrace struct.
     */
    struct SStackEntry
    {
        TId methodId;   //!< Method id
        U64 location;   //!< @brief Either the VM code offset from the beginning of the method,
                        //!  or the absolute native IP, depending on the requested stack
                        //!  trace type.
    };

    /**
     * @brief Defines a stack trace
     *
     * Defines stack trace information.
     *
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     **/
    struct SStackTrace
    {
        SStackEntry*      pStackEntries;    //!< Array of stack entries
        unsigned int      uiSize;           //!< @brief
                                            //!  The number of elements allocated for the 
                                            //!  @c pStackEntries array
        unsigned int      uiActualSize;     //!< @brief
                                            //!  Actual number of elements returned in the 
                                            //!  @c pStackEntries
                                            //!
                                            //!  If @c uiActualSize > @c uiSize it means that
                                            //!  not enough memory was allocated for the
                                            //!  @c pStackEntries array.
    };
    
    /**
     * @brief Defines .NET GC Moved References information
     *
     * Encapsulates .NET GC Moved references event info. Check clrprof.h for the original
     * API elements and the Profiling.doc document from the .NET SDK for more information
     **/
    struct SGgMovedRefInfo
    {
        int* oldObjectIDRangeStart;     //!< start of original range
        int* newObjectIDRangeStart;     //!< start of new range
        int* cObjectIDRangeLength;      //!< range length
        int cMovedObjectIDRanges;       //!< number of entries
    };

    /**
     * @brief Defines JVM internal dynamic code information
     *
     * Defines method JVM internal dynamic code information. When the MPI client requests 
     * DR_JAVA_DYNAMIC_CODE as part of the definition of the Java Dynamic Code Generated Event
     * Observer, the jvmDynamicCode field in the SJavaDynamicCodeData struct will be filled 
     * with a valid SJavaDynamicCode struct.
     **/
    struct SJavaDynamicCode
    {
        const char *szCodeName;                 //!< dynamic code name 
        const void *pCodeAddr;                  //!< dynamic code start address
        unsigned int uiCodeSize;                //!< dynamic code size in bytes
    };

    /**
     * @brief Defines a method's first and last source line numbers
     *
     * Defines the first and last source line numbers of a method. This structure is not 
     * related to the corresponding JITted code that is generated for the method. 
     * To receive method JIT information, the MPI client should register to the Jitted Method
     * Loaded event. This struct is useful when the line number table in the SMethodJit is
     * not available.
     * <br>
     * If the MPI client successfully requested the DR_METHOD_LINE_NUMBERS item in a call to
     * IMpi::GetMethodInfo, the methodLineNumbers field in the SMethodInfo struct will be 
     * filled with a valid SMethodLineNumbers struct.
     *
     * @see SMethodJit
     **/
    struct SMethodLineNumbers
    {
        unsigned int uiStart; //!< source line number of the beginning of the method
        unsigned int uiEnd;   //!< source line number of the end of the method
    };

    /**
     * @brief Defines array types
     *
     * Enumerator indicating the type of an array object. Used with the SObjectInfo struct.
     */
    enum EArrayType
    {
        AT_NONE,      //!< object is not an array
        AT_CLASS,     //!< array of classes
        AT_BOOL,      //!< array of booleans
        AT_CHAR,      //!< array of chars
        AT_S1,        //!< array of signed bytes
        AT_U1,        //!< array of unsigned bytes
        AT_S2,        //!< array of signed double-bytes
        AT_U2,        //!< array of unsigned double-bytes
        AT_S4,        //!< array of signed 4-bytes
        AT_U4,        //!< array of unsigned 4-bytes
        AT_S8,        //!< array of signed 8-bytes
        AT_U8,        //!< array of unsigned 8-bytes
        AT_FLOAT,     //!< array of floats
        AT_DOUBLE,    //!< array of doubles
        AT_STRING,    //!< array of strings
        AT_VOID,      //!< array of voids
        AT_OTHER,     //!< array of unknown types
        AT_LAST,
    };

    /**
     * @brief Defines array dimension information
     *
     * Defines array dimension information
     */
    struct SArrayDimensionInfo
    {
        unsigned int size;      //!< Number of elements in one of the array dimension 
        int lowerBound;         //!< Lower-bound of the array dimension
    };


    /**
     * @brief Defines array dimensions
     * 
     * Defines a data structure for storing dimension information for multi-dimensional arrays.
     * The @c infos array contains one entry for each dimension of the array.
     *
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     */
    struct SArrayDimensionsInfos
    {
        SArrayDimensionInfo *infos;     //!< Array of dimensions
        unsigned int uiSize;            //!< Number of elements allocated for the @c infos
                                        //!  array
        unsigned int uiActualSize;      //!< Actual number of elements returned in the @c infos
                                        //!  array. If @c uiActualSize > @c uiSize it means that
                                        //!  not enough memory was allocated for the 'infos'
                                        //!  array.
    };

    /**
     * @brief Defines array information
     *
     * Defines array information. Used with the SObjectInfo struct.
     */
    struct SArrayInfo
    {
        void    *dataPtr;                       //!< @brief
                                                //!  A pointer to the raw buffer for the array, 
                                                //!  which is laid out according to the C++
                                                //!  convention (.NET only)

        TId     arrayBaseClassId;               //!< Class ID of the base type of the array
        TId     arrayBaseModuleId;              //!< Module ID of the base type of the array
        TId     arrayBaseClassToken;            //!< @brief
                                                //!  Class Token of the base type of the array
                                                //!  (.NET only)
        U32     numDimensions;                  //!< Number of dimension in array
        SArrayDimensionsInfos dimensionsInfo;   //!< Size and lower bound of each dimension
    };

    /**
     * @brief Defines object information for the IMpi::GetObjectInfo API and
     * the Heap events.
     * 
     * @remark - for .NET arrays, it may not be possible to determine the
     *           @c moduleId and @c classToken of the object, even when the 
     *           base type of the array can be determined. <br>
     *           For example, given the following definition: <br>
     *           <code>Point [5] scores;</code> <br>
     *           we can determine the @c moduleId and @c classToken of
     *           @c Point , but not of @c scores. In this case, the caller can pass a 
     *           non-null @c moduleId and @c classToken to be used when they cannot be 
     *           determined from the .NET API calls.
     **/
    struct SObjectInfo
    {
        U64         uiSize;     //!< Object size in bytes
        TId         classId;    //!< Id of the object's class
        TId         moduleId;   //!< @brief Id of the module in which the object's class is 
                                //!  defined (.NET only)
        TId         classToken; //!< Token of the defining class (.NET only)
        EArrayType  arrayType;  //!< Array type, or AT_NONE if the object is not an array
        SArrayInfo  arrayInfo;  //!< Additional information for array objects (.NET only)
    };

    //////////////////////////////////////////////////////////////////////////
    // Data Request Items
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Data Type for MPI Data Request Types
     **/
    typedef U32 TDataRequestType;

    /**
     * @anchor data_request_types
     * @name Data Request Types
     * The Data Request Types supported by MPI. Used for specifying the data items for
     * an MPI event or a data request. Can be OR'ed together for specifying multiple items.
     */

    // Data items are defined as bits. The lower 16 bits are reserved for common items
    // that are used in multiple events and data requests.
    // The higher 16 bits are used for data items that are specific to a certain event
    // or a data request. Several data items can use the same bit, as long as
    // these data items can never be specified together in the same data request 
    // operation.

    //@{

    //! @brief No data
    const TDataRequestType DR_NONE                              = 0x00000000;

    // --- Data items which are common to multiple events and data requests ---

    //! @brief Thread id. Unique for the duration of the profiled application
    const TDataRequestType DR_THREAD_ID                         = 0x00000001;
    
    //! @brief Unique module id
    const TDataRequestType DR_MODULE_ID                         = 0x00000002;

    //! @brief Unique class id
    const TDataRequestType DR_CLASS_ID                          = 0x00000004;

    //! @brief Unique method id
    const TDataRequestType DR_METHOD_ID                         = 0x00000008;

    //! @brief Unique object id
    const TDataRequestType DR_OBJECT_ID                         = 0x00000010;

    // --- Data items for method information ---

    //! @brief Method name (developer-friendly)
    //!
    //! Method name in a developer friendly format. The internal VM method name
    //! is translated to the name used by developers in their programs, and is the one
    //  expected by developers.
    const TDataRequestType DR_METHOD_NAME                       = 0x00010000;

    //! @brief Method signature (developer-friendly)
    //!
    //! Method signature in a developer friendly format. The internal VM signature
    //! is translated to the signature expected by developers
    const TDataRequestType DR_METHOD_SIGNATURE                  = 0x00020000;

    //! @brief Name of the source file in which a method is defined
    const TDataRequestType DR_SOURCE_FILE_NAME                  = 0x00040000;

    //! @brief First and last source file line numbers of a method. 
    //  Used with SMethodLineNumbers
    const TDataRequestType DR_METHOD_LINE_NUMBERS               = 0x00080000;

    //! @brief Module name
    const TDataRequestType DR_MODULE_NAME                       = 0x00100000;

    //! @brief Class name
    const TDataRequestType DR_CLASS_NAME                        = 0x00200000;

    //! @brief Compiled method code
    const TDataRequestType DR_METHOD_CODE                       = 0x00400000;

    //! @brief Mapping of compiled method JITted (native) code to source line numbers
    const TDataRequestType DR_NATIVE_TO_SRC_LINE_MAP            = 0x00800000;

    //! @brief Mapping of compiled method JITted (native) code to managed code (Java byte-code
    //! or .NET IL) offsets
    const TDataRequestType DR_NATIVE_TO_MANAGED_LINE_MAP        = 0x01000000;

    //! @brief Mapping of managed code (Java byte-code or .NET IL) to source line numbers
    const TDataRequestType DR_MANAGED_TO_SRC_LINE_MAP           = 0x02000000;

    //! @brief Native (non-translated) class name (Java-only)
    const TDataRequestType DR_JAVA_NATIVE_CLASS_NAME            = 0x04000000;

    const TDataRequestType DR_DOTNET_MODULE_PREJIT              = 0x10000000;

    //! @brief Method token (.NET only)
	const TDataRequestType DR_DOTNET_METHOD_TOKEN               = 0x20000000;

    //! @brief Native (non-translated) method name (Java-only)
    const TDataRequestType DR_JAVA_NATIVE_METHOD_NAME           = 0x40000000;

    //! @brief Native (mangled) method signature (Java-only)
    const TDataRequestType DR_JAVA_NATIVE_METHOD_SIGNATURE      = 0x80000000;

    // --- Data items for thread information --- 

    //! @brief Thread name
    const TDataRequestType DR_THREAD_NAME                       = 0x00010000;

    //! @brief Thread group name. Currently supported only for Java
    const TDataRequestType DR_THREAD_GROUP                      = 0x00020000;

    //! @brief Name of the parent group of this thread's group. 
    //! Currently supported only for Java
    const TDataRequestType DR_THREAD_PARENT_GROUP               = 0x00040000;

    //! @brief Stack trace of the current thread, using absolute machine IP (.NET only)
	const TDataRequestType DR_ABSOLUTE_NATIVE_STACK_TRACE       = 0x00080000;

    //! @brief Stack trace of the current thread, using relative VM instruction offsets
	const TDataRequestType DR_VM_RELATIVE_STACK_TRACE           = 0x00100000;

    //! @brief Thread elapsed time
    const TDataRequestType DR_THREAD_ELAPSED_TIME               = 0x00200000;

    //! @brief Thread elapsed time
    const TDataRequestType DR_THREAD_CPU_TIME                   = 0x00400000;

    //! @brief Thread state
    const TDataRequestType DR_THREAD_STATE                      = 0x00800000;

    //! @brief Id of the object (if any) whose monitor the thread is waiting for (Java only)
    const TDataRequestType DR_CURRENT_MONITOR                   = 0x01000000;

    //! @brief Thread information (name, group and parent group)
    const TDataRequestType DR_THREAD_INFO                       = DR_THREAD_NAME |
                                                                  DR_THREAD_GROUP |
                                                                  DR_THREAD_PARENT_GROUP;

    // --- Data items for object and monitor information ---
    
    //! @brief Id of the thread owning the contended monitor
    const TDataRequestType DR_MONITOR_OWNER_THREAD_ID           = 0x00010000;
    
    //! @brief Object information
    const TDataRequestType DR_OBJECT_INFO                       = 0x00020000;

    //! @brief Monitor timeout for "wait" operations, in milliseconds
    const TDataRequestType DR_MONITOR_TIMEOUT                   = 0x00040000;

    //! @brief Monitor timeout expiration status for "wait" operations
    const TDataRequestType DR_MONITOR_TIMED_OUT                 = 0x00800000;

    //! @brief Type of interaction between threads,
    const TDataRequestType DR_THREAD_INTERACTION_TYPE           = 0x01000000; 

    //! @brief List of threads waiting to own the monitor
    const TDataRequestType DR_MONITOR_OWN_WAITERS               = 0x00080000;

    //! @brief List of threads waiting to be notified by the monitor
    const TDataRequestType DR_MONITOR_NOTIFY_WAITERS            = 0x00100000;

    //! @brief The VM instruction index (relative to the beginning of the method)
    //! where an object was allocated. Can be used with DR_METHOD_ID to identify
    //! the object allocation site (Java-only)
    const TDataRequestType DR_ALLOC_VM_INSTRUCTION_OFFSET       = 0x00200000;

    //! @brief The number of Garbage Collection cycles an object survived (Java-only) 
    const TDataRequestType DR_OBJECT_AGE                        = 0x00400000;
    // --- Data items for heap information ---

    //! @brief Heap usage information. Used with SHeapUsage (Java-only)
    const TDataRequestType DR_JAVA_HEAP_USAGE                   = 0x00010000;

    // --- Data items for configuration information ---

    //! @brief Activity path defined by an External Controller with the EV_EC_CONFIGURE
    //! event
    const TDataRequestType DR_ACTIVITY_PATH                     = 0x00010000;

    // --- Data items for Garbage Collection information ---

    //! @brief Information about objects moved during Garbage Collection (.NET only)
	const TDataRequestType DR_DOTNET_GC_MOVED_REF_INFO          = 0x00010000;

    // --- Data items for dynamic code information ---

    //! @brief Dynamic code information (Java-only)
    const TDataRequestType DR_JAVA_DYNAMIC_CODE                 = 0x00010000;

    // --- Data items for custom commands ---

    //! @brief Id of a Custom Command passed with the EV_EC_CUSTOM_COMMAND event
    const TDataRequestType DR_COMMAND_ID                        = 0x00010000;

    //! @brief Custom Command Data passed with the EV_EC_CUSTOM_COMMAND event
    const TDataRequestType DR_COMMAND_DATA                      = 0x00020000;
    //@}
    
    //////////////////////////////////////////////////////////////////////////
    // Generic Event Definition
    //////////////////////////////////////////////////////////////////////////

    /**
     * @interface IEventObserver
     * @brief A base interface for all Event Observer interfaces.
     *
     * This interface defines the base operations of all MPI Event Observers. A derived
     * interface class is defined for each MPI event.
     *
     * For more information about handling defining event observers and handling MPI events,
     * refer to @ref event_handling "MPI Event Handling".
     *
     * The MPI implementation retrieves the requested event type and data types from the Event 
     * Observer object via the @c Type() and @c EventDataTypes() interface functions.
     *
     * After registration, when the requested event occurs, the Event Observer's 
     * @c HandleEvent() function is called with the requested data as an argument.
     */
    class IEventObserver
    {
    public:
        virtual ~IEventObserver() {}
        
        /**
         * @brief Returns the event type supported by the observer.
         *
         * Returns the MPI event type requested by the observer. Implemented for each 
         * observer interface class. MPI clients that implement event-specific observer 
         * classes must not override this method.
         *
         * @returns the MPI event type
         */
        virtual TEventType Type() = 0;

        /**
         * @brief Returns the data items to provide with the event when it occurs.
         *
         * Specifies the data items to provide with the event when it occurs.
         * Data items are specified as a combination of TDataRequestType constants 
         * OR'ed together. By default, an event-specific observer interface
         * returns the maximum possible data items that can be returned with that
         * event. MPI clients can override this method to specify a subset of those data items
         * to be returned with the event. 
         *
         * @returns a combination of bits, each represents a data item to provide when the 
         *          event occurs.
         */
        virtual BitSet EventDataTypes() = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Event Observer Definitions
    //////////////////////////////////////////////////////////////////////////

    //////////////////////////////////////////////////////////////////////////
    // New Method Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines New Method event data
     *
     * Defines New Method event data for the 
     * INewMethodEventObserver::HandleEvent callback function.
     */
    struct SNewMethodEventData
    {
        TId methodId;       //!< Unique id of the new method
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface INewMethodEventObserver
     * @brief Observer interface for receiving New Method events.
     *
     * The New Method event is sent when a new method of the profiled application is
     * encountered. The event is sent for each method reported to the client in an event 
     * callback or a data request result. Therefore, it can be used for a "one-time" query
     * of method information.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     *
     * @remark The event may be sent on any thread, and may also arrive after the method
     * was reported in an event callback or a data request result.
     */
    class INewMethodEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_NEW_METHOD; }

        virtual BitSet EventDataTypes() { return DR_METHOD_ID; }

        /**
         * @brief Event Handler
         *
         * New Method event handler
         *
         * @param[in] data      event data
         */
        virtual void HandleEvent(SNewMethodEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Method Enter Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Method Enter event data
     *
     * Defines Method Enter event data for the 
     * IMethodEnterEventObserver::HandleEvent callback function.
     */
    struct SMethodEnterEventData
    {
        TId methodId;               //!< Method id
        TId threadId;               //!< @brief Id of the thread that entered the method
                                    //!  Unique for the lifetime of the profiled application
        TId moduleId;               //!< Module id (.NET only)
        TTimeStamp uiElapsedNanos;  //!< Elapsed time since thread start, in nanoseconds
        TTimeStamp uiCpuNanos;      //!< CPU time utilized by the thread, in nanoseconds
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IMethodEnterEventObserver
     * @brief Observer interface for receiving Method Enter events.
     *
     * The Method Enter event is sent just after the execution enters a method.
     *
     * The event is sent on the thread which is executing the method
     *
     * The event supports the following data items:
     * - DR_METHOD_ID
     * - DR_THREAD_ID
     * - DR_MODULE_ID
     * - DR_THREAD_ELAPSED_TIME
     * - DR_THREAD_CPU_TIME
     */
    class IMethodEnterEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_METHOD_ENTER; }

        virtual BitSet EventDataTypes() { return DR_METHOD_ID; }

        /**
         * @brief Event Handler
         *
         * Method Enter event handler
         * 
         * @param[in]   data    event data. The 'validData' field indicates which members
         *                      of the structure are valid.
         */
        virtual void HandleEvent(SMethodEnterEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Method Leave Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Method Leave event data
     *
     * Defines Method Leave event data for the 
     * IMethodLeaveEventObserver::HandleEvent callback function.
     */
    struct SMethodLeaveEventData
    {
        TId methodId;               //!< Method id (Java-only)
        TId threadId;               //!< @brief Id of the thread that left the method
                                    //!  Unique for the lifetime of the profiled application
        TTimeStamp uiElapsedNanos;  //!< Elapsed time since thread start, in nanoseconds
        TTimeStamp uiCpuNanos;      //!< CPU time utilized by the thread, in nanoseconds
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IMethodLeaveEventObserver
     * @brief Observer interface for receiving Method Leave events.
     *
     * The Method Leave event is sent just before the execution is about to leave a method.
     * The event is sent upon normal exit or upon abnormal exit (exception).
     *
     * The event is sent on the thread which is executing the method.
     *
     * The event is guaranteed to arrive in order, meaning it will always arrive after the 
     * Method Entry event for the same method. Unless this event is disabled 
     * during execution, the MPI client can safely assume it will receive this event for 
     * each Method Entry event it received.
     *
     * The event supports the following data items:
     * - DR_METHOD_ID
     * - DR_THREAD_ID
     * - DR_THREAD_ELAPSED_TIME
     * - DR_THREAD_CPU_TIME
     */
    class IMethodLeaveEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_METHOD_LEAVE; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Method Leave event handler
         *
         * @param[in]   data    event data. The 'validData' field indicates which members
         *                      of the structure are valid.
         */
        virtual void HandleEvent(SMethodLeaveEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // JITted Method Loaded Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines JITted Method Loaded event data
     *
     * Defines JITted Method Loaded event data for the 
     * IJittedMethodLoadedEventObserver::HandleEvent callback function.
     */
    struct SJittedMethodLoadedEventData
    {
        TId methodId;                               //!< Id of the JITted method
        SLineNumberTable nativeToSrcLineMap;        //!< @brief
                                                    //!  Maps the JITted (native) instructions
                                                    //!  to source line numbers
        SLineNumberTable nativeToManagedLineMap;    //!< @brief
                                                    //!  Maps the JITted (native) instructions
                                                    //!  to managed code (Java byte-code or.NET
                                                    //!  IL) offsets
        SLineNumberTable managedToSrcLineMap;       //!< @brief
                                                    //!  Maps the managed code to source line
                                                    //!  numbers
        SCodeInfo codeInfo;                         //!< Compiled method code
        SDotNetModulePrejit modulePrejitInfo;       //!< Module pre-JIT info (.NET only)
        BitSet validData;                           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IJittedMethodLoadedEventObserver
     * @brief Observer interface for receiving JITted Method Loaded events.
     *
     * The JITted Method Loaded event is sent after a method is JIT-compiled.
     * The compilation information (e.g., native code) is available at the time of
     * the event.
     *
     * The event is not neccessarily sent on the thread executing the method. The event
     * may be sent after a method has started to execute (i.e., after the Method Enter
     * event).
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     *
     */
    class IJittedMethodLoadedEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_JITTED_METHOD_LOADED; }

        virtual BitSet EventDataTypes()
        {
            return DR_METHOD_ID | DR_NATIVE_TO_MANAGED_LINE_MAP | 
                DR_MANAGED_TO_SRC_LINE_MAP | DR_NATIVE_TO_SRC_LINE_MAP | DR_METHOD_CODE 
                | DR_DOTNET_MODULE_PREJIT;
        }

        /**
         * @brief Event Handler
         *
         * JITted Method Loaded event handler
         *
         * @param[in]   data    event data. The 'validData' field indicates which members
         *                      of the structure are valid.
         */
        virtual void HandleEvent(SJittedMethodLoadedEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // JITted Method Unloaded Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines JITted Method Unloaded event data
     *
     * Defines JITted Method Unloaded event data for the 
     * IJittedMethodUnloadedEventObserver::HandleEvent callback function.
     */
    struct SJittedMethodUnloadedEventData
    {
        TId methodId;               //!< Id of the unloaded method
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IJittedMethodUnloadedEventObserver
     * @brief Observer interface for receiving JITted Method Unloaded events.
     *
     * The JITted Method Unloaded event is sent when a compiled method is unloaded
     * from memory.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     *
     */
    class IJittedMethodUnloadedEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_JITTED_METHOD_UNLOADED; }

        virtual BitSet EventDataTypes() { return DR_METHOD_ID; }

        /**
         * @brief Event Handler
         *
         * JITted Method Unloaded event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SJittedMethodUnloadedEventData &data) = 0;
    };


    //////////////////////////////////////////////////////////////////////////
    // Thread Events and Data
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Thread events data
     *
     * Defines Thread Start and Thread End events data for the 
     * IThreadStartEventObserver::HandleEvent and IThreadEndEventObserver::HandleEvent
     * callback functions.
     */
    struct SThreadEventData
    {
        TId threadId;       //!< Thread id. Unique for the lifetime of the profiled application
        TId objectId;       //!< Id of the thread's object (Java 5.0+ only)
        TId classId;        //!< Id of the thread's class (Java 5.0+ only)
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IThreadStartEventObserver
     * @brief Observer interface for receiving Thread Start events.
     *
     * The Thread Start event is sent when a thread is started. The event is
     * guaranteed to be the first event sent for the started thread. If thread information 
     * is not available when the event is sent (in this case, a Data Request for thread 
     * information will fail and return no data), the event will be sent again when the 
     * thread information becomes available.
     *
     * In Java 5.0, the event is sent on the newly started thread. In Java 1.4, this is 
     * not guaranteed.
     *
     * The event supports the following data items:
     * - DR_THREAD_ID : unique thread id
     * - DR_OBJECT_ID : id of the thread's object (Java 5.0+ only)
     * - DR_CLASS_ID  : id of the thread's class (Java 5.0+ only)
     */
    class IThreadStartEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_THREAD_START; }

        virtual BitSet EventDataTypes() { return DR_THREAD_ID; }

        /**
         * @brief Event Handler
         *
         * Thread Start event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SThreadEventData &data) = 0;
    };

    /**
     * @interface IThreadEndEventObserver
     * @brief Observer interface for receiving Thread End events.
     *
     * The Thread End event is sent when a thread ends. The event is guaranteed to be the 
     * last event sent for the thread. In some VMs, the event will not be sent for threads
     * that were implicitly terminated when the application terminated.
     *
     * The event is sent on the dying thread.
     *
     * The event supports the following data items:
     * - DR_THREAD_ID : unique thread id
     * - DR_OBJECT_ID : id of the thread's object (Java only)
     * - DR_CLASS_ID  : id of the thread's class (Java only)
     */
    class IThreadEndEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_THREAD_END; }

        virtual BitSet EventDataTypes() { return DR_THREAD_ID; }

        /**
         * @brief Event Handler
         *
         * Thread End event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SThreadEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Monitor Events and Data
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Monitor Wait event data
     *
     * Defines Monitor Wait event data for the IMonitorWaitEventObserver::HandleEvent
     * callback function.
     */
    struct SMonitorWaitEventData
    {
        TId threadId;       //!< Id of the waiting thread
        TId objectId;       //!< Id of the monitor object
        U64 timeoutMillis;  //!< The number of milliseconds the thread will wait
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IMonitorWaitEventObserver
     * @brief Observer interface for receiving Monitor Wait events.
     *
     * The Monitor Wait event is sent when a thread is waiting on a monitor as a result
     * of issuing a "wait" command (e.g., anObject.wait() in Java).
     *
     * The event is currently supported only in Java 5.0 and later.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IMonitorWaitEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_MONITOR_WAIT; }

        virtual BitSet EventDataTypes() 
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_MONITOR_TIMEOUT;
        }

        /**
         * @brief Event Handler
         *
         * Monitor Wait event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SMonitorWaitEventData &data) = 0;
    };

    /**
     * @brief Defines Monitor Waited event data
     *
     * Defines Monitor Waited event data for the IMonitorWaitedEventObserver::HandleEvent
     * callback function.
     */
    struct SMonitorWaitedEventData
    {
        TId threadId;       //!< Id of the waiting thread
        TId objectId;       //!< Id of the monitor object
        bool isTimedOut;    //!< Whether the wait operation timed out (true) or not (false)
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IMonitorWaitedEventObserver
     * @brief Observer interface for receiving Monitor Waited events.
     *
     * The Monitor Waited event is sent when a thread finishes waiting on a monitor.
     *
     * The event is currently supported only in Java 5.0 and later.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IMonitorWaitedEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_MONITOR_WAITED; }

        virtual BitSet EventDataTypes()
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_MONITOR_TIMED_OUT;
        }

        /**
         * @brief Event Handler
         *
         * Monitor Waited event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SMonitorWaitedEventData &data) = 0;
    };

    /**
     * @brief Defines Contended Monitor Enter event data
     *
     * Defines Contended Monitor Enter event data for the 
     * IContendedMonitorEnterEventObserver::HandleEvent callback function.
     */
    struct SContendedMonitorEnterEventData
    {
        TId threadId;       //!< Id of the thread attempting to enter the monitor
        TId objectId;       //!< Id of the monitor object
        TId ownerThreadId;  //!< Id of the thread currently owning the monitor, or 0 if the 
                            //!  monitor is not owned
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IContendedMonitorEnterEventObserver
     * @brief Observer interface for receiving Contended Monitor Enter events.
     *
     * The Contended Monitor Enter event is sent when a thread is attempting to enter
     * a monitor already acquired by another thread.
     *
     * The event is currently supported only in Java 5.0 and later.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IContendedMonitorEnterEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_CONTENDED_MONITOR_ENTER; }

        virtual BitSet EventDataTypes() 
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_MONITOR_OWNER_THREAD_ID; 
        }

        /**
         * @brief Event Handler
         *
         * Contended Monitor Enter event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SContendedMonitorEnterEventData &data) = 0;
    };

    /**
     * @brief Defines Contended Monitor Entered event data
     *
     * Defines Contended Monitor Entered event data for the 
     * IContendedMonitorEnteredEventObserver::HandleEvent callback function.
     */
    struct SContendedMonitorEnteredEventData
    {
        TId threadId;       //!< Id of the thread attempting to enter the monitor
        TId objectId;       //!< Id of the monitor object
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IContendedMonitorEnteredEventObserver
     * @brief Observer interface for receiving Contended Monitor Entered events.
     *
     * The Contended Monitor Entered event is sent when a thread enters a monitor after 
     * waiting for it to be released by another thread.
     *
     * The event is currently supported only in Java 5.0 and later.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IContendedMonitorEnteredEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_CONTENDED_MONITOR_ENTERED; }

        virtual BitSet EventDataTypes() { return DR_THREAD_ID | DR_OBJECT_ID; }

        /**
         * @brief Event Handler
         *
         * Contended Monitor Entered event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SContendedMonitorEnteredEventData &data) = 0;
    };

    /**
     * @brief Defines Thread Interaction types
     *
     * Defines Thread Interaction types
     */
    enum EThreadInteractionType
    {
        IT_NOTIFY_ALL = 1,  //!< @brief A thread generated a notification on a monitor to 
                            //!  wake up all waiting threads
        IT_NOTIFY = 2,      //!< @brief A thread generated a notification on a monitor to
                            //!  wake up a single thread
        IT_INTERRUPT = 3,   //!< @brief A thread interrupted the execution of another thread
        IT_START = 4,        //!< @brief A thread started a new thread
        IT_SLEEP_START = 5,	//!< @brief A thread started sleeping
        IT_SLEEP_END = 6	//!< @brief A thread finished sleeping
    };

    /**
     * @brief Defines Thread Interaction event data
     *
     * Defines Thread Interaction event data for the 
     * IThreadInteractionEventObserver::HandleEvent callback function.
     */
    struct SThreadInteractionEventData
    {
        TId threadId;                           //!< @brief Id of the thread performing 
                                                //!  the interaction
        TId objectId;                           //!< @brief ID of the object on which the 
                                                //!  the interaction is performed
        EThreadInteractionType interactionType; //!< The type of the interaction
        BitSet validData;                       //!< Bit-mask of the valid data in the struct
    };
    
    /**
     * @interface IThreadInteractionEventObserver
     * @brief Observer interface for receiving Thread Interaction events.
     *
     * The Thread Interaction event is sent when a thread is about to perform any of the 
     * following operations:
     * - Wake-up (notify) all threads waiting on a monitor object
     * - Wake-up (notify) a single thread waiting on a monitor object
     * - Interrupt the execution of another thread
     * - Start a newly created thread.
     *
     * The Thread Interaction event is always sent on the thread which performs the operation.
     * The event data provides information about the thread, the object it interacts with 
     * and the type of the interaction.
     *
     * The event is currently supported only in Java 5.0 and later.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IThreadInteractionEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_THREAD_INTERACTION; }

        virtual BitSet EventDataTypes()
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_THREAD_INTERACTION_TYPE;
        }

        /**
         * @brief Event Handler
         *
         * Thread Interaction event handler
         *
         * @param[in]   data event data.
         */
        virtual void HandleEvent(SThreadInteractionEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Heap Events and Data
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Heap events data
     *
     * Defines Object Alloc and Object Free events data for the 
     * IObjectAllocEventObserver::HandleEvent and IObjectFreeEventObserver::HandleEvent
     * callback functions.
     */
    struct SHeapEventData
    {
        TId threadId;               //!< @brief Id of the tread that allocated/deallocated 
                                    //!  the object. Unique for the lifetime of the application
        TId objectId;               //!< Object id. Unique for the lifetime of the application
        TId allocMethodId;          //!< Method in which the object was allocated. -1 if unknown
        U64 allocLocation;          //!< @brief Location within the method where the object 
                                    //!  was allocated. -1 If unknown of if the method is native
        SObjectInfo *pObjectInfo;   //!< Additional object information
        U32 objectAge;              //!< @brief The number of Garbage Collection cycles 
                                    //!  the object survived
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IObjectAllocEventObserver
     * @brief Observer interface for receiving Object Alloc events.
     *
     * The Object Alloc event is sent just after a new object was allocated on the 
     * managed heap. At the time of the event, the object is fully initialized and can be 
     * queried for additional information.
     *
     * The event is sent on the thread which allocated the new object.
     *
     * The event supports the following data items:
     * - DR_THREAD_ID : id of the thread allocating the object
     * - DR_OBJECT_ID : id of the newly allocated object
     * - DR_OBJECT_INFO : additional object information (size, class)
     * - DR_METHOD_ID : id of the method in which the object was allocated
     * - DR_ALLOC_VM_INSTRUCTION_OFFSET : the allocation instruction index
     * - DR_OBJECT_AGE : the number of Garbage Collection cycles the object survived.
     *                   this value is always set to 0 for Object Alloc events.
     *                   maximum age of the object is reported in the Object Free event
     */
    class IObjectAllocEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_OBJECT_ALLOC; }

        virtual BitSet EventDataTypes() 
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_OBJECT_INFO; 
        }

        /**
         * @brief Event Handler
         *
         * Object Alloc event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SHeapEventData &data) = 0;
    };

    /**
     * @interface IObjectFreeEventObserver
     * @brief Observer interface for receiving Object Free events.
     *
     * The Object Free event is sent when the garbage collector frees an object.
     *
     * After an object is deallocated, its id is not longer valid and cannot be used
     * in a data request.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method. Note, however, that only data items requested with the 
     * Object Allocated event will be available to this event observer. In other words, 
     * data items specified with this observer that are not specified in the Object Allocated
     * event observer will be ignored.
     *
     * @remark - In Java, only the IMpi::GetClassInfo data request can be used in the event
     *           handler. Other data requests may cause the JVM to crash.
     */
    class IObjectFreeEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_OBJECT_FREE; }

        virtual BitSet EventDataTypes() 
        {
            return DR_THREAD_ID | DR_OBJECT_ID | DR_OBJECT_INFO | DR_OBJECT_AGE;
        }

        /**
         * @brief Event Handler
         *
         * Object Alloc event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SHeapEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Garbage Collection Events
    //////////////////////////////////////////////////////////////////////////

    /**
     * @interface IGcStartEventObserver
     * @brief Observer interface for receiving Garbage Collection Start events
     *
     * The Garbage Collection Start event is sent when a garbage collection begins.
     *
     * @remark - In Java 5.0 and above, only stop-the-world collections are reported, 
     * i.e., collections during which all threads cease to modify the state of the 
     * JVM. This means that some collectors will never generate these events.
     */
    class IGcStartEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_GC_START; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Garbage Collection Start event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @interface IGcEndEventObserver
     * @brief Observer interface for receiving Garbage Collection Finish events
     *
     * The Garbage Collection End event is sent when a garbage collection cycle ends.
     *
     * @remark - In Java 5.0 and above, only stop-the-world collections are reported, 
     * i.e., collections during which all threads cease to modify the state of the 
     * JVM. This means that some collectors will never generate these events.
     */
    class IGcEndEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_GC_END; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Garbage Collection End event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @brief Defines Garbage Collection Moved References event data
     *
     * Defines Garbage Collection Moved References event data for the 
     * IGcMovedReferencesEventObserver::HandleEvent callback function.
     */
    struct SGcMovedRefEventData
    {
        SGgMovedRefInfo gcMovedReferences;  //!< .NET GC Moved References information
        BitSet validData;                   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IGcMovedReferencesEventObserver
     * @brief Observer interface for receiving Garbage Collection Moved References events
     * (.NET only)
     *
     * The Garbage Collection Moved References event is sent when objects are relocated on
     * the managed heap during a Garbage Collection cycle.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     *
     * @remark - The event is supported only in .NET.
     */
    class IGcMovedReferencesEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_GC_MOVED_REFERENCES; }

        virtual BitSet EventDataTypes() { return DR_DOTNET_GC_MOVED_REF_INFO; }

        /**
         * @brief Event Handler
         *
         * Garbage Collection Moved References event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SGcMovedRefEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // VM Init Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines VM Init event data
     *
     * Defines VM Init event data for the 
     * IVmInitEventObserver::HandleEvent callback function.
     */
    struct SVmInitEventData
    {
        TId threadId;       //!< Id of the thread dispatching the event. On most Java VMs, 
                            //!  this is the main thread of the application.
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IVmInitEventObserver
     * @brief Observer interface for receiving VM Init events.
     *
     * The VM Init event is sent when the VM has completed its initialization.
     *
     * Once this event is generated, the client is free to enable or disable events
     * to which it registered during the initialization phase.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IVmInitEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_VM_INIT; }
        
        virtual BitSet EventDataTypes() { return DR_THREAD_ID; }

        /**
         * @brief Event Handler
         *
         * VM Init event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SVmInitEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // VM Shutdown Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @interface IVmShutdownEventObserver
     * @brief Observer interface for receiving VM Shutdown events.
     * 
     * The VM Shutdown event is sent when the VM is about to shut down. No events will be sent
     * after this event.
     *
     * The event will be sent only if the VM was successfully initialized. It will not be sent
     * in case of a VM start-up failure.
     */
    class IVmShutdownEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_VM_SHUTDOWN; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * VM Shutdown event handler
         */
        virtual void HandleEvent() = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // .NET Method Instrumentation Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines .NET Method Instrumentation event data
     *
     * Defines NET Method Instrumentation event data for the 
     * IDotNetMethodInstrumentationPointEventObserver::HandleEvent callback function.
     */
    struct SDotNetMethodInstrEventData
    {
        TId methodId;       //!< Id of the method to instrument
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IDotNetMethodInstrumentationPointEventObserver
     * @brief Observer interface for receiving .NET Method Instrumentation Point events
     * (.NET only)
     *
     * The .NET Method Instrumentation Point event is sent when a JIT compilation of a 
     * .NET method starts. A client can use this event as a hook for instrumenting
     * the method's code.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IDotNetMethodInstrumentationPointEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_DOTNET_METHOD_INSTRUMENTATION_POINT; }

        virtual BitSet EventDataTypes() { return DR_METHOD_ID; }

        /**
         * @brief Event Handler
         *
         * .NET Method Instrumentation Point event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SDotNetMethodInstrEventData &data) = 0;
    };


    //////////////////////////////////////////////////////////////////////////
    // .NET Module Events
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines .NET Module events data
     *
     * Defines NET Module events data for the 
     * IDotNetModuleUnloadStartEventObserver::HandleEvent and the
     * IDotNetModuleLoadFinishEventObserver::HandleEvent callback functions.
     */
    struct SDotNetModuleEventData
    {
        TId moduleId;       //!< .NET module id
        BitSet validData;   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IDotNetModuleUnloadStartEventObserver
     * @brief Observer interface for receiving .NET Module Unload Started events
     * (.NET only)
     *
     * The .NET Module Unload Started event is sent when a module unload operation starts.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IDotNetModuleUnloadStartEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_DOTNET_MODULE_UNLOAD_STARTED; }

        virtual BitSet EventDataTypes() { return DR_MODULE_ID; }

        /**
         * @brief Event Handler
         *
         * .NET Module Unload Started event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SDotNetModuleEventData &data) = 0;
    };

    /**
     * @interface IDotNetModuleLoadFinishEventObserver
     * @brief Observer interface for receiving .NET Module Load Finished events
     * (.NET only)
     *
     * The .NET Module Load Finished event is sent after a .NET module was loaded.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IDotNetModuleLoadFinishEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_DOTNET_MODULE_LOAD_FINISHED; }

        virtual BitSet EventDataTypes() { return DR_MODULE_ID; }

        /**
         * @brief Event Handler
         *
         * .NET Module Load Finished event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SDotNetModuleEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Java Dynamic Code Generated Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Java Dynamic Code Generated event data
     *
     * Defines Java Dynamic Code Generated event data for the 
     * IJavaDynamicCodeGeneratedEventObserver::HandleEvent callback function.
     */
    struct SJavaDynamicCodeEventData
    {
        SJavaDynamicCode jvmDynamicCode;    //!< Dynamic code information
        BitSet validData;                   //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IJavaDynamicCodeGeneratedEventObserver
     * @brief Observer interface for receiving Java Dynamic Code Generated events
     * (Java only)
     *
     * The Java Dynamic Code Generated event is sent when the JVM generates code 
     * on-the-fly for its own use.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IJavaDynamicCodeGeneratedEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_JAVA_DYNAMIC_CODE_GENERATED; }

        virtual BitSet EventDataTypes() { return DR_JAVA_DYNAMIC_CODE; }

        /**
         * @brief Event Handler
         *
         * Java Dynamic Code Generated event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SJavaDynamicCodeEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Java Class File Load Hook (raw) Event
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Java Class File Load Hook event data
     *
     * Defines Java Class File Load Hook event data for the 
     * IJavaClassFileLoadHookEventObserver::HandleEvent callback function.
     * 
     */
    struct SClassFileLoadHookEventData
    {
        const char *className;          //!< @brief Class name. May be null if the name
                                        //!  is unknown
        const unsigned char* classData; //!< Class file data buffer
        S32 classDataLength;            //!< Class file data buffer length
    };

    
    /**
     * @interface IJavaClassFileLoadHookEventObserver
     * @brief Observer interface for receiving Java Class File Load Hook events
     * (Java 5.0+ only)
     *
     * The Java Class File Load Hook event is sent when the JVM obtains a class file data,
     * but before it constructs the in-memory representation of that class. An MPI Client
     * can instrument the existing class file buffer to include profiling/debugging hooks.
     * The client can use the IMpi::GetInstrumentor function to obtain a handle to the 
     * Martini Instrumentation Engine interface.
     */
    class IJavaClassFileLoadHookEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_JAVA_CLASS_FILE_LOAD_HOOK; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Java (raw) Class File Load Hook event handler
         *
         * @param[in]   data            Event data. Contains the raw bytes of the class to 
         *                              instrument
         * @param[in]   memAllocFunc    A memory allocation function for allocating
         *                              memory for the instrumented class file
         * @param[out]  newClassData    Pointer to the instrumented class file buffer.
         *                              Memory for this buffer must be allocated using the 
         *                              @c memAllocFunc function supplied in this event handler.
         *                              If the MPI Client does not wish to instrument the class
         *                              it should not set this argument.
         * @param[out] newClassDataLength  Pointer to the instrumented class buffer length. If 
         *                                 the MPI Client modifies the class buffer, it must set
         *                                 this argument to the length of the new (instrumented)
         *                                 class buffer.
         */
        virtual void HandleEvent(SClassFileLoadHookEventData &data,
                                 TMemoryAllocatorFunc memAllocFunc,
                                 unsigned char **newClassData,
                                 S32 *newClassDataLength) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // External Control Events
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines Set Output Directory event data
     *
     * Defines event data for the 
     * IEcSetOutputDirEventObserver::HandleEvent callback function.
     */
    struct SSetOutputDirEventData
    {
        const char *szActivityPath;     //!< activity path 
        BitSet validData;               //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IEcSetOutputDirEventObserver
     * @brief Observer interface for receiving Set Output Directory events from 
     * External Controllers
     *
     * The Set Output Directory event is sent when the output file destination is changed by 
     * the loaded External Control module. The directory specified in the event data
     * will be used by the Martini runtime for writing log messages.
     *
     * The data items supported by this event are specified by the default implementation of 
     * the EventDataTypes method
     */
    class IEcSetOutputDirEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_SET_OUTPUT_DIRECTORY; }

        virtual BitSet EventDataTypes() { return DR_ACTIVITY_PATH; }

        /**
         * @brief Event Handler
         *
         * Set Output Directory event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SSetOutputDirEventData &data) = 0;
    };

    /**
     * @interface IEcStartEventObserver
     * @brief Observer interface for receiving Start events from 
     * External Controllers
     *
     * The Start event is sent when a start notification is received from the loaded 
     * External Control module.
     * In response to this event, the MPI Client is expected
     * to start collecting profiling data.
     */
    class IEcStartEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_START; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Start event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @interface IEcStopEventObserver
     * @brief Observer interface for receiving Stop events from 
     * External Controllers
     *
     * The Stop event is sent when a stop notification is received from the loaded 
     * External Control module.
     * In response to this event, the MPI Client is expected
     * to stop collecting profiling data.
     */
    class IEcStopEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_STOP; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Stop event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @interface IEcAttachEventObserver
     * @brief Observer interface for receiving Attach events from 
     * External Controllers
     *
     * The Attach event is sent when an attach notification is received from the loaded 
     * External Control module.
     * In response to this event, the MPI Client is expected
     * to attach itself to the profiled application.
     */
    class IEcAttachEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_ATTACH; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Attach event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @interface IEcDetachEventObserver
     * @brief Observer interface for receiving Detach events from 
     * External Controllers
     *
     * The Detach event is sent when a detach notification is received from the loaded 
     * External Control module.
     * In response to this event, the MPI Client is expected
     * to detach itself from the profiled application.
     */
    class IEcDetachEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_DETACH; }

        virtual BitSet EventDataTypes() { return DR_NONE; }

        /**
         * @brief Event Handler
         *
         * Detach event handler
         */
        virtual void HandleEvent() = 0;
    };

    /**
     * @brief Defines Custom Command event data
     *
     * Defines Custom Command event data for the 
     * IEcCustomCommandEventObserver::HandleEvent callback function.
     */
    struct SCustomCommandEventData
    {
        unsigned int commandId;     //!< An id identifying the command
        void *pData;                //!< Additional command data
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @interface IEcCustomCommandEventObserver
     * @brief Observer interface for receiving Custom Command events from 
     * External Controllers
     *
     * The Custom Command event is used by the External Control module to pass a command
     * to the MPI client. The meaning of the command is specific to the implementation of the
     * MPI client and the External Control module.
     */
    class IEcCustomCommandEventObserver : public IEventObserver
    {
    public:
        virtual TEventType Type() { return EV_EC_CUSTOM_COMMAND; }

        virtual BitSet EventDataTypes() { return DR_COMMAND_ID | DR_COMMAND_DATA; }

        /**
         * @brief Event Handler
         *
         * Custom Command event handler
         *
         * @param[in]   data    event data.
         */
        virtual void HandleEvent(SCustomCommandEventData &data) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Call Graph Event Filter
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines data for the Call Graph events filter
     *
     * Defines data for the Call Graph events filter
     */
    struct SCallGraphFilterData
    {   
        const char *szModuleName;           //!< Module name (.NET only)
        const char *szClassName;            //!< Fully qualified class name
        const char *szMethodName;           //!< Method name
        const char *szMethodSignature;      //!< Method signature
        EMethodAccessType methodAccessType; //!< Method access type (visibility)
    };

    /**
     * @interface ICallGraphFilter
     * @brief Event Filter interface for defining selectivity for Call Graph events
     *
     * Event Filter interface for defining selectivity for the following
     * events: New Method, Method Enter and Method Leave.
     *
     * To implement selectivity, the client defines a filter class which implements this 
     * interface, and then registers an instance of this class with MPI using the 
     * IMpi::SetEventGroupFilter method.
     *
     * The MPI implementation will call the ShouldNotify method of the filter class for each
     * new method it encounters, in order to determine whether to generate Call Graph events
     * for this method.
     */
    class ICallGraphFilter : public IEventFilter
    {
    public:
        virtual EFilterType Type() { return FT_CALLGRAPH; }

        /**
         * @brief Invoked to determine event selectivity for a method
         *
         * This method is implemented by the filter class and called by the MPI implementation
         * for each method that is about to execute to determine whether to report Call Graph 
         * events for the method
         *
         * @param[in]   methodInfo  method information
         *
         * @retval      true        an implementation returns true to indicate that events
         *                          should be reported for this method
         *
         * @retval      false       an implementation returns false to indicate that events
         *                          should not be reported for this method
         *
         */
        virtual bool ShouldNotify(SCallGraphFilterData &methodInfo) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Heap Events Filter
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines data for the Heap events filter
     *
     * Defines data for the Heap events filter
     */
    struct SHeapFilterData
    {
        const char *szClassName;    //!< Fully qualified class name
    };

    /**
     * @interface IHeapFilter 
     * @brief Event Filter interface for defining selectivity for Heap events
     *
     * Event Filter interface for defining selectivity for the following
     * events: Object Alloc and Object Free
     *
     * To implement selectivity, the client defines a filter class which implements this 
     * interface, and then registers an instance of this class with MPI using the 
     * IMpi::SetEventGroupFilter method.
     *
     * The MPI implementation will call the ShouldNotify method of the filter class for each
     * new class it encounters, in order to determine whether to generate Heap events
     * for this class.
     *
     * @remark - The current implementation for Java calls the filter function for each
     *           allocated object. This will be optimized in future versions.
     */
    class IHeapFilter : public IEventFilter
    {
    public:
        virtual EFilterType Type() { return FT_HEAP; }

        /**
         * @brief Invoked to determine event selectivity for a class
         *
         * This method is implemented by the filter class and called by the MPI implementation
         * to determine whether to report heap events for a class
         *
         * @param[in]   objectInfo  class information
         *
         * @retval      true        an implementation returns true to indicate that events
         *                          should be reported for this class
         *
         * @retval      false       an implementation returns false to indicate that events
         *                          should not be reported for this class
         *
         */
        virtual bool ShouldNotify(SHeapFilterData &objectInfo) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Thread Interaction Events Filter
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Defines data for the Thread Interaction events filter
     *
     * Defines data for the Thread Interaction events filter
     */
    struct SThreadInteractionFilterData
    {
        const char *szClassName;    //!< Fully qualified class name
        const char *szMethodName;   //!< Method name
    };

    /**
     * @interface IThreadInteractionFilter 
     * @brief Event Filter interface for defining selectivity for Thread Interaction events
     *
     * Event Filter interface for defining selectivity for the following
     * events: Thread Interaction
     *
     * To implement selectivity, the client defines a filter class which implements this 
     * interface, and then registers an instance of this class with MPI using the 
     * IMpi::SetEventGroupFilter method.
     *
     * The MPI implementation will call the ShouldNotify method of the filter class for each
     * new method it encounters, in order to determine whether to report Thread
     * Interaction events that occur within the method
     */
    class IThreadInteractionFilter : public IEventFilter
    {
    public:
        virtual EFilterType Type() { return FT_THREAD_INTERACTION; }

        /**
         * @brief Invoked to determine event selectivity for a method
         *
         * This method is implemented by the filter class and called by the MPI implementation
         * for each method that is about to execute to determine whether to report Thread
         * Interaction events that occur within the method
         *
         * @param[in]   methodInfo  method information
         *
         * @retval      true        an implementation returns true to indicate that events
         *                          should be reported for this method
         *
         * @retval      false       an implementation returns false to indicate that events
         *                          should not be reported for this method
         *
         */
        virtual bool ShouldNotify(SThreadInteractionFilterData &methodInfo) = 0;
    };

    //////////////////////////////////////////////////////////////////////////
    // Data Request Structure Definitions
    //////////////////////////////////////////////////////////////////////////
    
    /**
     * @brief Defines method information for the IMpi::GetMethodInfo API
     * 
     * Used with IMpi::GetMethodInfo to request method information.
     * The validData field contains a bit set indicating which of the struct's elements are
     * valid.
     */
    struct SMethodInfo
    {
        const char *szModuleName;                       //!< module name 
        const char *szClassName;                        //!< class name 
        const char *szMethodName;                       //!< method name
        const char *szMethodSig;                        //!< method signature
        const char *szMethodSourceFileName;             //!< method source file name 
        const char *szJavaClassName;                    //!< @brief class name in the native 
                                                        //!  Java VM format
                                                        //!  (e.g., Lpackage/class;)
        const char *szJavaMethodName;                   //!< @brief method name in the 
                                                        //!  native Java VM format  
                                                        //!  (e.g., \<init\> for a constructor)
        const char *szJavaMethodSig;                    //!< @brief native Java VM signature
                                                        //!  (e.g., "[[I" for "int[][]")
        SMethodLineNumbers methodLineNumbers;           //!< method line numbers (compact)
        TId moduleId;                                   //!< module id
        TId classId;                                    //!< class id
        SLineNumberTable nativeToSrcLineMap;            //!< @brief
                                                        //!  Maps the JITted (native) 
                                                        //!  instructions to source line
                                                        //!  numbers
        SLineNumberTable nativeToManagedLineMap;        //!< @brief
                                                        //!  Maps the JITted (native) 
                                                        //!  instructions to managed code 
                                                        //!  (Java byte-code or.NET IL) offsets
        SLineNumberTable managedToSrcLineMap;           //!< @brief
                                                        //!  Maps the managed code to source 
                                                        //!  line numbers
        SCodeInfo codeInfo;                             //!< Compiled method code
        SDotNetModulePrejit modulePrejitInfo;           //!< Module pre-JIT info (.NET only)
        TId methodToken;                                //!< method token (.NET only)
        BitSet validData;                               //!< Bit-mask of the valid data in the struct
    };

    /**
     * @brief Defines class information for the IMpi::GetClassInfo API
     * 
     * Used with IMpi::GetClassInfo to request class information.
     * The validData field contains a bit set indicating which of the struct's elements are
     * valid.
     */
    struct SClassInfo
    {
        const char *szClassName;        //!< class name 
        const char *szSourceFileName;   //!< source file name, or \<Unknown\> if not known
        const char *szJavaClassName;    //!< @brief class name in the native Java VM format
                                        //!  (e.g., Lpackage/class;)
        BitSet validData;               //!< Bit-mask of the valid data in the struct
    };

    /**
     * @brief Defines object information for the IMpi::GetObjectInfo API
     * 
     * Used with IMpi::GetObjectInfo to request object information.
     * The validData field contains a bit set indicating which of the struct's elements are
     * valid.
     */
    struct SObjectDataRequest
    {
        SObjectInfo objectInfo;     //!< Object information
        TId monOwnerThreadId;       //!< @brief Id of the thread owning the object's monitor, 
                                    //!  or 0 if the monitor is unowned
        TIdArray monOwnWaiters;     //!< Array of threads waiting to own this object's monitor
        TIdArray monNotifyWaiters;  //!< @brief Array of threads waiting to be notified by this 
                                    //!  object's monitor
        BitSet validData;           //!< Bit-mask of the valid data in the struct
    };

    /**
     * @brief Defines thread states
     *
     * Enumerator indicating the a thread state. A thread can be in only one state at a given
     * point in time.
     *
     * @remark - In .NET, the TS_SLEEPING and TS_WAITING states are not supported. 
     *           The TS_BLOCKED state is used to identify a .NET thread which is blocked as a 
     *           result of a call to @c Wait, @c Sleep, or @c Join.
     */
    enum EThreadState
    {
        TS_UNKNOWN,         //!< Unknown state
        TS_UNSTARTED,       //!< Thread was created but not yet started
        TS_RUNNING,         //!< Thread is running
        TS_SLEEPING,        //!< Thread is sleeping (Java only)
        TS_BLOCKED,         //!< @brief Thread is blocked. In Java, the thread is waiting for 
                            //!         a monitor lock
        TS_WAITING,         //!< Thread is waiting on a monitor (Java only)
        TS_TERMINATED       //!< Thread has stopped execution
    };
    
    /**
     * @brief Defines thread information for the IMpi::GetThreadInfo API
     *
     * Used with IMpi::GetThreadInfo to request thread information.
     * The validData field contains a bit set indicating which of the struct's elements are
     * valid.
     *
     * @remark - On some virtual machines (such as Java 5.0), thread information is available
     * only after the VM has been fully initialized and the VM Init event has been
     * sent. Requesting thread information before it is available may result in NULL values
     * in some of the fields of SThreadInfo.
     */
    struct SThreadInfo
    {
        const char *szName;             //!< thread name
        const char *szGroupName;        //!< @brief name of the group to which the thread
                                        //!  belongs (Java only)
        const char *szParentGroupName;  //!< @brief name of the parent group of this 
                                        //!  thread's group (Java only)
        SStackTrace nativeAbsoluteStack;//!< @brief stack trace of this thread, using native
                                        //!  code addresses (.NET only)
        SStackTrace vmOffsetStack;      //!< @brief stack trace of this thread, using VM
                                        //!  (managed) instructions relative offsets 
        TTimeStamp uiElapsedNanos;      //!< Elapsed time since thread start, in nanoseconds
        TTimeStamp uiCpuNanos;          //!< CPU time utilized by the thread, in nanoseconds
        EThreadState state;             //!< Thread state (Java only)
        TId threadObjectId;             //!< Thread object id (Java only)
        TId currentMonitorId;           //!< @brief Monitor the thread is currently waiting for, 
                                        //!  of 0 if there is none (Java only)
        BitSet validData;               //!< Bit-mask of the valid data in the struct
    };

    /**
     * @brief callback function for unmanaged stack walk (.NET only)
     *
     * A .NET-specific callback function that informs clients of an unmanaged stack frame
     * encountered during a Stack Trace request (@see IMpi::GetThreadInfo). Multiple
     * consecutive unmanaged frames are reported as a single frame by this function.
     * The @c firstUnmanagedCallSiteIP and @c terminatingManagedCallSiteIP arguments can be 
     * used to determine the start and end IPs of the unmanged frame(s).
     *
     * @param[in,out]   pData   The @c pData argument supplied to @c GetThreadInfo. The client
     *                          can use this to populate the @c pData->nativeAbsoluteStack 
     *                          field with unmanged stack frame information.
     * @param[in]       context         The context (machine registers, etc) of the unmanged 
     *                                  frame.
     *
     * @param[in]       contextSize     The size of the @c context parameter in bytes
     *
     * @param[in]       firstUnmanagedCallSiteIP        IP of the first unmanaged call site.
     * @param[in]       terminatingManagedCallSiteIP    IP of the next managed frame following
     *                                                  this unmanaged frame. May be 0 if not
     *                                                  enough space was allocated for the
     *                                                  stack trace array.
     *
     * @return  The Client should return @c MRTE_RESULT_OK to notify the Martini implementation
     *          to continue stack walk, or any other value (e.g, @c MRTE_ERROR_FAIL) to
     *          terminate the stack walk.
     */
    typedef TResult (*TDotNetUnmanagedStackWalkCallBack)(SThreadInfo *pData, 
                                                         void *context,
                                                         unsigned int contextSize,
                                                         U64 firstUnmanagedCallSiteIP,
                                                         U64 terminatingManagedCallSiteIP);
    /**
     * @brief Defines information for an entry in an SThreadInfoArray
     *
     */
    struct SThreadInfoArrayEntry
    {
        TId threadId;                   //!< Thread id
        SThreadInfo threadInfo;         //!< Thread information
    };
    
    /**
     * @brief Defines thread information array for the IMpi::GetAllThreadsInfo API
     *
     * Used with IMpi::GetAllThreadsInfo to request information for all living threads.
     * Each entry in the @c pEntries array provides information for one living thread.
     *
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     * 
     * @see SThreadInfoArrayEntry
     */
    struct SThreadInfoArray
    {
        SThreadInfoArrayEntry *pEntries;    //!< Array entries
        unsigned int uiSize;                //!< @brief The number of entries allocated by the 
                                            //!  client
        unsigned int uiActualSize;          //!< @brief the actual number of entries available
    };

    /**
     * @brief Defines module information for the IMpi::GetModuleInfo API
     * 
     * Used with IMpi::GetModuleInfo to request module information.
     * The validData field contains a bit set indicating which of the struct's elements are
     * valid.
     */
    struct SModuleInfo
    {
        const char *szModuleName;       //!< module name
        BitSet validData;               //!< Bit-mask of the valid data in the struct
    };

    /**
     * @brief Defines object reference types
     */
    enum EObjectReferenceType
    {
        OR_CLASS = 1,                       //!< Reference from an object to its class
        OR_FIELD = 2,                       //!< @brief Reference from an object to the value
                                            //!  of one of its instance fields
        OR_ARRAY_ELEMENT = 3,               //!< Reference from an array to one of its elements
        OR_JAVA_CLASS_LOADER = 4,           //!< @brief Reference from a class to its 
                                            //!  class loader (Java only)
        OR_JAVA_REFERENCE_SIGNERS = 5,      //!< @brief Reference from a class to its 
                                            //!  signers array (Java only)
        OR_JAVA_PROTECTION_DOMAIN = 6,      //!< @brief Reference from a class to its
                                            //!  protection domain (Java only)
        OR_INTERFACE = 7,                   //!< Reference from a class to one of its interfaces
        OR_STATIC_FIELD = 8,                //!< @brief Reference from a class to the value 
                                            //!  of one of its static fields
        OR_CONSTANT_POOL = 9,               //!< @brief Reference from a class to a resolved 
                                            //!  entry in the constant pool
        OR_SUPERCLASS = 10,                 //!< Reference from a class to its superclass
        OR_JAVA_JNI_GLOBAL = 21,            //!< @brief Heap root reference: JNI global
                                            //!  reference (Java only)
        OR_SYSTEM_CLASS= 22,                //!< Heap root reference: System class
        OR_MONITOR = 23,                    //!< Heap root reference: monitor
        OR_STACK_LOCAL = 24,                //!< Heap root reference: local variable on the stack
        OR_JAVA_JNI_LOCAL = 25,             //!< Heap root reference: JNI local 
                                            //!  reference (Java only)
        OR_THREAD = 26,                     //!< Heap root reference: Thread
        OR_OTHER = 27                       //!< Heap root reference: other heap root reference
    };

    /**
     * @brief Defines information for an object reference
     */
    struct SObjectReference
    {
        EObjectReferenceType type;  //!< Reference type
        TId srcObjectId;            //!< @brief Referrer object. 
                                    //!  If 0, then the referenced object is a heap root
        TId targetObjectId;         //!< Referenced object
    };

    /**
     * @brief Defines object reference information for the IMpi::GetObjectReferences API
     * 
     * Used with IMpi::GetObjectReferences to request information about reachable objects
     * on the managed heap.
     *
     * For more information on using this structure, refer to 
     * @ref mpi_arrays "Working with MPI Arrays".
     */
    struct SObjectReferenceInfo
    {
        SObjectReference *pReferences;  //!< Array of object references
        U64 uiSize;                     //!< @brief The number of entries allocated by 
                                        //!  the client for the @c pReferences array
        U64 uiActualSize;               //!< @brief the actual number of entries available
                                        //!  after the call to IMpi::GetObjectReferences
                                        //!  returns
    };

    //////////////////////////////////////////////////////////////////////////
    // MPI API
    //////////////////////////////////////////////////////////////////////////

    /**
     * @nosubgrouping
     *
     * @interface IMpi
     * @brief Martini Profiling Interface functions
     *
     * Defines the Martini Profiling Interface API.
     */
    class IMpi
    {
    public:

        /**
         * @name Event Management Functions
         */

        //@{

        /**
         * @brief Registers for an MPI event
         *
         * This function registers an Event Observer for handling an MPI event.
         * If registration completes successfully, the MPI client will
         * be notified when the event occurs via the Observer object's HandleEvent() function, 
         * with the requested data as an argument.
         *
         * Registering Event Observers is only possible during initialization phase. The 
         * client should register for all needed events within its MPICL_Instantiate()  
         * function.
         *
         * An event for which an Observer is registered is automatically enabled. The client
         * can optionally disable event notifications for specific events using the 
         * DisableEventGroup function. If an event is disabled, the client will not be notified
         * when the event occurs. To get notifications for such events, the client must
         * explicitly enable them using the EnableEventGroup function.
         *
         * @param[in] clientId    The id of the client registering for the event
         * @param[in] observer    The Event Observer object to be registered. The Observer 
         *                        object defines the data items that should be communicated 
         *                        with the event, and the callback function 
         *                        to be called when the event occurs
         *
         * @retval MRTE_RESULT_OK               The Event Observer was successfully registered
         * @retval MRTE_ERROR_NOT_SUPPORTED     The event or its data item(s) are not supported
         *                                      by the MPI implementation
         * @retval MRTE_ERROR_ILLEGAL_ID        clientId is not a valid MPI client id
         * @retval MRTE_ERROR_CONFLICT          Another client has already registered to this 
         *                                      event and defined a selectivity filter for it.
         *                                      This limitation may be lifted in future 
         *                                      versions.
         * @retval MRTE_ERROR_OUT_OF_MEMORY     Out of memory
         * @retval MRTE_ERROR_PHASE_FAILURE     Can't register for an event after 
         *                                      initialization has completed
         *
         * @see EnableEventGroup
         * @see DisableEventGroup
         *
         **/
        virtual TResult RegisterEvent(TId clientId, 
                                      IEventObserver &observer) = 0;
        
        /**
         * @brief Applies a filter to a group of events
         *
         * Called by the MPI client to apply a filter to a group of events.
         * Applying a filter to an event group turns-on selectivity for all events in that
         * group. The MPI implementation will invoke the @c ShouldNotify callback of the filter
         * object for each element relevant to the group.
         *
         * Applying filters is only possible during the initialization phase. The 
         * client should apply all needed filters within its @c MPICL_Instantiate()  
         * function.
         *
         * Only one filter can be applied to each group. Applying a filter to a group that 
         * is already associated with another filter will return an error.
         *
         * The type of the filter must match the group to which it applies, or the operation
         * will fail. The following table summarizes the filter types that can be applied
         * to each event group:
         * 
         * <table>
         * <tr><td> <b>Event Group</b>   </td>   <td> <b>Supported Filter Types</b> </td></tr>
         * <tr><td> EG_CALLGRAPH         </td>   <td> ICallGraphFilter              </td></tr>
         * <tr><td> EG_HEAP              </td>   <td> IHeapFilter                   </td></tr>
         * <tr><td> EG_THREAD_INTERACTION</td>   <td> IThreadInteractionFilter      </td></tr>
         * </table>  
         *
         * @param[in] clientId      The id of the client setting the filter
         * @param[in] group         The event group to which the filter applies
         * @param[in] filter        The object implementing the filter interface
         *
         * @retval MRTE_RESULT_OK               The filter was successfully registered
         * @retval MRTE_ERROR_NOT_SUPPORTED     The filter is not supported for the specified
         *                                      group
         * @retval MRTE_ERROR_ILLEGAL_ID        clientId is not a valid MPI client id
         *
         * @retval MRTE_ERROR_OUT_OF_MEMORY     Out of memory
         * @retval MRTE_ERROR_PHASE_FAILURE     Can't apply a filter after 
         *                                      initialization has completed
         * @retval MRTE_ERROR_FAIL              An error occured while setting the filter.
         * @retval MRTE_ERROR_CONFLICT          Another client has already defined a filter
         *                                      for the event group. This limitation may be
         *                                      lifted in future versions.
         *
         * @see ICallGraphFilter
         * @see IHeapFilter
         * @see IThreadInteractionFilter
         **/
        virtual TResult SetEventGroupFilter(TId clientId, 
                                            EEventGroup group,
                                            IEventFilter &filter) = 0;

        /**
         * @brief Enables a group of events
         *
         * Called by the MPI client to enable a group of events. After enabling events,
         * Event Observers registered for these events will be notified when the events occur.
         * The client can enable an event group only if it registered to at least one of the
         * events included in that group.
         * Enabling an already enabled event group has no effect (no error is reported).
         * 
         * Events which were disabled during the client initialization phase can be enabled
         * only after the JVM has initialized. Calling this function
         * after the client initialization phase but before the VM has initialized will 
         * return an error. A client can implement and register a VM Init Event Observer
         * (IVmInitEventObserver interface) to get a notification when the VM has been 
         * initialized.
         *
         * This operation is supported for the following event groups:
         * - EEventGroup::EG_CALLGRAPH
         * - EEventGroup::EG_HEAP
         * - EEventGroup::EG_MONITOR
         * - EEventGroup::EG_THREAD_INTERACTION
         *
         * @param[in] clientId      The id of the client enabling the event group
         * @param[in] group         The event group to enable
         * 
         * @retval MRTE_RESULT_OK               The event group was successfully enabled
         * @retval MRTE_ERROR_NOT_SUPPORTED     The operation is not supported for the event
         *                                      group, or for the runtime environment
         *                                      (see Remarks below)
         * @retval MRTE_ERROR_ILLEGAL_ID        clientId is not a valid MPI client id
         * @retval MRTE_ERROR_PHASE_FAILURE     The function was called before the VM
         *                                      has completed its initialization.
         * @retval MRTE_ERROR_FAIL              An error occured while enabling the event 
         *                                      group, such as when the client is not 
         *                                      registered for at least one event in the group
         *                                      it tries to enable.
         *
         * @remark - This operation is supported for Java 5.0 (1.5) and above
         *
         * @remark - Although enabled, some events (such as New Method, Method Enter, 
         *           Method Leave) will not be generated for currently executing methods
         *           (i.e., methods that have an active frame on the Java thread stack).
         *  
         */
        virtual TResult EnableEventGroup(TId clientId, EEventGroup group) = 0;

        /**
         * @brief Disables a group of events
         *
         * Called by the MPI client to disable a group of events. After disabling events,
         * Event Observers registered for these events will be not be notified when the events
         * occur. The client can disable an event group only if it registered to at least  
         * one of the events included in that group.
         *
         * Disabling an already disabled event has no effect (no error is reported).
         * 
         * Events can be disabled immediately after registration (during the initialization)
         * phase or after the JVM has initialized. Calling this function
         * after the client initialization phase but before the VM has initialized will 
         * return an error. A client can implement and register a VM Init Event Observer
         * (IVmInitEventObserver interface) to get a notification when the VM has been 
         * initialized.
         *
         * This operation is supported for the following event groups:
         * - EEventGroup::EG_CALLGRAPH
         * - EEventGroup::EG_HEAP
         * - EEventGroup::EG_MONITOR
         * - EEventGroup::EG_THREAD_INTERACTION
         *
         * @param[in] clientId      The id of the client disabling the event group
         * @param[in] group         The event group to disable
         * 
         * @retval MRTE_RESULT_OK               The event group was successfully disabled
         * @retval MRTE_ERROR_NOT_SUPPORTED     The operation is not supported for the event
         *                                      group, or for the runtime environment
         *                                      (see Remarks below)
         * @retval MRTE_ERROR_ILLEGAL_ID        clientId is not a valid MPI client id
         * @retval MRTE_ERROR_PHASE_FAILURE     The function was called before the VM
         *                                      has completed its initialization.
         * @retval MRTE_ERROR_FAIL              An error occured while disabling the event 
         *                                      group, such as when the client is not 
         *                                      registered for at least one event in the group
         *                                      it tries to enable.
         *
         * @remark - This operation is supported for Java 5.0 (1.5) and above.
         *
         */
        virtual TResult DisableEventGroup(TId clientId, EEventGroup group) = 0;

        //@}

        /**
         * @anchor impi_data_request_group
         * @name Data Request Functions
         */

        //@{

        /**
         * @brief Data Request for Module information
         *
         * Returns information about a module. Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>    </td>   <td> <b>SModuleInfo Target Field</b> </td></tr>
         * <tr><td> DR_MODULE_NAME      </td>   <td> szModuleName                    </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types"
         */
        virtual TResult GetModuleInfo(TId clientId,
                                      TId moduleId,
                                      BitSet requestedDataTypes, 
                                      SModuleInfo *pData) = 0;

        /**
         * @brief Data Request for Class information
         *
         * Returns information about a class. Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>    </td>   <td> <b>SClassInfo Target Field</b>  </td></tr>
         * <tr><td> DR_CLASS_NAME       </td>   <td> szClassName                     </td></tr>
         * <tr><td> DR_SOURCE_FILE_NAME </td>   <td> szSourceFileName                </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types"
         */
        virtual TResult GetClassInfo(TId clientId,
                                     TId classId,
                                     BitSet requestedDataTypes, 
                                     SClassInfo *pData) = 0;

        /**
         * @brief Data Request for Method information
         *
         * Returns information about a method. Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>                    </td> <td> <b>SMethodInfo Target Field</b> </td></tr>
         * <tr><td> DR_METHOD_NAME                      </td> <td> szMethodName                    </td></tr>
         * <tr><td> DR_METHOD_SIGNATURE                 </td> <td> szMethodSig                     </td></tr>
         * <tr><td> DR_JAVA_NATIVE_METHOD_NAME          </td> <td> szJavaMethodName                </td></tr>
         * <tr><td> DR_JAVA_NATIVE_METHOD_SIGNATURE     </td> <td> szJavaMethodSig                 </td></tr>
         * <tr><td> DR_SOURCE_FILE_NAME                 </td> <td> szMethodSourceFileName          </td></tr>
         * <tr><td> DR_METHOD_LINE_NUMBERS              </td> <td> methodLineNumbers               </td></tr>
         * <tr><td> DR_CLASS_ID                         </td> <td> classId                         </td></tr>
         * <tr><td> DR_CLASS_NAME                       </td> <td> szClassName                     </td></tr>
         * <tr><td> DR_MODULE_ID                        </td> <td> moduleId                        </td></tr>
         * <tr><td> DR_MODULE_NAME                      </td> <td> szModuleName                    </td></tr>
         * <tr><td> DR_METHOD_CODE                      </td> <td> codeInfo                        </td></tr>
         * <tr><td> DR_NATIVE_TO_SRC_LINE_MAP           </td> <td> nativeToSrcLineMap              </td></tr>
         * <tr><td> DR_NATIVE_TO_MANAGED_LINE_MAP       </td> <td> nativeToManagedLineMap          </td></tr>
         * <tr><td> DR_MANAGED_TO_SRC_LINE_MAP          </td> <td> managedToSrcLineMap             </td></tr>
         * <tr><td> DR_DOTNET_MODULE_PREJIT             </td> <td> modulePrejitInfo                </td></tr>
         * <tr><td> DR_DOTNET_METHOD_TOKEN              </td> <td> methodToken                     </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types
         *
         * @remark - the following data items can be requested only on .NET:
         *           DR_METHOD_CODE, DR_NATIVE_TO_SRC_LINE_MAP, DR_NATIVE_TO_MANAGED_LINE_MAP,
         *           DR_DOTNET_MODULE_PREJIT, DR_DOTNET_METHOD_TOKEN.
         */
        virtual TResult GetMethodInfo(TId clientId,
                                      TId methodId,
                                      BitSet requestedDataTypes, 
                                      SMethodInfo *pData) = 0;


        /**
         * @brief Data Request for compiled method information (.NET only)
         *
         * Returns information about a compiled method based on a .NET module name and an 
         * IP offset within this module. If the IP is located within a method, the function
         * returns the information requested about the Method.
         * Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>                    </td> <td> <b>SMethodInfo Target Field</b> </td></tr>
         * <tr><td> DR_METHOD_ID                        </td> <td> methodId
         * <tr><td> DR_METHOD_NAME                      </td> <td> szMethodName                    </td></tr>
         * <tr><td> DR_METHOD_SIGNATURE                 </td> <td> szMethodSig                     </td></tr>
         * <tr><td> DR_JAVA_NATIVE_METHOD_NAME          </td> <td> szJavaMethodName                </td></tr>
         * <tr><td> DR_JAVA_NATIVE_METHOD_SIGNATURE     </td> <td> szJavaMethodSig                 </td></tr>
         * <tr><td> DR_SOURCE_FILE_NAME                 </td> <td> szMethodSourceFileName          </td></tr>
         * <tr><td> DR_METHOD_LINE_NUMBERS              </td> <td> methodLineNumbers               </td></tr>
         * <tr><td> DR_CLASS_ID                         </td> <td> classId                         </td></tr>
         * <tr><td> DR_CLASS_NAME                       </td> <td> szClassName                     </td></tr>
         * <tr><td> DR_MODULE_ID                        </td> <td> moduleId                        </td></tr>
         * <tr><td> DR_MODULE_NAME                      </td> <td> szModuleName                    </td></tr>
         * <tr><td> DR_METHOD_CODE                      </td> <td> codeInfo                        </td></tr>
         * <tr><td> DR_NATIVE_TO_SRC_LINE_MAP           </td> <td> nativeToSrcLineMap              </td></tr>
         * <tr><td> DR_NATIVE_TO_MANAGED_LINE_MAP       </td> <td> nativeToManagedLineMap          </td></tr>
         * <tr><td> DR_MANAGED_TO_SRC_LINE_MAP          </td> <td> managedToSrcLineMap             </td></tr>
         * <tr><td> DR_DOTNET_MODULE_PREJIT             </td> <td> modulePrejitInfo                </td></tr>
         * <tr><td> DR_DOTNET_METHOD_METADATA_ID        </td> <td> methodMetadataId                </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types
         *
         * @remark - the following data items can be requested only on .NET:
         *           DR_METHOD_CODE, DR_NATIVE_TO_SRC_LINE_MAP, DR_NATIVE_TO_MANAGED_LINE_MAP,
         *           DR_MANAGED_TO_SRC_LINE_MAP, DR_DOTNET_MODULE_PREJIT,
         *           DR_DOTNET_METHOD_METADATA_ID.
         */
        virtual TResult GetCompiledMethodInfo(TId clientId,
                                              const char* szModuleName,
                                              UIOP uiIpOffset,
                                              BitSet requestedDataTypes,
                                              TId *pMethodId,
                                              SMethodInfo *pData) = 0;

        /**
         * @brief Data Request for Object information
         *
         * Returns information about an object. Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>            </td> <td> <b>SObjectDataRequest Target Field</b> </td></tr>
         * <tr><td> DR_OBJECT_INFO              </td> <td> objectInfo                             </td></tr>
         * <tr><td> DR_MONITOR_OWNER_THREAD_ID  </td> <td> monOwnerThreadId                       </td></tr>
         * <tr><td> DR_MONITOR_OWN_WAITERS      </td> <td> monOwnWaiters                          </td></tr>
         * <tr><td> DR_MONITOR_NOTIFY_WAITERS   </td> <td> monOwnNotifyWaiters                    </td></tr>
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types
         */
        virtual TResult GetObjectInfo(TId clientId,
                                      TId objectId,
                                      BitSet requestedDataTypes,
                                      SObjectDataRequest *pData) = 0;

        /**
         * @brief Data Request for Thread information
         *
         * Returns information about a thread. Refer to the @ref data_request_functions
         * "Data Request Functions" section for usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>                </td> <td> <b>SThreadInfo Target Field</b> </td></tr>
         * <tr><td> DR_THREAD_NAME                  </td> <td> szName                          </td></tr>
         * <tr><td> DR_THREAD_GROUP                 </td> <td> szGroupName (Java only)         </td></tr>
         * <tr><td> DR_THREAD_PARENT_GROUP          </td> <td> szParentGroupName (Java only)   </td></tr>
         * <tr><td> DR_THREAD_ELAPSED_TIME          </td> <td> uiElapsedNanos                  </td></tr>
         * <tr><td> DR_THREAD_CPU_TIME              </td> <td> uiCpuNanos                      </td></tr>
         * <tr><td> DR_ABSOLUTE_NATIVE_STACK_TRACE  </td> <td> nativeAbsoluteStack (.NET only) </td></tr>
         * <tr><td> DR_VM_RELATIVE_STACK_TRACE      </td> <td> vmOffsetStack                   </td></tr>
         * <tr><td> DR_THREAD_STATE                 </td> <td> state (Java only)               </td></tr>
         * <tr><td> DR_OBJECT_ID                    </td> <td> threadObjectId (Java only)    </td></tr>
         * <tr><td> DR_CURRENT_MONITOR              </td> <td> currentMonitorId (Java only)    </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types"
         *
         * @remark - When requesting thread information, you can use 'threadId = 0' to request
         *           information for the current thread.
         *
         * @remark - When requesting a stack trace (DR_VM_RELATIVE_STACK_TRACE or 
         *           DR_ABSOLUTE_NATIVE_STACK_TRACE), use the 
         *           @c uiSize member of the @c vmOffsetStack or @c nativeAbsoluteStack
         *           structures to specify the number of stack frames to return
         * 
         * @remark - In .NET, stack trace requests are available only for the current thread.
         *
         * @remark - In Java, DR_THREAD_ELAPSED_TIME and DR_THREAD_CPU_TIME are available 
         *           only for the current thread.
         *           In .NET, DR_THREAD_ELAPSED_TIME and DR_THREAD_CPU_TIME are available 
         *           for any thread.
         *
         * @remark - In Java 5.0 VMs, DR_THREAD_INFO is available only after the VM has been
         *           initialized (i.e. the VM Init event was sent). When requesting this 
         *           information, the client must examine the return code to check whether
         *           the information was successfully retrieved.
         */
        virtual TResult GetThreadInfo(TId clientId,
                                      TId threadId,
                                      BitSet requestedDataTypes,
                                      SThreadInfo *pData,
                                      TDotNetUnmanagedStackWalkCallBack pfnStackWalker = NULL) = 0;
        /**
         * @brief Request information for all living threads
         *
         * Returns information about all living (non-terminated) threads. The information is
         * returned in an array (@c pData->pEntries) allocated by the caller. Each entry 
         * describes one thread.
         *
         * @param[in] clientId              the id of the client
         * @param[in] maxFrameCount         the maximum number of stack frames to return
         *                                  for each thread, when requesting
         *                                  @c DR_VM_RELATIVE_STACK_TRACE. Note that the buffers
         *                                  for storing the stack trace information should also
         *                                  be allocated by the caller.
         * @param[in] requestedDataTypes    the data items to retrieve, specified as a 
         *                                  combination of @c TDataRequestType consts OR'ed 
         *                                  together.
         * @param[out] pData                a reference to an allocated data structure.
         *                                  Upon return, MPI will write the requested data to 
         *                                  this structure.
         *
         * Refer to the @ref data_request_functions "Data Request Functions" section for 
         * additional usage information and return values.
         *
         * The function supports the following data items:
         *
         * <table>
         * <tr><td> <b>Data Item</b>                </td></tr>
         * <tr><td> DR_THREAD_NAME                  </td></tr>
         * <tr><td> DR_THREAD_GROUP                 </td></tr>
         * <tr><td> DR_THREAD_PARENT_GROUP          </td></tr>
         * <tr><td> DR_VM_RELATIVE_STACK_TRACE      </td></tr>
         * <tr><td> DR_THREAD_STATE                 </td></tr>
         * <tr><td> DR_CURRENT_MONITOR              </td></tr>
         * <tr><td> DR_OBJECT_ID                    </td></tr>
         * </table>  
         *
         * For more information about the supported data items, refer 
         * to @ref data_request_types "Data Request Types"
         *
         * @remark - This function is currently supported only on Java 5.0 and later.
         *
         */
        virtual TResult GetAllThreadsInfo(TId clientId,
                                          U32 maxFrameCount,
                                          BitSet requestedDataTypes,
                                          SThreadInfoArray *pData) = 0;

        /**
         * @brief Data Request for reachable objects on the managed heap.
         *
         * Returns object reference information for all reachable objects on the managed heap.
         *
         * This data request is currently supported only for Java 5.0 VMs.
         *
         * @remark - In Java VMs, the information is available only after the VM has been
         *           initialized (i.e. the VM Init event was sent). When requesting this 
         *           information, the client must examine the return code to check whether
         *           the information was successfully retrieved.
         *
         */
        virtual TResult GetObjectReferences(TId clientId,
                                            SObjectReferenceInfo *pData) = 0;
        //@}

        /**
         * @name Miscellaneous Functions
         */

        //@{

        /**
         * @brief Retrieves the runtime type 
         *
         * Retrieves the runtime type hosting the MPI implementation module
         *
         * @retval RT_JAVA              Java
         * @retval RT_DOTNET            .NET
         * @retval RT_JIT_PROFILING     JIT Profiling
         * @retval RT_UNDEFINED         Unknown/undefined
         */
        virtual ERuntimeType GetRuntimeType() = 0;

        /**
         * @brief Instantiates another MPI client
         * 
         * This function is called by any MPI client that wishes to instantiate another MPI
         * client.
         * The instantiation is done only if the requested module was not instantiated 
         * previously (either by a call to InstantiateClient or according to the initial
         * configuration) and only during initialization phase.
         *
         * @param[in] clientId                  The id of the requesting client
         *
         * @param[in] szRequestedClientName     Name of the client to instantiate. This is the 
         *                                      name of the shared library, without the 
         *                                      OS specific prefix/postfix. 
         *                                      For example, in order to instantiate 
         *                                      'profiler.dll' or 'libprofiler.so' use 
         *                                      "profiler" as the client's name. The correct
         *                                      library will be loaded according to the OS.
         *                                      The requested library should be in 
         *                                      JAVA_PROFILER_HOME directory or in the PATH if
         *                                      such an environment variable does not exist.
         * 
         * @param[in] szOptions                 Client-specific options in the following
         *                                      format: param=value[,param=value]*
         *
         * @retval MRTE_RESULT_OK               If successfully instantiated the client
         * @retval MRTE_RESULT_PRV_INSTANTIATED If requested client was already instantiated
         * @retval MRTE_ERROR_NULL_PTR          If client name is null 
         * @retval MRTE_ERROR_FAIL              If client was not found
         * @retval MRTE_ERROR_ILLEGAL_ID        If clientId is not a valid MPI client id
         * @retval MRTE_ERROR_PHASE_FAILURE     Can't instantiate after initialization is 
         *                                      complete 
         */
        virtual TResult InstantiateClient(TId clientId, 
                                          const char *szRequestedClientName,
                                          const char *szOptions = 0) = 0;

        /**
         * @brief Returns an instrumentation interface
         * 
         * Returns an interface for instrumenting managed code (Java Byte Code or .NET IL)
         *
         * @returns IMIE* pointer to the instrumentation interface, or NULL if the operation
         *          is not supported. For more information, refer to MieAPI.h
         *
         * @remark - in .NET, the client may cast the returned interface pointer to ICIE*
         *           in order to access the full functionality of the CLR Instrumentation
         *           interface.
         *
         * @remark - in Java, this operation is currently not supported, and the function will
         *           always return NULL.
         */
        virtual MIE::IMIE* GetInstrumentor() = 0;

        /**
         * @brief Configures the virtual machine and the profiling interface module
         *
         * This function is called by the MPI client to set the configuration of the virtual
         * machine and the profiling interface module. The possible configuration flags are 
         * defined by the EConfigurationFlags enumeration.
         *
         * This function can only be called during the initialization phase, i.e.
         * from the MPICL_Instantiate() function of the client.
         *
         * This function can be called several times for each configuration flag, or once
         * with the configuration flags OR'ed in the 'configuration' argument.
         *
         * @param[in] configuration  - A bit-mask that determines the configuration
         *                             as defined by EConfigurationFlags. 
         *                             This parameter is used as a bitwise mask to allow for
         *                             several configuration flags to be set together
         *
         * @param[in] pConfigData    - Additional configuration information
         *
         * @retval MRTE_RESULT_OK               Configuration was applied successfully
         * @retval MRTE_ERROR_ILLEGAL_ARGUMENT  At least one of the configuration flags
         *                                      requires additional data, but the data is
         *                                      missing 
         * @retval MRTE_ERROR_FAILURE           Failed to configure at least one of the flags
         * @retval MRTE_ERROR_NOT_SUPPORTED     At least one of the configuration flags are 
         *                                      not supported
         * @retval MRTE_ERROR_PHASE_FAILURE     Can't configure after initialization is 
         *                                      complete
         */
        virtual TResult Configure(BitSet configuration, 
                                  SConfigData *pConfigData = 0) = 0;

        /**
         * @brief Sends message to a log file and/or to the External Control module
         *
         * This function is called by the MPI client to send a text message to the log file
         * and/or to an External Control module, if one is loaded.
         *
         * @param[in] clientId  The id of the client
         *
         * @param[in] msgType   Message type. Can be one of the items defined in the
         *                      EMessageType enumerator.
         *
         * @param[in] msgDest   A bit-mask that determines the destination of the message
         *                      as defined by EMessageDest. Multiple destinations can be
         *                      specified by OR'ing several EMessageDest values together.
         *
         * @param[in] szMessage A null-terminated string that specifies the message to send
         *
         * @param[in] iLevel    The level of the message, relevant on Debug and Informative 
         *                      only. If not specified the highest level (0) is assumed.
         *                      Log messages can be filtered by levels by defining the logging
         *                      level in the MARTIN_LOGGER_LOG_LEVEL environment variable.
         *                      Only messages with a level less than or equal to
         *                      MARTINI_LOGGER_LOG_LEVEL will be sent to the log file.
         *                      The default logging level is 3.
         *
         * @retval MRTE_RESULT_OK       Message was sent successfully
         * @retval MRTE_ERROR_FAILURE   Failed to send the message
         *
         */
        virtual TResult MessageLog(TId clientId, EMessageType msgType, BitSet msgDest, 
                                   char* szMessage, unsigned int iLevel = 0) = 0;

        
        /**
         * @brief Returns the MPI version implemented by the loaded Martini module
         * 
         * This function returns the major and minor version numbers of the MPI interface
         * implemented by the loaded Martini module.
         *
         * @param[out] pVersionInfo     A pointer to a client allocated SVersionInfo structure
         *                              to which to write the version information.
         */
        virtual void GetVersion(SVersionInfo *pVersionInfo) = 0;

        /**
         * @brief Suspends all threads of the running application.
         *
         * This function suspends all threads of the running application.
         *
         * @retval MRTE_RESULT_OK               All threads successfully suspended
         * @retval MRTE_ERROR_PHASE_FAILURE     Wrong phase. The function can be called only
         *                                      after the VM has been initialized, and the 
         *                                      VM Init event was sent
         * @retval MRTE_ERROR_FAILURE           Failed to suspend all threads.
         *
         * @remark - The operation will fail if invoked from the context of any of the 
         *           application's threads (e.g., from within most event handlers), as it will
         *           cause the client itself to be suspended and the call will never return. 
         *           
         */
        virtual TResult SuspendVM() = 0;

        /**
         * @brief Resumes all threads of the running application.
         *
         * This function resumes all threads of the running application.
         *
         * @retval MRTE_RESULT_OK               All threads successfully resumed
         * @retval MRTE_ERROR_PHASE_FAILURE     Wrong phase. The function can be called only
         *                                      after the VM has been initialized, and the 
         *                                      VM Init event was sent
         * @retval MRTE_ERROR_FAILURE           Failed to resume all threads
         */
        virtual TResult ResumeVM() = 0;

        /**
         * @brief Forces the virtual machine to perform a garbage collection.
         * 
         * This function forces the virtual machine to perform a garbage collection.
         *
         * @retval MRTE_RESULT_OK               Success
         * @retval MRTE_ERROR_PHASE_FAILURE     Wrong phase. The function can be called only
         *                                      after the VM has been initialized, and the 
         *                                      VM Init event was sent
         * @retval MRTE_ERROR_FAILURE           Failure
         */
        virtual TResult RunGC() = 0;

        /**
         * @brief Runs heap instance data collection to analyze and obtain information about the Object.
         * 
         * This function collects information about the requested Object.
         *
         * @retval MRTE_RESULT_OK               Success
         * @retval MRTE_ERROR_FAILURE           Failure
         */
        virtual TResult RunHeapObjDataCollection(THeapObjectAnalysis *TId) = 0;

		/**
         * @brief Sets the heap instance data collection flag to enable or disable collection.
         * 
         * This function sets the flag to enable or disable heap instance data collection.
         * 
         */
		virtual void SetHeapObjDataCollection(bool heapObjDataCollection) = 0;
		
        /**
         * @brief Generates Object Allocated events for live objects on the heap.
         * 
         * This function generates Object Allocated events for live objects on the 
         * the heap. 
         * 
         * @param[in] clientId  The id of the client
         * 
         * @retval MRTE_RESULT_OK               Success
         * @retval MRTE_ERROR_PHASE_FAILURE     Wrong phase. The function can be called
         *                                      only after the VM has been initialized,
         *                                      and the VM Init event was sent
         * @retval MRTE_ERROR_FAILURE           Failure
         */

        virtual TResult GenerateObjectAllocEventForLiveObjects(TId clientId) = 0;

        //@}
        
    };

    //////////////////////////////////////////////////////////////////////////
    // MPI client entry point
    //////////////////////////////////////////////////////////////////////////

    /**
     * @brief Prototype of MPI client initialization function
     *
     * The MPI client must implement an initialization function whose name is MPICL_Instantiate
     * and whose type is TClientInstantiate. This method is called during initialization phase 
     * by the module that implements MPI. 
     * The client is expected to initialize itself and register for events from within this 
     * function.
     *
     * @param[in] pMpiAPI      MPI interface object. With this interface object all MPI API's 
     *                         can be called
     * @param[in] clientId     Id of the client, to be passed with all MPI functions.
     * @param[in] szOptions    Client-specific options in the following format:
     *                         param=value[,param=value]*
     *
     * @retval MRTE_RESULT_OK               Client successfully initialized
     * @retval MRTE_RESULT_FAIL             Client failed to initialize
     * @retval MRTE_RESULT_OUT_OF_MEMORY    Out of memory
     * @retval MRTE_ERROR_PHASE_FAILURE     Can't instantiate after initialization phase 
     */
    typedef TResult(*TClientInstantiate)(IMpi *pMpiApi, TId clientId, const char *szOptions);


}};

#endif
