/*****************************************************************************
 * Copyright (c) 1997, 2008 Intel Corporation.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/

/**
 * @file
 */

/**
 * @mainpage Martini API Specification
 * @version 2.0
 * 
 * This document is the API specification for Martini SDK, version 2.0.
 * The document contains the following sections:
 * -# @ref page_whatsnew "What's New in MPI 2.0"
 * -# @ref page_mpiapi "Martini Profiling Interface"
 * -# @ref page_ec "Martini External Control Interface"
 */

/** 
 * @page page_mpiapi MPI - Martini Profiling Interface
 *
 * @section page_mpiapi_content Table of Contents
 * -# @ref intro "Introduction"
 * -# @ref init "Client Initialization"
 * -# @ref event_handling "MPI Event Handling"
 * -# @ref event_filters "Event Groups and Filters"
 * -# @ref data_request_functions "Data Requests"
 * -# @ref mpi_arrays "Working with MPI Arrays"
 *
 * @section intro Introduction
 *
 * The Martini Profiling Interface (MPI) is an interface for managed-runtime profiling tools. 
 * It is defined to suit a wide-range of virtual machines, such as Java and 
 * .NET. Its main goals is to allow using the same tool with different managed-runtime 
 * environments.
 *
 * MPI is a declarative, event-based interface, offering a variety of well-defined events
 * and data request operations. Some of the events may be optionally restricted to specific 
 * selective criteria, such as specific classes rather than all, reducing the overhead and the
 * amount of data generated.
 *
 * An MPI client registers to relevant events and event-specific data by implementing Event 
 * Observer objects. Once the event occurs, the Observer's @c HandleEvent
 * callback function is called, with event-specific data passed as an argument. 
 * Additional data can be retrieved by calling one of the data request functions.
 * 
 * More that one profiler can use MPI at the same time. Each profiler registers to its own 
 * events and receives notification only for the events to which it registered.
 *
 * @section init Client Initialization 
 * 
 * The MPI client library is loaded by the module that implements MPI, according to its 
 * configuration (TBD). Once the client library is loaded, its @ref TClientInstantiate 
 * "MPICL_Instantiate()" entry-point function is called. The client is expected to initialize
 * itself and register for MPI events at this time. Registration to MPI events is possible
 * only at this phase. For certain events, the profiler can decide to initially disable these
 * events (meaning it will not get any notifications for these events) and enable them at a
 * later time.
 *
 * @section event_handling MPI Event Handling
 *
 * To handle MPI events, an MPI client must define an Event Observer class for the specific 
 * MPI event it needs to handle, and then register an instance of this class with Martini
 * using the @c IMpi::RegisterEvent function. The client can optionally override the observer's 
 * @c EventDataTypes method and specify the data items to be supplied with the
 * event.
 * After registration, when the requested event occurs, the Event Observer's 
 * @c HandleEvent function is called with the requested data as an argument.
 * 
 * The following example shows how to define and register an event observer for the 
 * Method Enter MPI event:
 *
 * @code
 *
 * using namespace Martini::MPI;
 *
 * // Define an event observer for the Method Enter event
 * class MethodEnterObserver : public IMethodEnterEventObserver
 * {
 * public:
 *
 *     // Specify the data items to be supplied with the event
 *     virtual BitSet EventDataTypes()
 *     {
 *         return DR_METHOD_ID | DR_THREAD_ID;
 *     }
 * 
 *     // Event handler
 *     virtual void HandleEvent(SMethodEnterEventData &data)
 *     {
 *         cout << "Method Enter event received. Method id = " 
 *              << data.methodId << " Thread id = " << data.threadId << endl;
 *     }
 * }
 *
 * // Event Observer instance
 * MethodEnterObserver g_methodEnterObserver;
 *
 * // MPI Client entry point
 * extern "C" API_EXPORT 
 * TResult MPICL_Instantiate(IMpi *pMpiApi, TId clientId, const char *szOptions)
 * {
 *     TResult res = pMpiApi->RegisterEvent(clientId, g_methodEnterObserver);
 *     return MRTE_RESULT_OK;
 * }
 *     
 * @endcode
 * 
 * @section event_filters Event Groups and Filters
 *
 * Some MPI events can be optionally restricted to specific selective criteria, 
 * in order to reduce the overhead and the amount of data generated. For example,
 * the MPI client can restrict the Method Enter and Method Leave events to be generated
 * only for specific methods rather than for all methods, thereby reducing the overhead
 * on the profiled application. The MPI Client define these restrictions by implementing
 * Event Filter objects and registering them with MPI.
 * 
 * Since applying filters usually make sense for a group of events rather than for an 
 * individual event, MPI Event Filters are applied to Event Groups and not to specific events.
 *
 * MPI defines the following event groups:
 * - EG_CALL_GRAPH: New Method, Method Enter and Method Leave
 * - EG_HEAP: Object Alloc and Object Free
 * - EG_MONITOR: Monitor Wait, Monitor Waited, Contended Monitor Enter and 
 *               Contended Monitor Entered
 * - EG_THREAD_INTERACTION: Thread Interaction
 * 
 * To apply a filter to an MPI event group, the MPI client must define an Event Filter class
 * for the specific event group it wants to restrict, and then register an instance of this 
 * class with Martini using the @c IMpi::SetEventGroupFilter function.
 *
 * After registration, The Event Filter's @c ShouldNotify function will be called by Martini
 * to determine the set of elements for which to generate the events belonging to the
 * restricted group.
 * 
 * The following example shows how to instruct Martini to generate Method Enter and 
 * Method Leave events only for public methods of the public class @c Foo:
 *
 * @code
 *
 * using namespace Martini::MPI;
 *
 * // Define a call-graph event filter
 * class CallGraphFilter : public ICallGraphFilter
 * {
 *     bool ShouldNotify(SCallGraphFilterData &methodInfo)
 *     {
 *         if (strcmp("Foo", data.szClassName) == 0 && data.methodAccessType == MAT_PACKAGE_API)
 *         {
 *             return true;
 *         }
 *         return false;
 *     }
 * };
 *
 * // Event Observer instance
 * MethodEnterObserver g_methodEnterObserver;
 * 
 * // Event Filter instance
 * CallGraphFilter g_callGraphFilter;
 *
 * // MPI Client entry point
 * extern "C" API_EXPORT 
 * TResult MPICL_Instantiate(IMpi *pMpiApi, TId clientId, const char *szOptions)
 * {
 *     // Register for the Method Enter event
 *     pMpiApi->RegisterEvent(clientId, g_methodEnterObserver);
 *
 *     // Set the call-graph filter
 *     pMpiApi->SetEventGroupFilter(clientId, EG_CALL_GRAPH, g_callGraphFilter);
 *
 *     return MRTE_RESULT_OK;
 * }
 *     
 * @endcode
 * 
 * @section data_request_functions Data Requests
 *
 * The Data Request functions are called by the MPI client to receive additional 
 * information about an element, such as a class, method, object and so on.
 * Refer to the documentation of the specific 
 * @ref impi_data_request_group "data request functions" for detailed information about each 
 * of these function.
 *
 * All data request functions has the following definition
 *
 * @code
 * virtual TResult GetElementInfo(TId clientId,
 *                                TId ElementId,
 *                                BitSet requestedDataTypes, 
 *                                SElementInfo *pData);
 *
 * @endcode
 *
 * Where <b>Element</b> represents the type of element for which to retrieve information.
 *
 * The data request functions are all used in the following way:
 * -# The client specifies its client id in the @c clientId parameter
 * -# The client specifies the id of the element to query in the @c ElementId
 *    parameter (@c moduleId, @c classId, @c objectId, etc...).
 * -# The client specifies the data items to retrieve in the @c requestedDataTypes
 *    parameter. Data items are specified as a combination of 
 *    TDataRequestType consts OR'ed together (see @ref data_request_types "Data Request Types).
 * -# The client passes a reference to an allocated data structure in the 
 *    @c pData parameter. Upon return, MPI will write the requested data to this 
 *    structure and will set the @c pData->validData field to indicate which data 
 *    items were successfully retrieved. 
 *
 * All data request return one of the following values:
 * @retval MRTE_RESULT_OK               If data successfully retrieved
 * @retval MRTE_ERROR_PARTIAL_INFO      If not all data items could be retrieved.
 *                                      The client should examine the 
 *                                      'pData->validData' field to check which data 
 *                                      items were returned.
 * @retval MRTE_ERROR_BUFFER_TOO_SHORT  For requests that return arrays, indicates that
 *                                      not enough space was allocated to return all 
 *                                      the available information. For more information,
 *                                      refer to the 
 *                                      @ref mpi_arrays "Working with MPI Arrays" section.
 * @retval MRTE_ERROR_NULL_PTR          If @c pData is NULL, or one of its requested
 *                                      pointer fields is NULL
 * @retval MRTE_ERROR_ILLEGAL_ID        If @c clientId is not a valid MPI client id
 * @retval MRTE_ERROR_NOT_SUPPORTED     If one of the specified data items is not
 *                                      supported by the data request.
 * @retval MRTE_ERROR_PHASE_FAILURE     If the requested information cannot be 
 *                                      retrieved before the VM has initialized.
 *                                      The client should request the information 
 *                                      only after the VM Init event has been sent.
 * @retval MRTE_ERROR_FAIL              If failed to retrieve data 
 *
 * @remark - The memory for the @c pData argument must be allocated by the client
 *           before calling the data request function. Memory must be allocated  
 *           for pointer fields that correspond to requested data items as well.
 *
 * The following example shows how to retreive the name and signature of a method:
 * @code
 * IMpi *g_pMpiApi;
 *
 * void CNewMethodObserver::HandleEvent(SNewMethodEventData &data)
 * {
 *     char szName[1000];
 *     char szSig[1000];
 *     SMethodInfo methodInfo;
 *     methodInfo.szMethodName = szName;
 *     methodInfo.szMethodSig = szSig;
 * 
 *     TResult res = g_pMpiApi->GetMethodInfo(m_clientId, data.methodId, 
 *         DR_METHOD_NAME | DR_METHOD_SIGNATURE, &methodInfo);
 * }
 * @endcode
 *
 * @section mpi_arrays Working with MPI Arrays
 * 
 * Several MPI events and data request functions return information in arrays. Generally,
 * arrays returned in an event callback are allocated and freed by the MPI implementation, 
 * while arrays returned by data request functions must be pre-allocated by the MPI Client.
 *
 * All MPI arrays are defined as structs similar to the following
 * @code
 * struct SArray
 * {
 *     TArrayElement *entries;
 *     unsigned int uiSize;
 *     unsigned int uiActualSize;
 * };
 * @endcode
 *
 * - The @c entries member is a pointer to the array elements
 * - When sent as part of an event data, the memory for the array is allocated by MPI 
 *   and the @c uiSize member indicates the size of the @c entries array.
 * - When used in a data request, the client is responsible for allocating memory for the 
 *   array. The client specifies the number of entries it allocated in the @c uiSize member.
 *   When the data request returns, the @c uiActualSize member will indicate the number of
 *   entries that were populated. If not enough entries were allocated, the first @c uiSize
 *   entries will be populated, the @c uiActualSize member will indicate the actual number
 *   of available entries and the Data Request operation will return the 
 *   MRTE_ERROR_BUFFER_TOO_SHORT error.
 */

/**
 * @page page_whatsnew What's New in MPI 2.0
 *
 * MPI version 2.0 adds new features that are designed to support additional events, 
 * data requests and diverse event filtering needs.
 *
 * @section whatsnew_event_observers Event-specific Interfaces
 *
 * MPI version 2.0 defines an Observer interface for each MPI event, rather than a generic
 * @c IEvent interface to be used for all events. Each observer has its
 * own @c HandleEvent callback function which delivers event-specific data.
 *
 * For more information, refer to the @ref event_handling "MPI Event Handling" section.
 *
 * @section whatsnew_data_requests Data Request Functions
 *
 * MPI version 2.0 defines a Data Request function for each type of MPI element 
 * (class, method, thread, object, etc). These data request functions replace the monolithic
 * @c DataRequest function in MPI version 1.0 and provide a simpler interface for obtaining
 * data.
 *
 * For more information, refer to the @ref data_request_functions "Data Requests" section.
 *
 * @section whatsnew_event_filters Event Filters
 *
 * MPI version 2.0 introduces a new concept called Event Filters. Event Filters replace
 * the @c ISelectiveEvent filtering mechanism in MPI version 1.0.
 *
 * For more information, refer to the @ref event_filters "Event Groups and Filters" section.
 */

#ifndef _MRTE_MPI_BASE_H
#define _MRTE_MPI_BASE_H

namespace Martini { namespace MPI {

    /**
     * @name Defines all MPI supported events
     *
     * This is a list of all supported MPI events. However there are multiple modules that 
     * implement MPI, not all of them support the entire set of events. Unsupported events
     * will result in a proper return value during event registration via 
     * @c IMpi::RegisterEvent()
     *
     * @remark - Java-specific events start with EV_JAVA
     *
     * @remark - .NET-specific events start with EV_DOTNET
     **/
    //@{
    typedef unsigned int TEventType;
    
    const TEventType EV_NONE = 0;
    const TEventType EV_METHOD_ENTER = 1;
    const TEventType EV_METHOD_LEAVE = 2;
    const TEventType EV_NEW_METHOD = 3;
    const TEventType EV_JITTED_METHOD_LOADED = 4;
    const TEventType EV_JITTED_METHOD_UNLOADED = 5; 
    const TEventType EV_THREAD_START = 6;
    const TEventType EV_THREAD_END = 7;
    const TEventType EV_OBJECT_ALLOC = 8;
    const TEventType EV_OBJECT_FREE = 9;
    const TEventType EV_GC_START = 10;
    const TEventType EV_GC_END = 11;
    const TEventType EV_GC_MOVED_REFERENCES = 12;
    const TEventType EV_VM_INIT = 13;
    const TEventType EV_VM_SHUTDOWN = 14;
    const TEventType EV_DOTNET_METHOD_INSTRUMENTATION_POINT = 15;
    const TEventType EV_DOTNET_MODULE_UNLOAD_STARTED = 16;        
    const TEventType EV_DOTNET_MODULE_LOAD_FINISHED = 17;         
    const TEventType EV_JAVA_DYNAMIC_CODE_GENERATED = 18; 
    const TEventType EV_EC_START = 19;              
    const TEventType EV_EC_STOP = 20;               
    const TEventType EV_EC_SET_OUTPUT_DIRECTORY = 21;
    const TEventType EV_EC_ATTACH = 22;             
    const TEventType EV_EC_DETACH = 23;  
    const TEventType EV_EC_CUSTOM_COMMAND = 24;
    const TEventType EV_MONITOR_WAIT = 25;
    const TEventType EV_MONITOR_WAITED = 26;
    const TEventType EV_CONTENDED_MONITOR_ENTER = 27;
    const TEventType EV_CONTENDED_MONITOR_ENTERED = 28;
    const TEventType EV_JAVA_CLASS_FILE_LOAD_HOOK = 29;
    const TEventType EV_THREAD_INTERACTION = 30;
    const TEventType EV_LAST = 31;
    //@}
        
    /**
     * @brief Bit set
     */
    typedef U32 BitSet;

    //////////////////////////////////////////////////////////////////////////
    // Generic Event Filter Definitions
    //////////////////////////////////////////////////////////////////////////

    /**
     * @interface IEventFilter
     * @brief Event Filter interface
     *
     * Defines the base interface for event filters
     */
    class IEventFilter
    {
    public:
        /**
         * @brief Defines supported filter types
         */
        enum EFilterType
        {
            FT_CALLGRAPH,          //!< Call-graph filter
            FT_HEAP,               //!< Heap filter
            FT_THREAD_INTERACTION  //!< Thread-interaction filter
        };

        /**
         * @brief Returns the filter type
         *
         * Returns the filter type. The MPI implementation is using the function
         * to determine the type of the filter.
         *
         * @return one of the filter types defined in EFilterType
         */
        virtual EFilterType Type() = 0;
    };

}};

#endif
