/*****************************************************************************
 * Copyright (c) 1997, 2010 Intel Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Intel Corporation - Initial API and implementation
 *
 * $Id$ 
 *****************************************************************************/


#ifndef __OSA_H__
#define __OSA_H__

#include "MRTEResults.h"
#include "MRTETypes.h"

#ifndef MAX_PATH
#define MAX_PATH 260
#endif

#ifdef LIN
#define API_EXPORT 
#define API_IMPORT
#define __cdecl
#define DIRECTORY_SEPARATOR     '/'
#define DIRECTORY_SEPARATOR_STR "/"
#else 
#define API_EXPORT __declspec(dllexport)
#define API_IMPORT __declspec(dllimport)
#define DIRECTORY_SEPARATOR     '\\'   
#define DIRECTORY_SEPARATOR_STR "\\"   
#endif


#ifdef MARTINIOSA_EXPORTS
#define OSA_EXPORT API_EXPORT
#else
#define OSA_EXPORT API_IMPORT
#endif

// Environment variable to be used by MRTE (Java/.NET loaders) to load Bistro binaries
#define BISTRO_INSTALL_DIR_ENV_VAR "BISTRO_INSTALL_DIR"


namespace Martini { namespace OSA
{
    /**
    *  @brief <b>IBase</b> Base class for all OSA classes
    */
    class IBase      
    {
    public:
        /**
        * <b>IBase::Destroy</b>Deletes the object 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    };
    
    
    /**
    *  @brief <b>IBaseSync</b> Base class of all OSA synchronization sub-classes
    */
    class IBaseSync : public IBase
    {
    public:
        
        /**
        * <b>IBaseSync::Enter</b> Enters a critical section (acquires a lock) <br> 
        *
        * Locks the object that contains this function. If already locked by another thread, 
        * blocks (waits) until lock is released. If already locked by the current thread, 
        * returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK Entered critical section 
        */
        virtual TResult Enter() = 0;
        
        /**
        * <b>IBaseSync::Leave</b> Leaves a critical section (releases a lock) <br> 
        *
        * Releases the object that contains this function. If not locked by the current thread, 
        * returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK Left critical section 
        */
        virtual TResult Leave() = 0;
        
        /**
        * <b>IBaseSync::TryEnter</b> Tries to enter a critical section (acquires a lock) <br> 
        *
        * Locks the object that contains this function. If already locked by another thread, 
        * returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK When the lock was taken. 
        *
        * @return MRTE_RESULT_FALSE When another thread has already acquired the lock. 
        */
        virtual TResult TryEnter() = 0;
        
        /**
        * <b>IBaseSync::Destroy</b> destroy the sync object and deletes the object 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    };
    
    /**
    *  @brief <b>ILibraryLoader</b> Dynamic library manipulation object
    */
    class ILibraryLoader : public IBase
    {
    public:
        
        /**
        * <b>ILibraryLoader::GetEntry</b> Retrieves the address of a function in a dynamically 
        *                                 loaded library. <br>
        *
        * @return The function pointer
        *
        * @return NULL - The function name has not been found in the library
        */
        virtual void *GetEntry(char *szEntryName) = 0;
        
        /**
        * <b>ILibraryLoader::Destroy</b> Deletes the library object but does not unload the 
        *                                library 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
        
        
    protected:    
        virtual TResult Create(const char *szLibName, const char *szLibPath, 
                               bool bLoadOnce = false) = 0;
        
    };
    
    /**
    *  @brief <b>IThreadSync</b> Thread synchronization capabilities
    */
    class IThreadSync : public IBaseSync 
    {
    public:
        /**
        * <b>IThreadSync::Enter</b> Enters a critical section (acquires a lock) <br> 
        *
        * Locks the object that contains this function. If already locked by another thread, 
        * blocks (waits) until lock is released. If already locked by the current thread, 
        * returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK Entered critical section 
        */
        virtual TResult Enter() = 0;

        /**
        * <b>IThreadSync::Leave</b> Leaves a critical section (releases a lock) <br> 
        *
        * Releases the object that contains this function. If not locked by the current  
        * thread, returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK Left critical section 
        */
        virtual TResult Leave() = 0;
        
        /**
        * <b>IThreadSync::TryEnter</b> Tries to enter a critical section (acquires a lock)<br> 
        *
        * Locks the object that contains this function. If already locked by another thread, 
        * returns immediately.
        *
        * @param None. 
        *
        * @return MRTE_RESULT_OK When the lock was taken. 
        *
        * @return MRTE_RESULT_FALSE When another thread has already acquired the lock. 
        */
        virtual TResult TryEnter() = 0;
        
        /**
        * <b>IThreadSync::Destroy</b> destroy the sync object and deletes the object 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    protected:
        virtual TResult Create() = 0;
    };
    
    /**
    *  @brief <b>IReadOnlyFileMapping</b> Enables retrieval of read-only file mapping 
    *                                     attributes.
    */
    class IReadOnlyFileMapping : public IBase  
    {
    public:
        
        /**
        * <b>IReadOnlyFileMapping::GetBaseAddress</b> Returns the memory address where the 
        *                                             file represented by this object starts. 
        *
        * @param None 
        *
        * @return The starting memory address of the mapped file
        */
        virtual void* GetBaseAddress() = 0;
        
        /**
        * <b>IReadOnlyFileMapping::GetLength</b> Returns the length in bytes of the mapped 
        *                                        file. 
        *
        * @param None 
        *
        * @return The length in bytes of the mapped file
        */
        virtual unsigned int GetLength() = 0;
        
        /**
        * <b>IReadOnlyFileMapping::Destroy</b> Unmaps the file represented by this object from 
        *                                      the memory 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    protected:    
        virtual TResult Open(const char *szFileName) = 0;
    };
    
    /**
     *  @brief <b>IDirectoryHandle</b> Represents a directory. User may receive valid reference 
     *                                 to this object only if the required directory exists.
     */
    class IDirectoryHandle : public IBase
    {
    public:
        
        /**
        * <b>IDirectoryHandle::GetNextFileName</b> Is an iterator over the file (subdirectory) 
        * names within the directory represented by this object. The iterator is reset when 
        * this object is created. The iteration may continue until "no more files" are left in 
        * the directory
        *
        * @param pbIsDirectory - [out] The flag that is set in case that the file being
        *                              reported to user is a directory
        *
        * @param szFileName - [out] User buffer to hold the file name. The written file name 
        *                           is null-terminated.
        *
        * @param uiNumOfBytesToWrite - [in] Maximal number of bytes to write into the buffer
        * 
        * @param puiNumOfBytesWritten - [out] The number of bytes actually written to the 
        *                                     buffer
        *
        * @return MRTE_RESULT_OK - The file name was successfully stored in the user buffer
        *
        * @return MRTE_ERROR_ILLEGAL_ARGUMENT - One or more argument is illegal
        *
        * @return MRTE_ERROR_BUFFER_TOO_SHORT - The user buffer is to short to hold the file 
        *                                       name the required size is stored in 
        *                                       puiNumOfBytesNeeded.
        *
        * @return MRTE_RESULT_END_OF_DATA - The iteration has terminated
        */
        virtual TResult GetNextFileName(bool* pbIsDirectory, char* szFileName, 
            unsigned int uiNumOfBytesToWrite, unsigned int *puiNumOfBytesNeeded) = 0;
        
        /**
        * <b>IDirectoryHandle::GetName</b> Returns directory full path
        *
        * @param None
        *
        * @return Directory full path
        */
        virtual const char* GetName() = 0;
        
        /**
        * <b>IBase::Destroy</b> Deletes this object 
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    protected:    
        virtual TResult Open(const char *cszDirectoryName) = 0;
    };
    
    /**
    *  @brief <b>IThreadLocalStorage</b> Represents a thread local storage. 
    */
    class IThreadLocalStorage : public IBase
    {
    public:
        /**
        * <b>IThreadLocalStorage::GetValue</b> Returns the value in the calling 
        * thread's slot in thread local storage (TLS) index represented by this object
        *
        * @param None
        *
        * @return Contents of calling thread's slot of TLS 
        */
        virtual void* GetValue() = 0;
        
        /**
        * <b>IThreadLocalStorage::SetValue</b> Stores a value in the calling thread's 
        * slot in thread local storage (TLS) index represented by this object
        *
        * @param Value to store
        *
        * @return None
        */
        virtual void SetValue(void* pValue) = 0;
        
        /**
        * <b>IBase::Destroy</b> Deletes this object and frees the TLS slot
        *
        * @param None 
        *
        * @return MRTE_RESULT_OK Object was destroyed successfully. 
        */
        virtual TResult Destroy() = 0;
    protected:
        virtual TResult Create() = 0;
    };
    
    
    
    // External interfaces
    // ---------------------------------------------------------------------------------------
    
    /**
    * <b>CreateThreadSync</b> Creates a process wide synchronization object. <br>
    *
    * It has an Enter and Leave functions that facilitate thread synchronization.
    *
    * @param None. 
    *
    * @return A reference to the synchronization object.
    */
    OSA_EXPORT IThreadSync* CreateThreadSync();
    
    /**
    * <b>CreateLibraryLoader</b> Creates a process wide dynamic library loader object.<br>
    *
    * It has a GetEntry function that facilitates retrieval of dynamic library 
    * function addresses. Calling the Destroy public function of the returned object 
    * does not unload the library, it just deletes the library loader object. <br> 
    *
    * @param szLibName - [in] A null terminated string containing the dynamic library stripped
    *                    name with no lib prefix, or .so/.dll suffix.
    *
    * @param szLibPath - [in] A null terminated string containing an optional directory path
    *                    where to look for the dynamic library. If this argument is null, 
    *                    the dynamic library is looked for per the PATH (Windows) or
    *                    LD_LIBRARY_PATH (Linux) environment variables.
    * @param bLoadOnce - [in] A boolean variable (default is false). If the variable is true
    *                    then the library will only be loaded once. If a library with the 
    *                    same name (szLibName) is already loaded, the new object will point 
    *                    to it.
    *
    * @return A reference to the library loader object, after successful dynamic library load
    *
    * @return NULL - The dynamic library has not been found.
    */
#ifndef HAVE_RPATH
    extern "C"
#endif
    OSA_EXPORT ILibraryLoader* CreateLibraryLoader(const char *szLibName, 
        const char *szLibPath, bool bLoadOnce = false);
    
    // File Mapping  
    // ---------------------------------------------------------------------------------------
    /**
    * <b>CreateReadOnlyFileMapping</b> Creates a read-only memory mapping of a whole file.<br>
    *
    * It has a <b>GetBaseAddress()</b> function that returns a base address of a mapped file view in memory
    * and </b>GetLength()</b> function that returns the file length.
    * The file and the mapping are opened with Read-Only permissions. Calling 
    * the Destroy public function of the returned object unmaps the memory and destroys 
    * the object<br> 
    *
    * @param szFileName - [in] A full name of the file to be mapped
    *
    * @return A reference to the file mapping object, after successful file mapping
    *
    * @return NULL - The file could not be mapped
    */
    OSA_EXPORT IReadOnlyFileMapping* CreateReadOnlyFileMapping(const char* szFileName);
    
    /**
    * <b>CreateDirectoryHandle</b> Creates a directory handle given full path to the directory.<br>
    *
    * It has a <b>GetNextFileName()</b> function that iterates over the files within the directory.
    * Calling the Destroy public function of the returned object destroys the object<br> 
    *
    * @param szDirectoryName - [in] A full path to the directory
    *
    * @return A reference to the directory handle object, after the directory was successfuly located
    *
    * @return NULL - The directory does not exist
    */
    OSA_EXPORT IDirectoryHandle* CreateDirectoryHandle(const char* szDirectoryName);			
    
    
    /**
    * <b>CreateThreadLocalStorage</b> Creates a thread local storage (TLS) index. <br>
    *
    * It has a GetValue and SetValue functions that facilitate per thread data 
    * storage and retrieval.
    *
    * @param None. 
    *
    * @return A reference to the TLS object.
    * 
    */
    OSA_EXPORT IThreadLocalStorage* CreateThreadLocalStorage();
    
    // Misc functions
    // ---------------------------------------------------------------------------------------
    
    /**
    * <b>GetTimeStamp</b> Retrieves the current CPU cycle count since system reset <br>
    *
    * @param None.
    *
    * @return A 64 bit unsigned integer containing the tick count since system reset.
    * 
    */
    OSA_EXPORT U64 GetTimeStamp();
    
    // Process functions
    // ---------------------------------------------------------------------------------------
    
    /**
    * <b>GetEnvironmentVar</b> Retrieves the string value of an environment variable defined
    *                          in the current process 
    *
    * @param szVarName - [in] A null terminated string containing the environment variable name
    *
    * @param szValueBuffer - [out] A pointer to a buffer that contains the retrieved environment
    *                        variable value upon return
    *
    * @param uiBufferSize - [in] The buffer size (counted in bytes) allocated by
    *                       the caller to accommodate the returned environment variable. 
    *
    * @param puiBufferSize - [out] A pointer to the actual buffer size (counted in bytes). 
    *                       If the buffer is too short this variable contains the 
    *                       required size.
    *
    *
    * @return MRTE_RESULT_OK - The environment variable value resides in target buffer pointed 
    *                          by szValueBuffer
    *
    * @return MRTE_RESULT_FALSE - The environment variable was not found
    *
    * @return MRTE_ERROR_ILLEGAL_ARGUMENT - One or more arguments is illegal
    *
    * @return MRTE_ERROR_BUFFER_TOO_SHORT - The buffer pointed by szValueBuffer is too
    *         short to accommodate the environment variable value. puiBufferSize contains the
    *         required size including its terminating null character. 
    * 
    */
#ifndef HAVE_RPATH
    extern "C"
#endif
    OSA_EXPORT TResult GetEnvironmentVar(char *szVarName, char *szValueBuffer, 
        unsigned int uiBufferSize, unsigned int *puiBufferSize);
	
    
    /**
    * <b>SetEnvironmentVar</b> Sets an environment variable to the provided value
    *                          in the current process 
    *
    * @param szVarName - [in] A null terminated string containing the environment variable name
    *
    * @param szValue - [in] A pointer to a buffer that contains the new value for the
    *                        environment variable
    *
    * @return MRTE_RESULT_OK - The environment variable has been set with the new value
    *
    * @return MRTE_ERROR_ILLEGAL_ARGUMENT - One or more arguments is illegal
    * 
    */
#ifndef HAVE_RPATH
    extern "C"
#endif
    OSA_EXPORT TResult SetEnvironmentVar(char *szVarName, char *szValue);
    
    /**
    * <b>MakeDirectory</b> Creates a new directory
    *
    * @param szDirPath - [in] A null terminated string containing the new 
    *                         directory name and path. The directory will
    *                         be created with no security flags - allowing
    *                         anyone to access the new directory.
    *
    * @return MRTE_RESULT_OK - The directory was created successfully
    *
    * @retrun MRTE_ERROR_OSA_FAILURE - Failed to create the directory
    */
    OSA_EXPORT TResult MakeDirectory(char *szDirPath); 
    
    /**
    * <b>GetExecutableFileName</b> Gets full file name of the executable that started this
    *                              process
    *
    * @param szBuffer - [in] A buffer to store the module file name.
    *
    * @param uiBufferSize - [in] the size of the buffer. If the buffer is too short, 
    *                            module file name is truncated
    *
    * @param puiRequiredBufferSize - [out] the actual size of executable file name
    *
    * @return MRTE_RESULT_OK - The current executable file name was written to szBuffer
    *
    * @return MRTE_ERROR_ILLEGAL_ARGUMENT - One or more arguments is illegal
    *
    * @return MRTE_ERROR_BUFFER_TOO_SHORT - The buffer pointed by szBuffer is too short to
    *         accommodate the executable file name. puiRequiredBufferSize contains the 
    *         required size including its terminating null character. 
    *
    * @return MRTE_ERROR_OSA_FAILURE - Failed to get the executable file name from the system
    */
    OSA_EXPORT TResult GetExecutableFileName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize);        
    
    /**
    * <b>GetCurrProcessID</b> Gets the process ID of this process
    *
    * @return The process ID
    */
    OSA_EXPORT unsigned int GetCurrProcessID();

    /**
    * <b>GetCurrThreadID</b> Gets the thread ID of this thread
    *
    * @return The process ID
    */
    OSA_EXPORT unsigned int GetCurrThreadID();
    
    /**
    * <b>GetCurrentProcCommandLine</b> Gets the full command line that executed the process 
    *                                  as one null terminating string
    *
    * @param szBuffer - [in] A buffer to store the command line.
    *
    * @param uiBufferSize - [in] the size of the buffer. 
    *
    * @param puiRequiredBufferSize - [out] the actual size of executable file name
    *
    * @return MRTE_RESULT_OK - The command line was written to buffer
    *
    * @return MRTE_ERROR_BUFFER_TOO_SHORT - The buffer pointed by szBuffer is too short to
    *         accommodate the environment variable value. puiRequiredBufferSize contains the
    *         required size including its terminating null character. 
    *
    * @return MRTE_ERROR_OSA_FAILURE - Failed to get the command line from the system
    */
    OSA_EXPORT TResult GetCurrentProcCommandLine(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize);        
    
    /**
    * <b>GetThisMachineName</b> Get the machine name 
    *
    * @param szBuffer - [in] A buffer to store the module file name.
    *
    * @param uiBufferSize - [in] the size of the buffer. If the buffer is too short, 
    *                            module file name is truncated
    *
    * @param puiRequiredBufferSize - [out] the actual size of executable file name
    *
    * @return MRTE_RESULT_OK - The current module's file name was written to buffer
    *
    * @return MRTE_ERROR_BUFFER_TOO_SHORT - The buffer pointed by szBuffer is too short to 
    *         accommodate the environment variable value. puiRequiredBufferSize contains the
    *         required size including its terminating null character. 
    *
    * @return MRTE_ERROR_OSA_FAILURE - Failed to get the machine name from the system
    */
    OSA_EXPORT TResult GetThisMachineName(char* szBuffer, unsigned int uiBufferSize,
        unsigned int *puiRequiredBufferSize);
    
    /**
     * <b> Sleep </b> Suspends the execution of the current thread for at least the specified
     *                interval.
     * 
     * @param uiMilliseconds    [in]    : minimum time interval for suspending the current
     *                                    thread.
     */
    OSA_EXPORT void Sleep(unsigned int uiMilliseconds);
        
     
	/* ICONV functions - src from RAC */
	#if defined(MVS) || defined(__OS400__) || defined(_HPUX)
		OSA_EXPORT size_t unicode2native(char** out, char* in, size_t size_in);
		OSA_EXPORT size_t native2unicode(char** out, char* in, size_t size_in);
		OSA_EXPORT int convert(char** out, char* in, size_t size_in, char* to_cp, char* from_cp);
		
		OSA_EXPORT void logMessage(char* message);
		OSA_EXPORT void logErrorMessage(char* message);
	#endif
}} // namespace


#endif //__OSA_H__


